package com.biz.crm.tpm.business.withholding.summary.local.util;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.identity.FacturerUserDetails;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.mdm.business.businessunit.sdk.service.MdmBusinessUnitVoService;
import com.biz.crm.mdm.business.businessunit.sdk.vo.MdmBusinessUnitVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.sales.org.sdk.enums.SalesOrgLevelTypeEnum;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mn.third.system.sap.fi.sdk.dto.BcBpmCeWithholdingSummaryDto;
import com.biz.crm.tpm.business.withholding.summary.local.entity.WithholdingSummaryDetailEntity;
import com.biz.crm.tpm.business.withholding.summary.local.entity.WithholdingSummaryEntity;
import com.biz.crm.tpm.business.withholding.summary.local.repository.TpmWithholdingSummaryDetailRepository;
import com.biz.crm.tpm.business.withholding.summary.local.repository.TpmWithholdingSummaryRepository;
import com.biz.crm.tpm.business.withholding.summary.sdk.constant.WithholdingSummaryActivityTypeTypeEnum;
import com.biz.crm.tpm.business.withholding.summary.sdk.constant.WithholdingSummaryConstant;
import com.biz.crm.tpm.business.withholding.summary.sdk.constant.WithholdingSummarySupplierTypeEnum;
import com.biz.crm.tpm.business.withholding.summary.sdk.constant.YesOrNoEnum;
import com.biz.crm.tpm.business.withholding.summary.sdk.dto.WithholdingSummaryDetailDto;
import com.biz.crm.tpm.business.withholding.summary.sdk.dto.WithholdingSummaryDto;
import com.biz.crm.tpm.business.withholding.summary.sdk.dto.WithholdingSummaryROrgDto;
import com.biz.crm.tpm.business.withholding.summary.sdk.vo.*;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * <p>
 *
 * </p>
 *
 * @author chenshuang
 * @since 2023-02-14
 */
@Slf4j
@Component
public class WithholdingSummaryUtil {

    @Autowired(required = false)
    private TpmWithholdingSummaryRepository tpmWithholdingSummaryRepository;

    @Autowired(required = false)
    private TpmWithholdingSummaryDetailRepository tpmWithholdingSummaryDetailRepository;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private MdmBusinessUnitVoService mdmBusinessUnitVoService;

    /**
     * 新增验证
     *
     * @param dto
     */
    public void valCreate(WithholdingSummaryDto dto) {
        Validate.notNull(dto, "新增时，对象信息不能为空！");
        dto.setId(null);
        Validate.notBlank(dto.getTenantCode(), "新增数据时，租户编号不能为空！");
        Validate.notBlank(dto.getWithholdingFormulaCode(), "新增数据时，预提汇总规则编码不能为空！");
        Validate.notBlank(dto.getWithholdingFormulaName(), "新增数据时，预提汇总规则名称不能为空！");
        Validate.notBlank(dto.getWithholdingYearMonth(), "新增数据时，预提年月不能为空！");
    }

    /**
     * 构建汇总明细数据
     * 1、获取汇总维度最小组织纬度
     * 2、汇总规则组织及下级所有组织-过滤预提明细
     * 3、查询预提明细并汇总
     * 4、销售组织信息
     * 5、预提汇总明细
     * 6、预提汇总关联组织
     *
     * @param withholdingSummaryFormulaVo
     * @param dto
     * @return
     */
    public List<WithholdingSummaryVo> buildDetailList(WithholdingSummaryFormulaVo withholdingSummaryFormulaVo,
                                                      WithholdingSummaryDto dto) {
        //获取业务单元税率
        MdmBusinessUnitVo businessUnitVo = mdmBusinessUnitVoService.findByCode(withholdingSummaryFormulaVo.getBusinessUnitCode());
        BigDecimal taxRate = Objects.nonNull(businessUnitVo.getTaxRatio()) ? businessUnitVo.getTaxRatio() : BigDecimal.ONE;
        Validate.isTrue(taxRate.compareTo(BigDecimal.ZERO) != 0, "税率配置异常：税率为0");

        //1、获取汇总维度最小组织纬度
        String salesOrgLevel = this.summaryOrgDimension(withholdingSummaryFormulaVo);

        //2、汇总规则组织及下级所有组织-过滤预提明细
        List<SalesOrgVo> salesOrgVos = salesOrgVoService.findAllChildrenBySalesOrgCodes(withholdingSummaryFormulaVo.getOrgVoList().stream().map(WithholdingSummaryROrgVo::getSalesOrgCode).collect(Collectors.toList()));
        List<String> orgCodes = salesOrgVos.stream().map(SalesOrgVo::getSalesOrgCode).collect(Collectors.toList());

        //3、查询预提明细并汇总
        List<TpmWithholdingDetailSummaryDetailRespVo> detailRespVos = this.tpmWithholdingSummaryRepository.findWithholdingDetailList(dto, withholdingSummaryFormulaVo, orgCodes, salesOrgLevel);
        log.info("查询预提明细并汇总====>" + detailRespVos.size());

        //4、销售组织信息
        orgCodes = detailRespVos.stream().filter(e -> StringUtils.isNotEmpty(e.getSalesOrgCode())).map(TpmWithholdingDetailSummaryDetailRespVo::getSalesOrgCode).distinct().collect(Collectors.toList());
        salesOrgVos = salesOrgVoService.findBySalesOrgCodes(orgCodes);
        Map<String, SalesOrgVo> salesOrgVoMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(salesOrgVos)) {
            salesOrgVoMap.putAll(salesOrgVos.stream().collect(Collectors.toMap(SalesOrgVo::getSalesOrgCode, v -> v, (v1, v2) -> v2)));
        }

        //5、预提汇总明细
        List<WithholdingSummaryDetailVo> detailVoList = (List<WithholdingSummaryDetailVo>) nebulaToolkitService.copyCollectionByWhiteList(detailRespVos, TpmWithholdingDetailSummaryDetailRespVo.class, WithholdingSummaryDetailVo.class, HashSet.class, ArrayList.class);
        if (!StringUtils.equals(YesOrNoEnum.YES.getCode(), withholdingSummaryFormulaVo.getDiffActivityType())) {
            log.info("汇总数量（不区分活动类型）====>" + detailVoList.size());
            return Lists.newArrayList(this.createDetailList(withholdingSummaryFormulaVo, detailVoList, salesOrgVoMap, taxRate, null));
        }

        Map<String, List<WithholdingSummaryDetailVo>> mapList = detailVoList.stream().filter(e -> StringUtils.isNotEmpty(e.getActivityTypeType())).collect(Collectors.groupingBy(WithholdingSummaryDetailVo::getActivityTypeType));

        List<WithholdingSummaryVo> resultList = new ArrayList<>();
        mapList.forEach((key, list) -> {
            WithholdingSummaryVo summaryVo = this.createDetailList(withholdingSummaryFormulaVo, list, salesOrgVoMap, taxRate, key);
            summaryVo.setActivityTypeType(key);
            resultList.add(summaryVo);
        });
        log.info("汇总数量（区分活动类型）====>" + resultList.size());
        return resultList;
    }

    public WithholdingSummaryVo createDetailList(WithholdingSummaryFormulaVo withholdingSummaryFormulaVo,
                                                 List<WithholdingSummaryDetailVo> detailVoList,
                                                 Map<String, SalesOrgVo> salesOrgVoMap,
                                                 BigDecimal taxRate, String activityTypeType) {
        WithholdingSummaryVo summaryVo = new WithholdingSummaryVo();
        //生成汇总上传编码
//        String ruleCode = WithholdingSummaryConstant.WITHHOLDING_SUMMARY_UPLOAD_CODE_PREFIX + com.biz.crm.mn.common.base.util.DateUtil.format(new Date(), "yyyyMMdd");
        String withholdingUploadCode = generateCodeService.generateCode(WithholdingSummaryConstant.WITHHOLDING_SUMMARY_UPLOAD_CODE_PREFIX, 1, 6, 2, TimeUnit.DAYS).get(0);
        summaryVo.setWithholdingUploadCode(withholdingUploadCode);
        WithholdingSummaryActivityTypeTypeEnum activityTypeTypeEnum = WithholdingSummaryActivityTypeTypeEnum.activityTypeTypeToBusinessTitle(activityTypeType);
        summaryVo.setBusinessTitle(activityTypeTypeEnum.getBusinessTitle());
        summaryVo.setBusinessTitleName(activityTypeTypeEnum.getBusinessTitleName());
        detailVoList.forEach(v -> {
//            String detailRuleCode = WithholdingSummaryConstant.WITHHOLDING_SUMMARY_DETAIL_CODE_PREFIX + DateUtil.format(new Date(), "yyyyMMdd");
            String withholdingDetailCode = generateCodeService.generateCode(WithholdingSummaryConstant.WITHHOLDING_SUMMARY_DETAIL_CODE_PREFIX, 1, 6, 2, TimeUnit.DAYS).get(0);
            v.setWithholdingDetailCode(withholdingDetailCode);
            v.setBusinessExpItem(activityTypeTypeEnum.getBusinessExpItemDefault());
            v.setQuantity(v.getQuantity().setScale(WithholdingSummaryConstant.QUANTITY_SCALE, RoundingMode.HALF_DOWN));
            v.setWithholdingAmount(v.getWithholdingTaxAmount().setScale(WithholdingSummaryConstant.AMOUNT_SCALE, RoundingMode.HALF_DOWN));
            if (Objects.nonNull(v.getQuantity()) && BigDecimal.ZERO.compareTo(v.getQuantity()) != 0) {
                v.setPrice(v.getWithholdingAmount().divide(v.getQuantity(), WithholdingSummaryConstant.PRICE_SCALE, RoundingMode.HALF_DOWN));
            } else {
                v.setPrice(BigDecimal.ZERO);
            }
            v.setEstimateTaxAmount(BigDecimal.ZERO);
            v.setAttachmentNum(BigDecimal.ZERO);
            v.setPromotionAmount(BigDecimal.ZERO);
            v.setTenantCode(TenantUtils.getTenantCode());
            v.setWithholdingUploadCode(summaryVo.getWithholdingUploadCode());
            if (WithholdingSummaryActivityTypeTypeEnum.personnel_cost.getDictCode().equals(activityTypeType)) {
                v.setContractCode(withholdingSummaryFormulaVo.getContractCode());
                v.setContractName(withholdingSummaryFormulaVo.getContractName());
            }
            if (StringUtils.isNotEmpty(v.getSalesOrgCode()) && salesOrgVoMap.containsKey(v.getSalesOrgCode())) {
                v.setSaleOrgName(salesOrgVoMap.get(v.getSalesOrgCode()).getSalesOrgName());
            }
        });

        //6、预提汇总关联组织
        List<WithholdingSummaryROrgVo> orgVoList = (List<WithholdingSummaryROrgVo>) nebulaToolkitService.copyCollectionByWhiteList(withholdingSummaryFormulaVo.getOrgVoList(), WithholdingSummaryROrgVo.class, WithholdingSummaryROrgVo.class, HashSet.class, ArrayList.class);
        orgVoList.forEach(v -> {
            v.setWithholdingUploadCode(summaryVo.getWithholdingUploadCode());
            v.setTenantCode(TenantUtils.getTenantCode());
        });
        summaryVo.setDetailVoList(detailVoList);
        summaryVo.setOrgVoList(orgVoList);
        return summaryVo;
    }

    /**
     * 编辑验证
     *
     * @param dto
     */
    public void valUpdate(WithholdingSummaryDto dto) {
        Validate.notNull(dto, "编辑时，对象信息不能为空！");
        Validate.notBlank(dto.getTenantCode(), "编辑时，租户编号不能为空！");
        Validate.notBlank(dto.getId(), "编辑时，数据ID不能为空！");
        Validate.notBlank(dto.getProfitCenter(), "编辑时，利润中心不能为空！");
        Validate.notNull(dto.getExchangeRate(), "编辑时，汇率不能为空！");
        Validate.notBlank(dto.getTradeCurrency(), "编辑时，交易货币不能为空！");
        Validate.notBlank(dto.getExpenseContent(), "编辑时，开支内容不能为空！");

        //TPM客商类型字典
        Map<String, String> merchantsTypeMap = dictToolkitService.findMapByDictTypeCode(WithholdingSummaryConstant.TPM_MERCHANTS_TYPE);
        //业务细类
        List<DictDataVo> accruedAccountList = dictToolkitService.tree(WithholdingSummaryConstant.TPM_ACCRUED_ACCOUNT);
        Map<String, DictDataVo> accruedAccountMap = Maps.newHashMap();
        if (!CollectionUtils.isEmpty(accruedAccountList)) {
            accruedAccountMap = accruedAccountList.stream().collect(Collectors.toMap(DictDataVo::getDictCode, v -> v));
        }
        int index = 1;
        for (WithholdingSummaryDetailDto v : dto.getDetailDtoList()) {
            //根据业务细类验证是否必填客商信息
            Validate.notBlank(v.getBusinessExpItem(), "明细信息第[%s]行，[业务细类]为空！", index);
            Validate.isTrue(accruedAccountMap.containsKey(v.getBusinessExpItem()), "明细信息第[%s]行，业务细类[%s]在数据字典[%s]中未维护！", index, v.getBusinessExpItem(), WithholdingSummaryConstant.TPM_ACCRUED_ACCOUNT);
            DictDataVo businessExpItemDictVo = accruedAccountMap.get(v.getBusinessExpItem());
            Validate.isTrue(Objects.nonNull(businessExpItemDictVo.getExtendMap()) && businessExpItemDictVo.getExtendMap().containsKey("yes_or_no"),
                    "明细信息第[%s]行，业务细类[%s]在数据字典[%s]中未维护是否必填客商类型！", index, v.getBusinessExpItem(), WithholdingSummaryConstant.TPM_ACCRUED_ACCOUNT);
            boolean needSupplier = StringUtils.equals(YesOrNoEnum.YES.getCode(), businessExpItemDictVo.getExtendMap().get("yes_or_no"));
            if (needSupplier) {
                Validate.notBlank(v.getCustomerClassifyCode(), "明细信息第[%s]行，[客商分类编码]为空！", index);
                Validate.isTrue(merchantsTypeMap.containsKey(v.getCustomerClassifyCode()), "明细信息第[%s]行，[客商类型]不存在！", index);
                v.setCustomerClassifyName(merchantsTypeMap.get(v.getCustomerClassifyCode()));
                Validate.notBlank(v.getSupplierCode(), "明细信息第[%s]行，[客商编码]为空！", index);
                Validate.notBlank(v.getSupplierName(), "明细信息第[%s]行，[客商名称]为空！", index);
            } else {
                Validate.isTrue(StringUtils.isEmpty(v.getCustomerClassifyCode()), "明细信息第[%s]行，业务细类[%s]不允许录入客商信息！", index, businessExpItemDictVo.getDictValue());
                v.setCustomerClassifyName("");
                Validate.isTrue(StringUtils.isEmpty(v.getSupplierCode()), "明细信息第[%s]行，业务细类[%s]不允许录入客商信息！", index, businessExpItemDictVo.getDictValue());
                Validate.isTrue(StringUtils.isEmpty(v.getSupplierName()), "明细信息第[%s]行，业务细类[%s]不允许录入客商信息！", index, businessExpItemDictVo.getDictValue());
            }

//            Validate.notBlank(v.getSalesOrgCode(), "明细信息第[%s]行，[销售组织编码]为空！", index);
//            Validate.notBlank(v.getSaleOrgName(), "明细信息第[%s]行，[销售组织名称]为空！", index);
            if (StringUtils.isNotEmpty(v.getActivityFormCode())) {
                Validate.notBlank(v.getActivityFormName(), "明细信息第[%s]行，[活动形式名称]为空！", index);
            }
            if (StringUtils.isNotEmpty(v.getCustomerCode())) {
                Validate.notBlank(v.getCustomerName(), "明细信息第[%s]行，[客户名称]为空！", index);
            }
            Validate.notBlank(v.getCostCenter(), "明细信息第[%s]行，[成本中心编码]为空！", index);
            Validate.notBlank(v.getCostCenterName(), "明细信息第[%s]行，[成本中心名称]为空！", index);
            Validate.isTrue(Objects.nonNull(v.getWithholdingAmount()), "明细信息第[%s]行，[不含税金额]为空！", index);
            Validate.isTrue(Objects.nonNull(v.getEstimateTaxAmount()), "明细信息第[%s]行，[暂估税金]为空！", index);
            Validate.isTrue(Objects.nonNull(v.getAttachmentNum()), "明细信息第[%s]行，[附件张数]为空！", index);
            index++;
        }
    }

    /**
     * 编辑验证必填-区分人员非人员费用必填字段
     *
     * @param detailDtoList
     * @param activityTypeType
     * @param index
     */
    public void valUpdateExtend(List<WithholdingSummaryDetailDto> detailDtoList, String activityTypeType, int index) {
        if (StringUtils.isEmpty(activityTypeType)) {
            return;
        }
        for (WithholdingSummaryDetailDto v : detailDtoList) {
            if (!WithholdingSummaryActivityTypeTypeEnum.personnel_cost.getDictCode().equals(activityTypeType)) {
                v.setQuantity(null);
                v.setPrice(null);
                v.setContractCode("");
                v.setContractName("");
            } else {
                Validate.notNull(v.getQuantity(), "第[%s]行,数量为空！", index);
                Validate.notBlank(v.getContractCode(), "第[%s]行,合同编码为空！", index);
                Validate.notBlank(v.getContractName(), "第[%s]行,合同名称为空！", index);
            }
            index++;
        }
    }

    /**
     * 编辑时更新汇总明细
     *
     * @param dto
     * @return
     */
    public List<WithholdingSummaryDetailVo> buildUpdateDetailList(WithholdingSummaryDto dto) {
        Validate.isTrue(!CollectionUtils.isEmpty(dto.getDetailDtoList()), "缺少汇总明细信息！");
        List<String> detailCodeList = this.tpmWithholdingSummaryDetailRepository.lambdaQuery()
                .eq(WithholdingSummaryDetailEntity::getWithholdingUploadCode, dto.getWithholdingUploadCode())
                .select(WithholdingSummaryDetailEntity::getWithholdingDetailCode)
                .list()
                .stream()
                .map(WithholdingSummaryDetailEntity::getWithholdingDetailCode)
                .collect(Collectors.toList());

        int index = 1;
        for (WithholdingSummaryDetailDto v : dto.getDetailDtoList()) {
            if (StringUtils.isNotEmpty(v.getWithholdingDetailCode())) {
                Validate.isTrue(detailCodeList.contains(v.getWithholdingDetailCode()),
                        "明细信息第[%s]行，预提明细编号[%s]需由系统生成，请勿手动填写！", index, v.getWithholdingDetailCode());
            }
            index++;
        }
        for (WithholdingSummaryDetailDto v : dto.getDetailDtoList()) {
            if (StringUtils.isEmpty(v.getWithholdingDetailCode())) {
//                String ruleCode = WithholdingSummaryConstant.WITHHOLDING_SUMMARY_DETAIL_CODE_PREFIX + DateUtil.format(new Date(), "yyyyMMdd");
                String withholdingDetailCode = generateCodeService.generateCode(WithholdingSummaryConstant.WITHHOLDING_SUMMARY_DETAIL_CODE_PREFIX, 1, 6, 2, TimeUnit.DAYS).get(0);
                v.setWithholdingDetailCode(withholdingDetailCode);
            }
        }
        Collection<WithholdingSummaryDetailVo> detailVos = nebulaToolkitService.copyCollectionByWhiteList(dto.getDetailDtoList(), WithholdingSummaryDetailDto.class, WithholdingSummaryDetailVo.class, HashSet.class, ArrayList.class);
        detailVos.forEach(detail -> {
            if (StringUtils.isEmpty(detail.getCustomerClassifyCode())) {
                detail.setCustomerClassifyName("");
                detail.setSupplierCode("");
                detail.setSupplierName("");
            }
            detail.setEstimateTaxAmount(detail.getEstimateTaxAmount().setScale(WithholdingSummaryConstant.AMOUNT_SCALE, RoundingMode.HALF_DOWN));
            detail.setWithholdingAmount(detail.getWithholdingAmount().setScale(WithholdingSummaryConstant.AMOUNT_SCALE, RoundingMode.HALF_DOWN));
            detail.setWithholdingTaxAmount(detail.getEstimateTaxAmount().add(detail.getWithholdingAmount()).setScale(WithholdingSummaryConstant.AMOUNT_SCALE, RoundingMode.HALF_DOWN));
            if (Objects.nonNull(detail.getQuantity()) && BigDecimal.ZERO.compareTo(detail.getQuantity()) != 0) {
                detail.setPrice(detail.getWithholdingAmount().divide(detail.getQuantity(), WithholdingSummaryConstant.PRICE_SCALE, RoundingMode.HALF_DOWN));
            } else {
                detail.setPrice(BigDecimal.ZERO);
            }
            if (StringUtils.equals(WithholdingSummaryConstant.TPM_MERCHANTS_TYPE_SUPPLIER, detail.getCustomerClassifyCode())) {
                detail.setSpecialTag(WithholdingSummaryConstant.SPECIAL_TAG);
            } else {
                detail.setSpecialTag("");
            }
            detail.setTenantCode(TenantUtils.getTenantCode());
        });
        for (WithholdingSummaryROrgDto v : dto.getOrgVoList()) {
            v.setWithholdingUploadCode(dto.getWithholdingUploadCode());
            v.setTenantCode(TenantUtils.getTenantCode());
        }
        return Lists.newArrayList(detailVos);
    }

    public void validateUploadParam(WithholdingSummaryVo withholdingSummaryVo) {
        Validate.notBlank(withholdingSummaryVo.getSalesOrgCode(), "汇总上传编码[%s]，销售组织编码为空！", withholdingSummaryVo.getWithholdingUploadCode());
        Validate.notBlank(withholdingSummaryVo.getProfitCenter(), "汇总上传编码[%s]，利润中心为空！", withholdingSummaryVo.getWithholdingUploadCode());
        Validate.notBlank(withholdingSummaryVo.getTradeCurrency(), "汇总上传编码[%s]，交易货币为空！", withholdingSummaryVo.getWithholdingUploadCode());
        Validate.notNull(withholdingSummaryVo.getExchangeRate(), "汇总上传编码[%s]，汇率为空！", withholdingSummaryVo.getWithholdingUploadCode());
        Validate.notBlank(withholdingSummaryVo.getExpenseContent(), "汇总上传编码[%s]，开支内容为空！", withholdingSummaryVo.getWithholdingUploadCode());
        Validate.notNull(withholdingSummaryVo.getWithholdingAmount(), "汇总上传编码[%s]，预提金额为空！", withholdingSummaryVo.getWithholdingUploadCode());

        Validate.notEmpty(withholdingSummaryVo.getDetailVoList(), "汇总上传编码[%s]，汇总明细为空！", withholdingSummaryVo.getWithholdingUploadCode());

        //业务细类
        List<DictDataVo> accruedAccountList = dictToolkitService.tree(WithholdingSummaryConstant.TPM_ACCRUED_ACCOUNT);
        Map<String, DictDataVo> accruedAccountMap = Maps.newHashMap();
        if (!CollectionUtils.isEmpty(accruedAccountList)) {
            accruedAccountMap = accruedAccountList.stream().collect(Collectors.toMap(DictDataVo::getDictCode, v -> v));
        }
        for (WithholdingSummaryDetailVo v : withholdingSummaryVo.getDetailVoList()) {
            Validate.notBlank(v.getBusinessExpItem(), "汇总上传编码[%s]，汇总明细编码[%s]，业务细类为空！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode());
            Validate.isTrue(accruedAccountMap.containsKey(v.getBusinessExpItem()),
                    "汇总上传编码[%s]，预提汇总明细编码[%s]，业务细类[%s]在数据字典[%s]中未维护！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode(), v.getBusinessExpItem(), WithholdingSummaryConstant.TPM_ACCRUED_ACCOUNT);
            DictDataVo businessExpItemDictVo = accruedAccountMap.get(v.getBusinessExpItem());
            Validate.isTrue(Objects.nonNull(businessExpItemDictVo.getExtendMap()) && businessExpItemDictVo.getExtendMap().containsKey("yes_or_no"),
                    "汇总上传编码[%s]，预提汇总明细编码[%s]，业务细类[%s]在数据字典[%s]中未维护是否必填客商类型！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode(), v.getBusinessExpItem(), WithholdingSummaryConstant.TPM_ACCRUED_ACCOUNT);
            boolean needSupplier = StringUtils.equals(YesOrNoEnum.YES.getCode(), businessExpItemDictVo.getExtendMap().get("yes_or_no"));
            if (needSupplier) {
                Validate.notBlank(v.getSupplierCode(),
                        "汇总上传编码[%s]，预提汇总明细编码[%s]，[客商编码]为空！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode());
                Validate.notBlank(v.getSupplierName(),
                        "汇总上传编码[%s]，预提汇总明细编码[%s]行，[客商名称]为空！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode());
            } else {
                Validate.isTrue(StringUtils.isEmpty(v.getCustomerClassifyCode()),
                        "汇总上传编码[%s]，预提汇总明细编码[%s]，业务细类[%s]不允许录入客商信息！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode(), businessExpItemDictVo.getDictValue());
                v.setCustomerClassifyName("");
                Validate.isTrue(StringUtils.isEmpty(v.getSupplierCode()),
                        "汇总上传编码[%s]，预提汇总明细编码[%s]，业务细类[%s]不允许录入客商信息！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode(), businessExpItemDictVo.getDictValue());
                Validate.isTrue(StringUtils.isEmpty(v.getSupplierName()),
                        "汇总上传编码[%s]，预提汇总明细编码[%s]，业务细类[%s]不允许录入客商信息！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode(), businessExpItemDictVo.getDictValue());
            }
            Validate.notBlank(v.getCostCenter(), "汇总上传编码[%s]，汇总明细编码[%s]成本中心编码为空！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode());
            Validate.notBlank(v.getCostCenterName(), "汇总上传编码[%s]，汇总明细编码[%s]成本中心名称为空！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode());
            Validate.notNull(v.getEstimateTaxAmount(), "汇总上传编码[%s]，汇总明细编码[%s]暂估税金为空！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode());
            Validate.notNull(v.getWithholdingAmount(), "汇总上传编码[%s]，汇总明细编码[%s]不含税金额为空！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode());

            if (WithholdingSummaryActivityTypeTypeEnum.personnel_cost.getDictCode().equals(withholdingSummaryVo.getActivityTypeType())) {
                Validate.notNull(v.getQuantity(), "汇总上传编码[%s]，汇总明细编码[%s]数量为空！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode());
                Validate.notNull(v.getPrice(), "汇总上传编码[%s]，汇总明细编码[%s]不含税单价为空！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode());
                Validate.notNull(v.getPromotionAmount(), "汇总上传编码[%s]，汇总明细编码[%s]优惠金额为空！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode());
                Validate.notBlank(v.getContractCode(), "汇总上传编码[%s]，汇总明细编码[%s]合同编号为空！", withholdingSummaryVo.getWithholdingUploadCode(), v.getWithholdingDetailCode());
            }
        }
    }

    public BcBpmCeWithholdingSummaryDto buildUploadParam(WithholdingSummaryVo withholdingSummaryVo, FacturerUserDetails loginDetails) {
        //附件总数
        BigDecimal attachCount = withholdingSummaryVo.getDetailVoList().stream().filter(e -> Objects.nonNull(e.getAttachmentNum())).map(WithholdingSummaryDetailVo::getAttachmentNum).reduce(BigDecimal.ZERO, BigDecimal::add);
        BcBpmCeWithholdingSummaryDto dto = new BcBpmCeWithholdingSummaryDto();
        //销售公司
        Validate.notBlank(withholdingSummaryVo.getSalesOrgCode(), "汇总上传编码[%s]，未查询到销售公司！", withholdingSummaryVo.getWithholdingUploadCode());

        //头
        BcBpmCeWithholdingSummaryDto.HeaderData headerData = new BcBpmCeWithholdingSummaryDto.HeaderData();
        headerData.setOUT_DOCUMENT(withholdingSummaryVo.getWithholdingUploadCode());
        headerData.setAPPLICANT(loginDetails.getRealName());
        headerData.setAPPLICANT_NO(loginDetails.getAccount());
        headerData.setBUKRS(withholdingSummaryVo.getSalesOrgCode());
        headerData.setCREATED_BY(loginDetails.getAccount());
        headerData.setCREATED_BY_TXT(loginDetails.getRealName());
        headerData.setCURRENCY(withholdingSummaryVo.getTradeCurrency());
        headerData.setRATE(withholdingSummaryVo.getExchangeRate().toString());
        headerData.setATTACHCOUNT(attachCount.toString());
        headerData.setDESCRIPTION(withholdingSummaryVo.getExpenseContent());
        headerData.setAMOUNT(withholdingSummaryVo.getWithholdingAmount().setScale(2, BigDecimal.ROUND_HALF_UP).toString());
        headerData.setPRCTR(Integer.valueOf(withholdingSummaryVo.getProfitCenter()).toString());
        headerData.setEXP_TYPE(withholdingSummaryVo.getBusinessTitle());

        //明细
        List<BcBpmCeWithholdingSummaryDto.DetailList> detailList = new ArrayList<>();
        int index = 1;
        for (WithholdingSummaryDetailVo v : withholdingSummaryVo.getDetailVoList()) {
            BcBpmCeWithholdingSummaryDto.DetailList detail = new BcBpmCeWithholdingSummaryDto.DetailList();
            detail.setSEQ(index + "");
            detail.setEXP_ITEM(v.getBusinessExpItem());
            detail.setACCOUNT_TYPE(
                    StringUtils.isEmpty(v.getCustomerClassifyCode()) ?
                            v.getCustomerClassifyCode() :
                            WithholdingSummarySupplierTypeEnum.codeToEnum(v.getCustomerClassifyCode()).getValue());
            detail.setLIFNR(v.getSupplierCode());
            detail.setLIFNR_TXT(v.getSupplierName());
            detail.setPAYAMOUNT(v.getWithholdingTaxAmount().setScale(WithholdingSummaryConstant.AMOUNT_SCALE, RoundingMode.HALF_DOWN).toString());
            detail.setKOSTL(v.getCostCenter());
            detail.setKOSTL_TXT(v.getCostCenterName());
            detail.setTAX_AMT(v.getEstimateTaxAmount().setScale(WithholdingSummaryConstant.AMOUNT_SCALE, RoundingMode.HALF_DOWN).toString());
            detail.setTAX_FREE_AMT(v.getWithholdingAmount().setScale(WithholdingSummaryConstant.AMOUNT_SCALE, RoundingMode.HALF_DOWN).toString());
            if (Objects.isNull(v.getPrice())) {
                v.setPrice(BigDecimal.ZERO);
            }
            if (Objects.isNull(v.getQuantity())) {
                v.setQuantity(BigDecimal.ZERO);
            }
            if (WithholdingSummaryActivityTypeTypeEnum.personnel_cost.getDictCode().equals(withholdingSummaryVo.getActivityTypeType())) {
                detail.setPRICE_WITHOUT_TAX(v.getPrice().setScale(WithholdingSummaryConstant.PRICE_SCALE, RoundingMode.HALF_DOWN).toString());
                detail.setCOUNT_N(v.getQuantity().setScale(WithholdingSummaryConstant.QUANTITY_SCALE, RoundingMode.HALF_DOWN).toString());
                detail.setCONTRACT_ID(v.getContractCode());
            }
            if (Objects.isNull(v.getPromotionAmount())) {
                v.setPromotionAmount(BigDecimal.ZERO);
            }
            detail.setDISCOUNT_AMT(v.getPromotionAmount().setScale(WithholdingSummaryConstant.AMOUNT_SCALE, RoundingMode.HALF_DOWN).toString());
            detailList.add(detail);
            index++;
        }
        dto.setHEDAER(headerData);
        dto.setEXPENSE_DETAIL_LIST(detailList);
        return dto;
    }

    public String getRedisCacheKey(String cacheKey) {
        return WithholdingSummaryConstant.WITHHOLDING_SUMMARY_DETAIL_CACHE_PREFIX + cacheKey;
    }

    /**
     * 验证汇总是否能查询到明细数据
     *
     * @param dto
     */
    public void checkDetailCount(WithholdingSummaryDto dto) {
        //1、汇总规则
        WithholdingSummaryFormulaVo withholdingSummaryFormulaVo = this.tpmWithholdingSummaryRepository.findHeadInfoByWithholdingFormulaCode(dto.getWithholdingFormulaCode());
        Validate.isTrue(Objects.nonNull(withholdingSummaryFormulaVo), "预提汇总规则[%s]不存在！", dto.getWithholdingFormulaCode());
        Validate.notEmpty(withholdingSummaryFormulaVo.getOrgVoList(), "预提汇总规则[%s]未配置销售组织！", withholdingSummaryFormulaVo.getWithholdingFormulaCode());

        //2、汇总销售组织范围
        List<SalesOrgVo> salesOrgVos = salesOrgVoService.findAllChildrenBySalesOrgCodes(withholdingSummaryFormulaVo.getOrgVoList().stream().map(WithholdingSummaryROrgVo::getSalesOrgCode).collect(Collectors.toList()));
        Validate.notEmpty(salesOrgVos, "预提汇总规则编码[%s]销售组织信息不存在或未启用！", withholdingSummaryFormulaVo.getWithholdingFormulaCode());
        List<String> orgCodes = salesOrgVos.stream().map(SalesOrgVo::getSalesOrgCode).collect(Collectors.toList());


        //3、汇总明细数据
        //获取汇总维度最小组织纬度
        String salesOrgLevel = this.summaryOrgDimension(withholdingSummaryFormulaVo);
        Long detailCount = this.tpmWithholdingSummaryRepository.countSummaryDetail(dto, withholdingSummaryFormulaVo, orgCodes, salesOrgLevel);
        Validate.isTrue(detailCount > 0, "预提汇总规则[%s]，预提年月[%s]未查询到汇总明细！", dto.getWithholdingFormulaCode(), dto.getWithholdingYearMonth());
    }

    /**
     * 获取汇总维度最小组织纬度
     *
     * @param withholdingSummaryFormulaVo
     * @return
     */
    public String summaryOrgDimension(WithholdingSummaryFormulaVo withholdingSummaryFormulaVo) {
        String salesOrgLevel = null;
        if (StringUtils.equals(YesOrNoEnum.YES.getCode(), withholdingSummaryFormulaVo.getSaleGroupCb())) {
            salesOrgLevel = SalesOrgLevelTypeEnum.GROUP.getCode();
        } else if (StringUtils.equals(YesOrNoEnum.YES.getCode(), withholdingSummaryFormulaVo.getSaleOrgDepartmentCb())) {
            salesOrgLevel = SalesOrgLevelTypeEnum.DEPARTMENT.getCode();
        } else if (StringUtils.equals(YesOrNoEnum.YES.getCode(), withholdingSummaryFormulaVo.getSaleOrgCb())) {
            salesOrgLevel = SalesOrgLevelTypeEnum.MECHANISM.getCode();
        }
        return salesOrgLevel;
    }
}
