package com.biz.crm.tpm.business.year.budget.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mn.common.base.util.DataTooLongGenerateRespUtil;
import com.biz.crm.mn.common.base.vo.CommonSelectVo;
import com.biz.crm.tpm.business.month.budget.sdk.constant.BudgetLockConstant;
import com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesGoalVerticalImportsVo;
import com.biz.crm.tpm.business.year.budget.local.entity.YearBudgetEntity;
import com.biz.crm.tpm.business.year.budget.local.service.YearBudgetGenerateMonthBudgetAsyncService;
import com.biz.crm.tpm.business.year.budget.sdk.constant.YearBudgetConstant;
import com.biz.crm.tpm.business.year.budget.sdk.dto.YearBudgetDto;
import com.biz.crm.tpm.business.year.budget.local.service.YearBudgetService;
import com.biz.crm.tpm.business.year.budget.sdk.vo.YearBudgetImportsVo;
import com.biz.crm.tpm.business.year.budget.sdk.vo.YearBudgetVo;
import com.biz.crm.mn.common.base.service.RedisLockService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * @author huojia
 * @date 2022年10月26日 16:44
 */
@RestController
@RequestMapping("/v1/year/budget")
@Slf4j
@Api(tags = "年度预算")
public class YearBudgetController {

    @Autowired(required = false)
    private YearBudgetService yearBudgetService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private YearBudgetGenerateMonthBudgetAsyncService yearBudgetGenerateMonthBudgetAsyncService;

    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByConditions")
    public Result<Page<YearBudgetVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                       @ApiParam(name = "YearBudgetDto", value = "年度预算信息") YearBudgetDto dto) {
        try {
            Page<YearBudgetVo> page = this.yearBudgetService.findByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @ApiOperation(value = "通过主键查询单条数据")
    @GetMapping("detail")
    public Result<YearBudgetVo> detail(@RequestParam("id") @ApiParam(name = "id", value = "主键id") String id,
                                       @RequestParam("editFlag") @ApiParam(name = "editFlag", value = "是否编辑") String editFlag) {
        try {
            return Result.ok(this.yearBudgetService.findById(id, editFlag));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 新增数据
     *
     * @param dto 实体对象
     * @return 新增结果
     */
    @ApiOperation(value = "新增数据")
    @PostMapping
    public Result<YearBudgetDto> create(@ApiParam(name = "YearBudgetDto", value = "TPM-年度预算") @RequestBody YearBudgetDto dto) {
        try {
            Result<YearBudgetDto> result = Result.ok(this.yearBudgetService.create(dto));
            result.setMessage("新增成功");
            return result;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(DataTooLongGenerateRespUtil.generateRespStr(e, YearBudgetEntity.class));
        }
    }

    /**
     * 修改数据
     *
     * @param dto 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "修改数据")
    @PatchMapping
    public Result<YearBudgetVo> update(@ApiParam(name = "YearBudgetDto", value = "TPM-年度预算") @RequestBody YearBudgetDto dto) {
        try {
            YearBudgetVo result = this.yearBudgetService.update(dto);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(DataTooLongGenerateRespUtil.generateRespStr(e, YearBudgetEntity.class));
        }
    }

    /**
     * 修改数据
     *
     * @param dto 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "修改数据")
    @PostMapping("update")
    public Result<YearBudgetVo> updateForPost(@ApiParam(name = "YearBudgetDto", value = "TPM-年度预算") @RequestBody YearBudgetDto dto) {
        try {
            YearBudgetVo result = this.yearBudgetService.update(dto);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 手动触发修改月度预算
     *
     * @param ids id集合
     * @return 修改结果
     */
    @ApiOperation(value = "手动触发修改月度预算")
    @PatchMapping(value = "manualUpdateMonth")
    public Result manualUpdateMonth(@ApiParam(name = "yearBudgetDto", value = "TPM-年度预算") @RequestBody List<String> ids) {
        try {
            this.yearBudgetService.manualUpdateMonth(ids);
            return Result.ok("触发成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 生成月度预算
     *
     * @param dto 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "生成月度预算")
    @PostMapping("generateMonthBudget")
    public Result<?> generateMonthBudget(@RequestBody YearBudgetDto dto) {
        try {
            if (CollectionUtils.isEmpty(dto.getIds())) {
                //通过除id外的筛选条件选择数据，控制下最大维度，别一次扫全表，处理数据
                if ( StringUtil.isBlank(dto.getBusinessFormatCode())
                        || StringUtils.isBlank(dto.getBusinessUnitCode())
                        || StringUtils.isBlank(dto.getYearLy()) ) {

                    throw new IllegalArgumentException("注意，请通过'勾选'(优先取) 或者 查询条件（至少填写业态、业务单元、年份)，两种方式来选择数据！");
                }
            }
            if (CollectionUtils.isEmpty(dto.getIds())) {
                AbstractCrmUserIdentity loginUser = loginUserService.getAbstractLoginUser();
                this.yearBudgetGenerateMonthBudgetAsyncService.generateMonthBudgetAsync(loginUser,dto);
                return Result.ok("正在生成月度预算中，请稍等...");

            } else {
                this.yearBudgetService.generateMonthBudget(dto);
                return Result.ok("生成月度预算成功");
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 生成月度预算-能力中心
     *
     * @param ids 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "生成月度预算-能力中心")
    @PostMapping("generateMonthBudgetForOut")
    public Result<?> generateMonthBudgetForOut(@RequestBody List<String> ids) {
        try {
            if (CollectionUtils.isEmpty(ids)) {
                throw new RuntimeException("请选择要操作的数据");
            }
            ids.forEach(id -> {
                if (redisLockService.isLock(YearBudgetConstant.YEAR_BUDGET_LOCK + id)) {
                    throw new RuntimeException("月度预算生成中中，请稍后");
                }
            });
            List<String> monthBudgetIdList = this.yearBudgetService.generateMonthBudgetForOut(ids);
            return Result.ok(monthBudgetIdList);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 删除数据
     *
     * @param ids 主键结合
     * @return 删除结果
     */
    @DeleteMapping
    @ApiOperation(value = "删除数据")
    public Result<?> delete(@RequestParam("ids")  List<String> ids) {
        try {
            this.yearBudgetService.delete(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 导入测试
     *
     * @param importsVos 分页对象
     * @return 所有数据
     */
    @ApiOperation(value = "导入测试")
    @PostMapping("importTest")
    public Result<?> importTest(@RequestBody @ApiParam(name = "importsVos", value = "年度预算导入信息") List<YearBudgetImportsVo> importsVos) {
        try {
            this.yearBudgetService.importTest(importsVos);
            return Result.ok("导入成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 费用归口下拉框
     *
     * @param businessFormatCode 业态
     * @param businessUnitCode 业务单元
     * @return 所有数据
     */
    @ApiOperation(value = "费用归口下拉框")
    @GetMapping("findBelongSelect")
    public Result<List<CommonSelectVo>> findBelongSelect(@RequestParam(value = "businessFormatCode", required = false) @ApiParam(name = "businessFormatCode", value = "业态") String businessFormatCode,
                                                         @RequestParam(value = "businessUnitCode", required = false) @ApiParam(name = "businessUnitCode", value = "业务单元") String businessUnitCode) {
        try {
            List<CommonSelectVo> result = this.yearBudgetService.findBelongSelect(businessFormatCode, businessUnitCode);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 能力中心分页查询年度预算数据
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findPageForOut")
    public Result<Page<YearBudgetVo>> findPageForOut(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                     YearBudgetDto dto) {
        try {
            Page<YearBudgetVo> page = this.yearBudgetService.findPageForOut(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
}
