package com.biz.crm.tpm.business.year.budget.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.TenantFlagOpEntity;
import com.biz.crm.business.common.local.entity.UuidEntity;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.tpm.business.year.budget.local.entity.YearBudgetEntity;
import com.biz.crm.tpm.business.year.budget.local.mapper.YearBudgetMapper;
import com.biz.crm.tpm.business.year.budget.sdk.dto.YearBudgetDto;
import com.biz.crm.tpm.business.year.budget.sdk.dto.YearBudgetToralPointDto;
import com.biz.crm.tpm.business.year.budget.sdk.vo.YearBudgetReportVo;
import com.biz.crm.tpm.business.year.budget.sdk.vo.YearBudgetVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author huojia
 * @date 2022年10月26日 16:46
 */
@Component
public class YearBudgetRepository extends ServiceImpl<YearBudgetMapper, YearBudgetEntity> {

    @Autowired(required = false)
    private YearBudgetMapper yearBudgetMapper;

    /**
     * @param id
     * @param delFlag
     * @return com.biz.crm.tpm.business.year.budget.local.entity.YearBudgetEntity
     * @author huojia
     * @date 2022/10/27 11:30
     **/
    public YearBudgetEntity getById(String id, String delFlag) {
        if (StringUtils.isEmpty(id)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(UuidEntity::getId, id)
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(UuidFlagOpEntity::getDelFlag, delFlag)
                .one();
    }

    /**
     * 根据ids批量查询
     *
     * @param ids
     * @param delFlag
     * @return java.util.List<com.biz.crm.tpm.business.year.budget.local.entity.YearBudgetEntity>
     * @author huojia
     * @date 2022/11/4 15:13
     **/
    public List<YearBudgetEntity> listByIds(List<String> ids, String delFlag) {
        if (CollectionUtils.isEmpty(ids)) {
            return null;
        }
        return this.lambdaQuery()
                .in(UuidEntity::getId, ids)
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(UuidFlagOpEntity::getDelFlag, delFlag)
                .list();
    }

    /**
     * 根据编码获取
     *
     * @param codes
     * @return java.util.List<com.biz.crm.tpm.business.year.budget.local.entity.YearBudgetEntity>
     * @author huojia
     * @date 2022/11/10 20:11
     **/
    public List<YearBudgetEntity> listByCodes(List<String> codes) {
        if (CollectionUtils.isEmpty(codes)) {
            return null;
        }
        return this.lambdaQuery()
                .in(YearBudgetEntity::getYearBudgetCode, codes)
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 获取维度相同的年度预算数量
     *
     * @param dto 分页参数dto
     * @return Integer
     */
    public Integer checkRepeat(YearBudgetDto dto) {
        return this.baseMapper.checkRepeat(dto);
    }

    /**
     * 根据必填项查询出已有的年度预算
     *
     * @param dto
     * @return {@link List}<{@link YearBudgetVo}>
     */
    public List<YearBudgetVo> findByConditions(YearBudgetDto dto) {
        Page<YearBudgetVo> page = new Page<>(1, 50);
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());
        Page<YearBudgetVo> yearBudgetVoPage = this.baseMapper.findByConditions(page, dto);
        if (CollectionUtils.isEmpty(yearBudgetVoPage.getRecords())) {
            return Lists.newArrayList();
        }
        return yearBudgetVoPage.getRecords();
    }

    /**
     * 根据编码查询年度预算
     *
     * @param yearBudgetCode 年度预算编码
     * @return YearBudgetEntity
     */
    public YearBudgetEntity getBudgetByCode(String yearBudgetCode) {
        if (StringUtils.isBlank(yearBudgetCode)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(YearBudgetEntity::getYearBudgetCode, yearBudgetCode)
                .eq(YearBudgetEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(YearBudgetEntity::getTenantCode, TenantUtils.getTenantCode())
                .one();
    }

    public List<YearBudgetReportVo> findYearBudgetSumVo(YearBudgetDto queryDto) {
        return yearBudgetMapper.findYearBudgetSumVo(queryDto);
    }

    /**
     * 查询年度预算总点数
     *
     * @param pointDto 参数
     * @return BigDecimal
     */
    public BigDecimal getYearBudgetTotalPoint(YearBudgetToralPointDto pointDto) {
        return yearBudgetMapper.getYearBudgetTotalPoint(pointDto, TenantUtils.getTenantCode());
    }

    public List<YearBudgetVo> getYearBudgetTotalPointGroupBySalesOrg(YearBudgetToralPointDto pointDto) {
        return yearBudgetMapper.getYearBudgetTotalPointGroupBySalesOrg(pointDto, TenantUtils.getTenantCode());
    }

    public List<YearBudgetEntity> findByYearBudgetCodes(List<String> yearBudgetCodes) {
        if (CollectionUtils.isEmpty(yearBudgetCodes)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .in(YearBudgetEntity::getYearBudgetCode, yearBudgetCodes)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    public List<YearBudgetVo> findYearBudgetByYearBudgetCodes(List<String> yearBudgetCodes) {
        return yearBudgetMapper.findYearBudgetByYearBudgetCodes(yearBudgetCodes, TenantUtils.getTenantCode(), DelFlagStatusEnum.NORMAL.getCode(), EnableStatusEnum.ENABLE.getCode());
    }

    public Page<YearBudgetVo> findPageForOut(Pageable pageable, YearBudgetDto dto) {
        Page<YearBudgetVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.baseMapper.findPageForOut(page, dto);
    }

    public BigDecimal getYearBudgetTotalPointForCustomerGroup(YearBudgetToralPointDto pointDto) {
        return yearBudgetMapper.getYearBudgetTotalPointForCustomerGroup(pointDto, TenantUtils.getTenantCode());
    }

    public List<YearBudgetDto> getListByConditions(YearBudgetDto dto) {
        return yearBudgetMapper.getListByConditions(dto);
    }

    public List<YearBudgetVo> findListForFR(List<YearBudgetDto> yearBudgetDtos) {
        if (CollectionUtils.isEmpty(yearBudgetDtos)) {
            return Collections.emptyList();
        }
        yearBudgetDtos = yearBudgetDtos.stream().filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(yearBudgetDtos)) {
            return Collections.emptyList();
        }
        return yearBudgetMapper.findListForFR(yearBudgetDtos, TenantUtils.getTenantCode());
    }
}
