package com.biz.crm.tpm.business.variable.local.executeIndicator.service.internal;

import com.alibaba.excel.util.Validate;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.biz.crm.mdm.business.customer.retailer.sdk.service.CustomerRetailerVoService;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.constant.AuditExecuteIndicatorConstant;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.dto.AuditExecuteIndicatorDto;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.service.AuditExecuteIndicatorService;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.vo.AuditExecuteIndicatorDeductionAmountImportVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.*;

/**
 * @author: hd
 * @description: 结案核销-执行指标表(AuditExecuteIndicator)导入实现层
 */
@Slf4j
@Component
public class AuditExecuteIndicatorDeductionAmountImportProcess implements ImportProcess<AuditExecuteIndicatorDeductionAmountImportVo> {

    @Autowired(required = false)
    private AuditExecuteIndicatorService auditExecuteIndicatorService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;
    @Autowired(required = false)
    private CustomerVoService customerVoService;
    @Autowired(required = false)
    private CustomerChannelVoService customerChannelVoService;
    @Autowired(required = false)
    private CustomerRetailerVoService customerRetailerVoService;
    @Autowired(required = false)
    private TerminalVoService terminalVoService;

    private static final String REGEX = "\\d{4}-\\d{2}";

    @Override
    public Integer getBatchCount() {
        return 5000;
    }

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, AuditExecuteIndicatorDeductionAmountImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        for (Map.Entry<Integer, AuditExecuteIndicatorDeductionAmountImportVo> row : data.entrySet()) {
            int rowNum = row.getKey();
            AuditExecuteIndicatorDeductionAmountImportVo vo = row.getValue();

            this.validateIsTrue(StringUtils.isNotBlank(vo.getBusinessFormatCode()), "业态不能为空!");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getBusinessUnitCode()), "业务单元不能为空!");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getYearAndMonthStr()), "年月不能为空!");
            if (StringUtil.isEmpty(vo.getYearAndMonthStr())) {
                this.validateIsTrue(false, "年月不能为空!");
            } else {
                try {
                    Date date = DateUtil.date_yyyy_MM.parse(vo.getYearAndMonthStr().trim());
                    this.validateIsTrue(vo.getYearAndMonthStr().trim().equals(DateUtil.date_yyyy_MM.format(date)), "年月格式错误【yyyy-MM】");
                    vo.setYearAndMonth(date);
                } catch (Exception e) {
                    this.validateIsTrue(false, "年月格式错误【yyyy-MM】");
                }
            }
            this.validateIsTrue(StringUtils.isNotBlank(vo.getIndicatorName()), "执行指标名称不能为空!");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getChannelCode()), "渠道编码不能为空!");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getIsQualified()), "状态不能为空!");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getIsQualifiedReason()), "不合格原因不能为空!");
            this.validateIsTrue(StringUtil.isNotEmpty(vo.getRegionInspectCostsStr()), "扣减大区金额不能为空!");
            this.validateIsTrue(StringUtil.isNotEmpty(vo.getInspectCostsStr()), "扣减活动金额不能为空!");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getStoresCode()), "门店编码不能为空!");

            if (StringUtil.isEmpty(vo.getActivityStatus())) {
                this.validateIsTrue(false, "是否关闭活动不能为空!");
            } else {
                this.validateIsTrue(BooleanEnum.FALSE.getSure().equals(vo.getActivityStatus())
                        || BooleanEnum.TRUE.getSure().equals(vo.getActivityStatus()), "是否关闭活动只能是[是/否]!");
            }

            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }
        return errMap;
    }

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, AuditExecuteIndicatorDeductionAmountImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Validate.notNull(data, "导入数据不能为空！");
        log.error("执行指标导入第一步");

        Map<Integer, String> resultErrorMap = new HashMap<>();
        this.validate(data, resultErrorMap);
        if (!resultErrorMap.isEmpty()) {
            return resultErrorMap;
        }
        //大批量保存
        List<AuditExecuteIndicatorDto> importList = (List<AuditExecuteIndicatorDto>) nebulaToolkitService.copyCollectionByWhiteList(data.values(), AuditExecuteIndicatorDeductionAmountImportVo.class, AuditExecuteIndicatorDto.class, LinkedHashSet.class, ArrayList.class);
        auditExecuteIndicatorService.bulkImportSave(importList);
        log.error("执行指标导入第二步");
        return resultErrorMap;
    }

    /**
     * 校验数据
     *
     * @param data data
     **/
    private void validate(LinkedHashMap<Integer, AuditExecuteIndicatorDeductionAmountImportVo> data, Map<Integer, String> resultErrorMap) {

        //获取业态字典
        Map<String, String> formatMap = dictToolkitService.findConvertMapByDictTypeCode(AuditExecuteIndicatorConstant.AUDIT_EXECUTE_BUSINESS_FORMAT);
        //获取业务单元字典
        Map<String, String> unitMap = dictToolkitService.findConvertMapByDictTypeCode(AuditExecuteIndicatorConstant.AUDIT_EXECUTE_BUSINESS_UNIT);
        // MDM_CUSTOMIZE_ORG编码  大区数据字典
        Map<String, String> regionMap = dictToolkitService.findConvertMapByDictTypeCode(AuditExecuteIndicatorConstant.MDM_CUSTOMIZE_ORG);
        Validate.isTrue(ObjectUtils.isNotEmpty(regionMap), "未查询到零售商区域数据字典!");
        Map<String, String> indicatorsMap = dictToolkitService.findConvertMapByDictTypeCode(AuditExecuteIndicatorConstant.TPM_IMPLEMENTATION_INDICATORS);
        Validate.isTrue(ObjectUtils.isNotEmpty(indicatorsMap), "未查询到执行指标数据字典!");
        Map<String, String> qualifiedMap = dictToolkitService.findConvertMapByDictTypeCode(AuditExecuteIndicatorConstant.IS_QUALIFIED);
        Validate.isTrue(ObjectUtils.isNotEmpty(qualifiedMap), "未查询到状态数据字典!");
        Map<String, String> yesMap = dictToolkitService.findConvertMapByDictTypeCode(AuditExecuteIndicatorConstant.YES_OR_NO);
        Validate.isTrue(ObjectUtils.isNotEmpty(yesMap), "未查询到是否关闭活动是个数据字典yes_or_no!");

        data.forEach((index, item) -> {
            try {
                this.validateData(item, formatMap, unitMap, regionMap, indicatorsMap, qualifiedMap, yesMap, resultErrorMap, index);
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                String msg = "第【" + index + "】行错误，错误信息为：" + e.getMessage();
                resultErrorMap.put(index, msg);
            }
        });
    }


    /**
     * 校验数据
     *
     * @param dto       导入数据
     * @param formatMap 业态map
     * @param unitMap   业务单元map
     **/
    private void validateData(AuditExecuteIndicatorDeductionAmountImportVo dto, Map<String, String> formatMap, Map<String, String> unitMap,
                              Map<String, String> regionMap, Map<String, String> indicatorsMap, Map<String, String> qualifiedMap, Map<String, String> yesMap, Map<Integer, String> errMap, Integer rowNum) {

        CustomerChannelVo channelVo = customerChannelVoService.findByCode(dto.getChannelCode());
        if (ObjectUtils.isNotEmpty(channelVo)) {
            dto.setChannelName(channelVo.getCustomerChannelName());
        } else {
            this.validateIsTrue(false, "渠道编码" + dto.getChannelCode() + "未匹配到，请检查！");
        }
        String employeeCode = indicatorsMap.get(dto.getIndicatorName());
        if (StringUtils.isNotBlank(employeeCode)) {
            dto.setIndicatorCode(employeeCode);
            dto.setIndicatorName(employeeCode);
        } else {
            this.validateIsTrue(false, "执行指标名称:" + dto.getIndicatorName() + "未匹配到执行指标编码!");
        }
        String isQualifiedCode = qualifiedMap.get(dto.getIsQualified());
        if (StringUtils.isNotBlank(isQualifiedCode)) {
            dto.setIsQualifiedCode(isQualifiedCode);
        } else {
            this.validateIsTrue(false, "状态:" + dto.getIsQualified() + "未匹配到状态编码!");
        }
        String regionName = dto.getRegionName();
        if (StringUtils.isNotEmpty(regionName)) {
            String regionCode = regionMap.get(regionName);
            if (StringUtils.isNotBlank(regionCode)) {
                dto.setRegionCode(regionCode);
            } else {
                this.validateIsTrue(false, "所属大区名称:" + regionName + "未匹配到所属大区编码!");
            }
        }
        String retailerName = dto.getCustomerRetailerName();
        if (StringUtils.isNotEmpty(retailerName)) {
            List<String> customerRetailerVoList = customerRetailerVoService.findByName(retailerName);
            if (CollectionUtils.isNotEmpty(customerRetailerVoList)) {
                if (customerRetailerVoList.size() == 1) {
                    dto.setCustomerRetailerCode(customerRetailerVoList.get(0));
                } else {
                    this.validateIsTrue(false, "所属零售商名称:" + regionName + "匹配到所属零售商编码不止一个!");
                }
            } else {
                this.validateIsTrue(false, "所属零售商名称:" + regionName + "未匹配到所属零售商编码!");
            }
        }
        if (formatMap.containsKey(dto.getBusinessFormatCode())) {
            dto.setBusinessFormatCode(formatMap.get(dto.getBusinessFormatCode()));
        } else {
            this.validateIsTrue(false, "业态未能识别");
        }
        if (unitMap.containsKey(dto.getBusinessUnitCode())) {
            dto.setBusinessUnitCode(unitMap.get(dto.getBusinessUnitCode()));
        } else {
            this.validateIsTrue(false, "业务单元未能识别");
        }
        try {
            dto.setYearAndMonth(DateUtil.date_yyyy_MM.parse(dto.getYearAndMonthStr()));
        } catch (Exception e) {
            this.validateIsTrue(false, "年月格式错误【yyyy-MM】");
        }
        if (StringUtil.isEmpty(dto.getRegionInspectCostsStr())) {
            this.validateIsTrue(false, "扣减大区金额不能为空!");
        } else {
            try {
                dto.setRegionInspectCosts(new BigDecimal(dto.getRegionInspectCostsStr()));
            } catch (Exception e) {
                log.error("", e);
                this.validateIsTrue(false, "扣减活动金额不合法!");
            }
        }

        if (StringUtil.isEmpty(dto.getInspectCostsStr())) {
            this.validateIsTrue(false, "扣减活动金额不能为空!");
        } else {
            try {
                dto.setInspectCosts(new BigDecimal(dto.getInspectCostsStr()));
            } catch (Exception e) {
                log.error("", e);
                this.validateIsTrue(false, "扣减活动金额不合法!");
            }

        }
        this.validateIsTrue(StringUtil.isNotEmpty(dto.getIndicatorValueStr()), "指标值不能为空!");
        try {
            BigDecimal indicatorValue = new BigDecimal(dto.getIndicatorValueStr());
            dto.setIndicatorValue(indicatorValue);
        } catch (Exception e) {
            this.validateIsTrue(false, "指标值(数字)格式有误");
        }
        dto.setDockingSystem(AuditExecuteIndicatorConstant.AUDIT_EXECUTE_DOCKING_SYSTEM);
        String customerCode = dto.getCustomerCode();
        if (StringUtils.isNotEmpty(customerCode) && StringUtils.isEmpty(dto.getCustomerName())) {
            List<String> codes = new ArrayList<>(1);
            codes.add(customerCode);
            List<CustomerVo> baseByCustomerCodes = customerVoService.findBaseByCustomerCodes(codes);
            if (CollectionUtils.isNotEmpty(baseByCustomerCodes) && baseByCustomerCodes.size() > 0) {
                dto.setCustomerName(baseByCustomerCodes.get(0).getCustomerName());
            } else {
                this.validateIsTrue(false, "客户编码" + customerCode + "未匹配到，请检查！");
            }
        }
        String storesCode = dto.getStoresCode();
        List<String> terminalCodeList = new ArrayList<>(1);
        terminalCodeList.add(storesCode);
        List<TerminalVo> terminalVos = terminalVoService.findDetailsByIdsOrTerminalCodes(null, terminalCodeList);
        if (CollectionUtils.isNotEmpty(terminalVos) && terminalVos.size() > 0) {
            dto.setStoresName(terminalVos.get(0).getTerminalName());
        } else {
            this.validateIsTrue(false, "门店编码" + storesCode + "未匹配到，请检查！");
        }
        String activityStatus = dto.getActivityStatus();
        String yesKey = yesMap.get(activityStatus);
        if (StringUtils.isNotBlank(yesKey)) {
            dto.setActivityStatusCode(yesKey);
        } else {
            this.validateIsTrue(false, "是否关闭活动:" + yesKey + "未匹配到是否关闭活动编码!");
        }

        String errInfo = this.validateGetErrorInfo();
        if (errInfo != null) {
            errMap.put(rowNum, errInfo);
        }
    }


    /**
     * 获取数据实体
     *
     * @return Class<AuditExecuteIndicatorImportVo>
     */
    @Override
    public Class<AuditExecuteIndicatorDeductionAmountImportVo> findCrmExcelVoClass() {
        return AuditExecuteIndicatorDeductionAmountImportVo.class;
    }

    /**
     * 获取业务编码
     *
     * @return String
     */
    @Override
    public String getBusinessCode() {
        return "TPM_AUDIT_EXECUTE_INDICATOR_IMPORT";
    }

    /**
     * 获取业务名称
     *
     * @return String
     */
    @Override
    public String getBusinessName() {
        return "TPM-结案核销执行指标表";
    }

    /**
     * 获取数据实体
     *
     * @return String
     */
    @Override
    public String getTemplateCode() {
        return "TPM_AUDIT_EXECUTE_INDICATOR_DEDUCTION_AMOUNT_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return String
     */
    @Override
    public String getTemplateName() {
        return "TPM-结案核销执行指标表-核减金额";
    }
}
