package com.biz.crm.tpm.business.variable.local.executeIndicator.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.excel.util.Validate;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerOrgChannelStoresVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.product.brand.sdk.service.ProductBrandService;
import com.biz.crm.mdm.business.product.level.sdk.service.ProductLevelVoSdkService;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mdm.business.promoters.sdk.service.PromotersVoService;
import com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.terminal.channel.sdk.service.MdmTerminalChannelVoService;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.service.ActivityDetailPlanItemSdkService;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.vo.ActivityDetailPlanItemVo;
import com.biz.crm.tpm.business.activity.form.sdk.service.ActivityFormService;
import com.biz.crm.tpm.business.activity.form.sdk.vo.ActivityFormVo;
import com.biz.crm.tpm.business.activity.type.sdk.service.ActivityTypeService;
import com.biz.crm.tpm.business.activity.type.sdk.vo.ActivityTypeFormDetailVo;
import com.biz.crm.tpm.business.activity.type.sdk.vo.ActivityTypeRelationDetailVo;
import com.biz.crm.tpm.business.budget.item.sdk.service.BudgetItemService;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemControlConditionVo;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo;
import com.biz.crm.tpm.business.subsidiary.activity.detail.plan.sdk.service.SubComActivityDetailPlanItemVoService;
import com.biz.crm.tpm.business.subsidiary.activity.detail.plan.sdk.vo.SubComActivityDetailPlanItemVo;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.constant.AuditExecuteIndicatorConstant;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.dto.AuditExecuteIndicatorDto;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.service.AuditExecuteIndicatorService;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.vo.AuditExecuteIndicatorImportVo;
import com.biz.crm.tpm.business.variable.sdk.register.FormulaVariableRegister;
import com.biz.crm.tpm.business.variable.sdk.service.SourceCustomVariableService;
import com.biz.crm.tpm.business.variable.sdk.vo.RegisterVariableVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author: chenlong
 * @date: 2022/11/15 15:29
 * @description: 结案核销-执行指标表(AuditExecuteIndicator)导入实现层
 */
@Slf4j
@Component
public class AuditExecuteIndicatorImportProcess implements ImportProcess<AuditExecuteIndicatorImportVo> {

    @Autowired(required = false)
    private AuditExecuteIndicatorService auditExecuteIndicatorService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private List<FormulaVariableRegister> registerList;
    @Autowired(required = false)
    private ActivityTypeService activityTypeService;
    @Autowired(required = false)
    private ActivityFormService activityFormService;
    @Autowired(required = false)
    private ActivityDetailPlanItemSdkService activityDetailPlanItemSdkService;
    @Autowired(required = false)
    private SubComActivityDetailPlanItemVoService subComActivityDetailPlanItemVoService;
    @Autowired(required = false)
    private MdmTerminalChannelVoService mdmTerminalChannelVoService;
    @Autowired(required = false)
    private ProductBrandService productBrandService;
    @Autowired(required = false)
    private ProductLevelVoSdkService productLevelVoSdkService;
    @Autowired(required = false)
    private ProductVoService productVoService;
    @Autowired(required = false)
    private BudgetItemService budgetItemService;
    @Autowired(required = false)
    private TerminalVoService terminalVoService;
    @Autowired(required = false)
    private PromotersVoService promotersVoService;
    @Autowired(required = false)
    private SourceCustomVariableService sourceCustomVariableService;

    @Override
    public Integer getBatchCount() {
        return 50000;
    }

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, AuditExecuteIndicatorImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Validate.notNull(data, "导入数据不能为空！");
        log.error("执行指标导入第一步");
        Map<Integer, String> resultErrorMap = new HashMap<>();
        this.validate(data, resultErrorMap);
        if (CollectionUtil.isEmpty(resultErrorMap)) {
            //大批量保存
            log.error("执行指标导入第二步");
            Collection<AuditExecuteIndicatorDto> dtoList = nebulaToolkitService.copyCollectionByWhiteList(data.values(), AuditExecuteIndicatorImportVo.class, AuditExecuteIndicatorDto.class, LinkedHashSet.class, ArrayList.class);
            auditExecuteIndicatorService.bulkImportSave(new ArrayList<>(dtoList));
        }
        return resultErrorMap;
    }

    /**
     * 校验数据
     *
     * @param data data
     **/
    private void validate(LinkedHashMap<Integer, AuditExecuteIndicatorImportVo> data, Map<Integer, String> resultErrorMap) {

        //获取业态字典
        Map<String, String> formatMap = dictToolkitService.findConvertMapByDictTypeCode(AuditExecuteIndicatorConstant.AUDIT_EXECUTE_BUSINESS_FORMAT);
        //获取业务单元字典
        Map<String, String> unitMap = dictToolkitService.findConvertMapByDictTypeCode(AuditExecuteIndicatorConstant.AUDIT_EXECUTE_BUSINESS_UNIT);
        //获取人员类型字典
        Map<String, String> personMap = dictToolkitService.findConvertMapByDictTypeCode(AuditExecuteIndicatorConstant.AUDIT_EXECUTE_PERSONNEL_TYPE);
        Set<String> cusCodes = new HashSet<>();
        Set<String> typeCodes = new HashSet<>();
        Set<String> formCodes = new HashSet<>();
        Set<String> spCodes = new HashSet<>();

        Set<String> detailPlanItemCodes = new HashSet<>();
        Set<String> subDetailPlanItemCodes = new HashSet<>();

        Set<String> terChannelCodes = new HashSet<>();
        Set<String> brandCodes = new HashSet<>();
        Set<String> proItemCodes = new HashSet<>();
        Set<String> productCodes = new HashSet<>();
        Set<String> budgetCodes = new HashSet<>();
        Set<String> termailCodes = new HashSet<>();
        Set<String> personIds = new HashSet<>();
        data.forEach((index, item) -> {
            try {
                this.validateData(item, formatMap, unitMap, personMap, cusCodes, typeCodes, formCodes, spCodes,
                        detailPlanItemCodes, terChannelCodes, brandCodes, proItemCodes, productCodes, budgetCodes,
                        termailCodes, personIds, subDetailPlanItemCodes, index, resultErrorMap);
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                String msg = "第【" + index + "】行错误，错误信息为：" + e.getMessage();
                resultErrorMap.put(index, msg);
            }
        });

        List<List<String>> cusList = Lists.partition(new ArrayList<>(cusCodes), 500);
        //获取客户绑定信息
        Map<String, CustomerOrgChannelStoresVo> cusMap = customerVoService.getCustomerByCodes(cusList);
        //获取活动分类信息
        Map<String, Map<String, ActivityTypeRelationDetailVo>> typeMap = new HashMap<>();
        if (!typeCodes.isEmpty()) {
            List<ActivityTypeRelationDetailVo> typeList = activityTypeService.getRelationDetailByTypeCodes(new ArrayList<>(typeCodes));
            if (!CollectionUtils.isEmpty(typeList)) {
                typeMap = typeList.stream().collect(Collectors.groupingBy(ActivityTypeRelationDetailVo::getActivityTypeCode,
                        Collectors.toMap(ActivityTypeRelationDetailVo::getBusinessUnitCode, Function.identity())));
            }
        }
        //获取活动形式信息
        Map<String, ActivityFormVo> formMap = new HashMap<>();
        if (!formCodes.isEmpty()) {
            List<ActivityFormVo> formList = activityFormService.findActivityFormByCode(formCodes);
            if (!CollectionUtils.isEmpty(formList)) {
                formMap = formList.stream().collect(Collectors.toMap(ActivityFormVo::getActivityFormCode, Function.identity(), (oldForm, newForm) -> newForm));
            }
        }
        //获取活动分类与活动形式关系
        Map<String, ActivityTypeFormDetailVo> spMap = new HashMap<>();
        if (!spCodes.isEmpty()) {
            List<ActivityTypeFormDetailVo> spList = activityTypeService.getTypeFormBySplicingCodes(new ArrayList<>(spCodes));
            if (!CollectionUtils.isEmpty(spList)) {
                spMap = spList.stream().collect(Collectors.toMap(ActivityTypeFormDetailVo::getSplicingCode, Function.identity(), (oldForm, newForm) -> newForm));
            }
        }
        //获取细案明细信息
        Map<String, ActivityDetailPlanItemVo> detailPlanItemMap = new HashMap();
        if (!detailPlanItemCodes.isEmpty()) {
            List<ActivityDetailPlanItemVo> itemVos = activityDetailPlanItemSdkService.findByActivityDetailItemCodes(detailPlanItemCodes, null);
            if (!CollectionUtils.isEmpty(itemVos)) {
                detailPlanItemMap = itemVos.stream().collect(Collectors.toMap(ActivityDetailPlanItemVo::getDetailPlanItemCode, Function.identity(), (oldForm, newForm) -> newForm));
            }
        }

        //获取分子细案明细信息
        Map<String, SubComActivityDetailPlanItemVo> subDetailPlanItemMap = new HashMap();
        if (!subDetailPlanItemCodes.isEmpty()) {
            List<SubComActivityDetailPlanItemVo> itemVos = subComActivityDetailPlanItemVoService.findItemsByPlanItemCodes(new ArrayList<>(subDetailPlanItemCodes));
            if (!CollectionUtils.isEmpty(itemVos)) {
                subDetailPlanItemMap = itemVos.stream().collect(Collectors.toMap(SubComActivityDetailPlanItemVo::getConstituentDetailPlanItemCode, Function.identity(), (oldForm, newForm) -> newForm));
            }
        }

        //获取门店渠道信息
        Map<String, String> terChannelMap = new HashMap<>();
        if (!terChannelCodes.isEmpty()) {
            terChannelMap = mdmTerminalChannelVoService.findNameByCodes(new ArrayList<>(terChannelCodes));
        }
        //获取品牌信息
        Map<String, String> brandMap = new HashMap<>();
        if (!brandCodes.isEmpty()) {
            brandMap = productBrandService.findNameByCodeList(new ArrayList<>(brandCodes));
        }
        //获取品类品项信息
        Map<String, String> proItemMap = new HashMap<>();
        if (!proItemCodes.isEmpty()) {
            List<ProductLevelVo> levelVos = productLevelVoSdkService.findListByCodes(new ArrayList<>(proItemCodes));
            if (!CollectionUtils.isEmpty(levelVos)) {
                proItemMap = levelVos.stream().collect(Collectors.toMap(ProductLevelVo::getProductLevelCode, ProductLevelVo::getProductLevelName, (oldForm, newForm) -> newForm));
            }
        }
        //获取产品信息
        Map<String, ProductVo> productVoMap = new HashMap<>();
        if (!productCodes.isEmpty()) {
            List<ProductVo> productVos = productVoService.findByCodes(new ArrayList<>(productCodes));
            if (!CollectionUtils.isEmpty(productVos)) {
                productVoMap = productVos.stream().collect(Collectors.toMap(ProductVo::getProductCode, Function.identity(), (oldForm, newForm) -> newForm));
            }
        }
        //获取预算信息
        Map<String, BudgetItemVo> budgetMap = new HashMap<>();
        if (!budgetCodes.isEmpty()) {
            List<BudgetItemVo> itemVos = budgetItemService.listByCodes(new ArrayList<>(budgetCodes));
            if (!CollectionUtils.isEmpty(itemVos)) {
                budgetMap = itemVos.stream().collect(Collectors.toMap(BudgetItemVo::getBudgetItemCode, Function.identity(), (oldForm, newForm) -> newForm));
            }
        }
        //获取门店信息
        Map<String, TerminalVo> termialMap = new HashMap<>();
        if (!termailCodes.isEmpty()) {
            List<TerminalVo> terminalVos = terminalVoService.findBaseByTerminalCodes(new ArrayList<>(termailCodes));
            if (!CollectionUtils.isEmpty(terminalVos)) {
                termialMap = terminalVos.stream().collect(Collectors.toMap(TerminalVo::getTerminalCode, Function.identity(), (oldForm, newForm) -> newForm));
            }
        }
        //获取人员信息
        Map<String, PromotersVo> personIdMap = new HashMap<>();
        if (!personIds.isEmpty()) {
            List<PromotersVo> promotersVos = promotersVoService.findByIdCard(new ArrayList<>(personIds));
            if (!CollectionUtils.isEmpty(promotersVos)) {
                personIdMap = promotersVos.stream().collect(Collectors.toMap(PromotersVo::getIdentityCard, Function.identity(), (oldForm, newForm) -> newForm));
            }
        }
        this.validateCorrect(data, cusMap, typeMap, formMap, spMap,
                detailPlanItemMap, terChannelMap, brandMap, proItemMap, productVoMap, budgetMap,
                termialMap, personIdMap, subDetailPlanItemMap, resultErrorMap);

    }


    /**
     * 校验数据
     *
     * @param dto                 导入数据
     * @param formatMap           业态map
     * @param unitMap             业务单元map
     * @param personMap           人员类型map
     * @param cusCodes            客户编码集合
     * @param typeCodes           活动分类集合
     * @param formCodes           活动形式集合
     * @param spCodes             拼接编码集合
     * @param detailPlanItemCodes 细案明细编码集合
     * @param terChannelCodes     门店渠道编码集合
     * @param brandCodes          品牌编码集合
     * @param proItemCodes        品类品项编码集合
     * @param productCodes        产品编码集合
     * @param budgetCodes         预算项目编码集合
     * @param termailCodes        门店编码集合
     * @param personIds           人员省份证集合
     **/
    private void validateData(AuditExecuteIndicatorImportVo dto, Map<String, String> formatMap, Map<String, String> unitMap, Map<String, String> personMap,
                              Set<String> cusCodes, Set<String> typeCodes, Set<String> formCodes, Set<String> spCodes,
                              Set<String> detailPlanItemCodes, Set<String> terChannelCodes, Set<String> brandCodes,
                              Set<String> proItemCodes, Set<String> productCodes, Set<String> budgetCodes,
                              Set<String> termailCodes, Set<String> personIds,
                              Set<String> subDetailPlanItemCodes,
                              Integer rowNum, Map<Integer, String> errMap) {
        this.validateIsTrue(StringUtils.isNotBlank(dto.getBusinessFormatCode()), "业态不能为空");
        this.validateIsTrue(StringUtils.isNotBlank(dto.getBusinessUnitCode()), "业务单元不能为空");
        this.validateIsTrue(StringUtils.isNotBlank(dto.getYearAndMonthStr()), "年月不能为空");
        this.validateIsTrue(StringUtils.isNotBlank(dto.getSalesInstitutionCode()), "销售机构编码不能为空");
        this.validateIsTrue(StringUtils.isNotBlank(dto.getChannelCode()), "渠道编码不能为空");
        this.validateIsTrue(StringUtils.isNotBlank(dto.getCustomerCode()), "客户编码不能为空");
        this.validateIsTrue(StringUtils.isNotBlank(dto.getVariableCode()), "指标编码不能为空");
        this.validateIsTrue(StringUtils.isNotBlank(dto.getIndicatorName()), "执行指标名称不能为空");
        this.validateIsTrue(StringUtils.isNotBlank(dto.getIndicatorValueStr()), "指标值不能为空");


        if (formatMap.containsKey(dto.getBusinessFormatCode())) {
            dto.setBusinessFormatCode(formatMap.get(dto.getBusinessFormatCode()));
        } else {
            this.validateIsTrue(false, "业态未能识别");
        }
        if (unitMap.containsKey(dto.getBusinessUnitCode())) {
            dto.setBusinessUnitCode(unitMap.get(dto.getBusinessUnitCode()));
        } else {
            this.validateIsTrue(false, "业务单元未能识别");
        }
        try {
            Date date = DateUtil.date_yyyy_MM.parse(dto.getYearAndMonthStr().trim());
            this.validateIsTrue(dto.getYearAndMonthStr().trim().equals(DateUtil.date_yyyy_MM.format(date)), "年月格式错误【yyyy-MM】");
            dto.setYearAndMonth(date);
        } catch (Exception e) {
            this.validateIsTrue(false, "年月格式错误【yyyy-MM】");
        }
        if (StringUtils.isNotBlank(dto.getIndicatorValueStr())) {
            try {
                dto.setIndicatorValue(new BigDecimal(dto.getIndicatorValueStr()));
            } catch (Exception e) {
                this.validateIsTrue(false, "指标值(数字)必须是数字!");
            }
        }
        if (StringUtils.isNotBlank(dto.getActivityBeginDate())) {
            try {
                DateUtil.date_yyyy_MM_dd.parse(dto.getActivityBeginDate());
            } catch (Exception e) {
                this.validateIsTrue(false, "活动开始时间格式错误【yyyy-MM-dd】");
            }
        }
        if (StringUtils.isNotBlank(dto.getActivityEndDate())) {
            try {
                DateUtil.date_yyyy_MM_dd.parse(dto.getActivityEndDate());
            } catch (Exception e) {
                this.validateIsTrue(false, "活动结束时间格式错误【yyyy-MM-dd】");
            }
        }

        //若填写了人员类型
        if (StringUtils.isNotBlank(dto.getPersonnelType())) {
            if (personMap.containsKey(dto.getPersonnelType())) {
                dto.setPersonnelType(personMap.get(dto.getPersonnelType()));
            } else {
                this.validateIsTrue(false, "人员类型未能识别");
            }
        }
        if (StringUtils.isNotBlank(dto.getActivityTypeCode())) {
            typeCodes.add(dto.getActivityTypeCode());
        }
        if (StringUtils.isNotBlank(dto.getActivityFormCode())) {
            formCodes.add(dto.getActivityFormCode());
        }
        if (StringUtils.isNotBlank(dto.getActivityTypeCode()) && StringUtils.isNotBlank(dto.getActivityFormCode())) {
            spCodes.add(dto.getActivityTypeCode() + "-" + dto.getActivityFormCode());
        }
        if (StringUtils.isNotBlank(dto.getActiveNumber())) {
            //若是分子，则需要查询分子的活动
            if (BusinessUnitEnum.SON_COMPANY.getCode().equals(dto.getBusinessUnitCode())) {
                subDetailPlanItemCodes.add(dto.getActiveNumber());
            } else {
                detailPlanItemCodes.add(dto.getActiveNumber());
            }
        }
        if (StringUtils.isNotBlank(dto.getFirstChannelCode())) {
            terChannelCodes.add(dto.getFirstChannelCode());
        }
        if (StringUtils.isNotBlank(dto.getSecondChannelCode())) {
            terChannelCodes.add(dto.getSecondChannelCode());
        }
        if (StringUtils.isNotBlank(dto.getProductBrandCode())) {
            brandCodes.add(dto.getProductBrandCode());
        }
        if (StringUtils.isNotBlank(dto.getProductCategoryCode())) {
            proItemCodes.add(dto.getProductCategoryCode());
        }
        if (StringUtils.isNotBlank(dto.getProductItemCode())) {
            proItemCodes.add(dto.getProductItemCode());
        }
        if (StringUtils.isNotBlank(dto.getProductCode())) {
            productCodes.add(dto.getProductCode());
        }
        if (StringUtils.isNotBlank(dto.getHeadBudgetItemCode())) {
            budgetCodes.add(dto.getHeadBudgetItemCode());
        }
        if (StringUtils.isNotBlank(dto.getMonthBudgetCode())) {
            budgetCodes.add(dto.getMonthBudgetCode());
        }
        if (StringUtils.isNotBlank(dto.getStoresCode())) {
            termailCodes.add(dto.getStoresCode());
        }
        if (StringUtils.isNotBlank(dto.getPersonIdCard())) {
            personIds.add(dto.getPersonIdCard());
        }
        if (StringUtils.isNotBlank(dto.getIsTemporary())) {
            this.validateIsTrue(BooleanEnum.TRUE.getSure().equals(dto.getIsTemporary()) || BooleanEnum.FALSE.getSure().equals(dto.getIsTemporary()),
                    "是否临时活动数据异常，请填写是或否 ");
        }
        dto.setDockingSystem(AuditExecuteIndicatorConstant.AUDIT_EXECUTE_DOCKING_SYSTEM);
        cusCodes.add(dto.getCustomerCode() + dto.getSalesInstitutionCode() + dto.getChannelCode() + dto.getBusinessFormatCode());
        String errInfo = this.validateGetErrorInfo();
        if (errInfo != null) {
            errMap.put(rowNum, errInfo);
        }
    }

    /**
     * 校验数据正确性
     *
     * @param cusMap 客户信息
     **/
    private void validateCorrect(LinkedHashMap<Integer, AuditExecuteIndicatorImportVo> data, Map<String, CustomerOrgChannelStoresVo> cusMap,
                                 Map<String, Map<String, ActivityTypeRelationDetailVo>> typeMap, Map<String, ActivityFormVo> formMap,
                                 Map<String, ActivityTypeFormDetailVo> spMap,
                                 Map<String, ActivityDetailPlanItemVo> detailPlanItemMap, Map<String, String> terChannelMap, Map<String, String> brandMap,
                                 Map<String, String> proItemMap, Map<String, ProductVo> productVoMap, Map<String, BudgetItemVo> budgetMap,
                                 Map<String, TerminalVo> termialMap, Map<String, PromotersVo> personIdMap, Map<String, SubComActivityDetailPlanItemVo> subDetailPlanItemMap, Map<Integer, String> resultErrorMap) {
        //获取变量编码map
        Map<String, String> varMap = new HashMap<>();

        Map<String, String> customVarMap = new HashMap<>();
        try {
            varMap = registerList.stream().collect(Collectors.toMap(FormulaVariableRegister::getVariableCode, FormulaVariableRegister::getVariableName));

            List<RegisterVariableVo> cusVarList = sourceCustomVariableService.findAllVariableList();
            if (!CollectionUtils.isEmpty(cusVarList)) {
                customVarMap = cusVarList.stream().collect(Collectors.toMap(RegisterVariableVo::getCode, RegisterVariableVo::getName, (oldV, newV) -> newV));
            }
        } catch (Exception e) {
            log.error("变量转换map失败：{}", e.getMessage());
            throw new IllegalArgumentException(e);
        }
        List<String> onlyKeys = new ArrayList<>();
        int index;
        for (Map.Entry<Integer, AuditExecuteIndicatorImportVo> entry : data.entrySet()) {
            AuditExecuteIndicatorImportVo dto = entry.getValue();
            index = entry.getKey();
            try {
                String cusKey = dto.getCustomerCode() + dto.getSalesInstitutionCode() + dto.getChannelCode() + dto.getBusinessFormatCode();
                if (!cusMap.containsKey(cusKey)) {
                    this.validateIsTrue(false, "根据客户编码未能查找到客户信息");
                } else {
                    CustomerOrgChannelStoresVo cVo = cusMap.get(cusKey);
                    dto.setCustomerCode(cVo.getCustomerCode());
                    dto.setCustomerName(cVo.getCustomerName());
                    dto.setCustomerErpCode(cVo.getErpCode());
                    dto.setSalesInstitutionCode(cVo.getSalesInstitutionCode());
                    dto.setSalesInstitutionName(cVo.getSalesInstitutionName());
                    dto.setSalesInstitutionErpCode(cVo.getSalesInstitutionErpCode());
                    dto.setSalesRegionCode(cVo.getSalesRegionCode());
                    dto.setSalesRegionName(cVo.getSalesRegionName());
                    dto.setSalesRegionErpCode(cVo.getSalesRegionErpCode());
                    dto.setSalesOrgCode(cVo.getSalesOrgCode());
                    dto.setSalesOrgName(cVo.getSalesOrgName());
                    dto.setSalesOrgErpCode(cVo.getSalesOrgErpCode());
                    if (cVo.getChannelCode().equals(dto.getChannelCode())) {
                        dto.setChannelName(cVo.getChannelName());
                    } else {
                        this.validateIsTrue(false, "此客户未绑定此渠道");
                    }
                }
                if (!varMap.containsKey(dto.getVariableCode()) && !customVarMap.containsKey(dto.getVariableCode())) {
                    this.validateIsTrue(false, "根据指标编码未能查找到指标信息");
                }

                if (StringUtils.isNotBlank(dto.getActivityTypeCode())) {
                    this.validateIsTrue(typeMap.containsKey(dto.getActivityTypeCode()), "未查找到此活动分类");
                    Map<String, ActivityTypeRelationDetailVo> tMap = typeMap.getOrDefault(dto.getActivityTypeCode(), new HashMap<>());
                    this.validateIsTrue(tMap.containsKey(dto.getBusinessUnitCode()), "此活动分类不属于当前业务单元");
                    if (tMap.containsKey(dto.getBusinessUnitCode())) {
                        ActivityTypeRelationDetailVo detailVo = tMap.getOrDefault(dto.getBusinessUnitCode(), new ActivityTypeRelationDetailVo());
                        this.validateIsTrue(StringUtils.isNotBlank(detailVo.getBusinessFormatCode()), "未查找到此活动分类的业态");
                        this.validateIsTrue(detailVo.getBusinessFormatCode().equals(dto.getBusinessFormatCode()), "此活动分类不属于当前业态");
                        dto.setActivityTypeName(detailVo.getActivityTypeName());
                    }
                }
                if (StringUtils.isNotBlank(dto.getActivityFormCode())) {
                    this.validateIsTrue(formMap.containsKey(dto.getActivityFormCode()), "未查找到此活动形式");
                    ActivityFormVo form = formMap.getOrDefault(dto.getActivityFormCode(), new ActivityFormVo());
                    if (formMap.containsKey(dto.getActivityFormCode())) {
                        this.validateIsTrue(StringUtils.isNotBlank(form.getBusinessFormatCode()), "未查找到此活动形式的业态");
                        this.validateIsTrue(form.getBusinessFormatCode().equals(dto.getBusinessFormatCode()), "此活动形式不属于当前业态");
                        this.validateIsTrue(StringUtils.isNotBlank(form.getBusinessUnitCode()), "未查找到此活动形式的业务单元");
                        this.validateIsTrue(form.getBusinessUnitCode().equals(dto.getBusinessUnitCode()), "此活动形式不属于当前业务单元");
                        dto.setActivityFormName(form.getActivityFormName());
                    }
                }

                if (StringUtils.isNotBlank(dto.getActivityTypeCode()) && StringUtils.isNotBlank(dto.getActivityFormCode())) {
                    String spCode = dto.getActivityTypeCode() + "-" + dto.getActivityFormCode();
                    this.validateIsTrue(spMap.containsKey(spCode), "此活动分类未关联此活动形式");
                }

                if (StringUtils.isNotBlank(dto.getActiveNumber())) {
                    this.validateIsTrue(detailPlanItemMap.containsKey(dto.getActiveNumber()), "未查找到此活动细案明细");
                    ActivityDetailPlanItemVo detailPlanItemVo = detailPlanItemMap.getOrDefault(dto.getActiveNumber(), new ActivityDetailPlanItemVo());
                    if (detailPlanItemMap.containsKey(dto.getActiveNumber())) {
                        this.validateIsTrue(StringUtils.isNotBlank(detailPlanItemVo.getBusinessFormatCode()) && detailPlanItemVo.getBusinessFormatCode().equals(dto.getBusinessFormatCode()),
                                "此活动细案明细不属于当前业态");
                        this.validateIsTrue(StringUtils.isNotBlank(detailPlanItemVo.getBusinessUnitCode()) && detailPlanItemVo.getBusinessUnitCode().equals(dto.getBusinessUnitCode()),
                                "此活动细案明细不属于当前业务单元");
                    }
                }

                if (StringUtils.isNotBlank(dto.getFirstChannelCode())) {
                    if (terChannelMap.containsKey(dto.getFirstChannelCode())) {
                        dto.setFirstChannelName(terChannelMap.get(dto.getFirstChannelCode()));
                    } else {
                        this.validateIsTrue(false, "未查找到此一级渠道");
                    }
                }
                if (StringUtils.isNotBlank(dto.getSecondChannelCode())) {
                    if (terChannelMap.containsKey(dto.getSecondChannelCode())) {
                        dto.setSecondChannelName(terChannelMap.get(dto.getSecondChannelCode()));
                    } else {
                        this.validateIsTrue(false, "未查找到此二级渠道");
                    }
                }
                if (StringUtils.isNotBlank(dto.getProductBrandCode())) {
                    if (brandMap.containsKey(dto.getProductBrandCode())) {
                        dto.setProductBrandName(brandMap.get(dto.getProductBrandCode()));
                    } else {
                        this.validateIsTrue(false, "未查找到此品牌");
                    }
                }
                if (StringUtils.isNotBlank(dto.getProductCategoryCode())) {
                    if (proItemMap.containsKey(dto.getProductCategoryCode())) {
                        dto.setProductCategoryName(proItemMap.get(dto.getProductCategoryCode()));
                    } else {
                        this.validateIsTrue(false, "未查找到此品类");
                    }
                }
                if (StringUtils.isNotBlank(dto.getProductItemCode())) {
                    if (proItemMap.containsKey(dto.getProductItemCode())) {
                        dto.setProductItemName(proItemMap.get(dto.getProductItemCode()));
                    } else {
                        this.validateIsTrue(false, "未查找到此品项");
                    }
                }
                if (StringUtils.isNotBlank(dto.getProductCode())) {
                    if (productVoMap.containsKey(dto.getProductCode())) {
                        ProductVo productVo = productVoMap.get(dto.getProductCode());
                        dto.setProductBrandCode(productVo.getProductBrandCode());
                        dto.setProductCategoryCode(productVo.getProductCategoryCode());
                        dto.setProductItemCode(productVo.getProductLevelCode());
                        dto.setProductName(productVo.getProductName());
                    } else {
                        this.validateIsTrue(false, "未查找到此产品");
                    }
                }
                if (StringUtils.isNotBlank(dto.getHeadBudgetItemCode())) {
                    if (budgetMap.containsKey(dto.getHeadBudgetItemCode())) {
                        BudgetItemVo budgetItemVo = budgetMap.get(dto.getHeadBudgetItemCode());
                        this.validateIsTrue(!CollectionUtils.isEmpty(budgetItemVo.getControlConditionDtoList()), "此总部统筹预算项目不属于当前业态业务单元");
                        Set<String> bfSet = budgetItemVo.getControlConditionDtoList().stream().map(BudgetItemControlConditionVo::getBusinessFormatCode).collect(Collectors.toSet());
                        Set<String> buSet = budgetItemVo.getControlConditionDtoList().stream().map(BudgetItemControlConditionVo::getBusinessUnitCode).collect(Collectors.toSet());
                        this.validateIsTrue(!bfSet.isEmpty() && bfSet.contains(dto.getBusinessFormatCode()),
                                "此总部统筹预算项目不属于当前业态");
                        this.validateIsTrue(!buSet.isEmpty() && buSet.contains(dto.getBusinessUnitCode()),
                                "此总部统筹预算项目不属于当前业务单元");
                    } else {
                        this.validateIsTrue(false, "未查找到此总部统筹预算项目");
                    }
                }
                if (StringUtils.isNotBlank(dto.getMonthBudgetCode())) {
                    if (budgetMap.containsKey(dto.getMonthBudgetCode())) {
                        BudgetItemVo budgetItemVo = budgetMap.get(dto.getMonthBudgetCode());
                        this.validateIsTrue(!CollectionUtils.isEmpty(budgetItemVo.getControlConditionDtoList()), "此大区预算项目不属于当前业态业务单元");
                        Set<String> bfSet = budgetItemVo.getControlConditionDtoList().stream().map(BudgetItemControlConditionVo::getBusinessFormatCode).collect(Collectors.toSet());
                        Set<String> buSet = budgetItemVo.getControlConditionDtoList().stream().map(BudgetItemControlConditionVo::getBusinessUnitCode).collect(Collectors.toSet());
                        this.validateIsTrue(!bfSet.isEmpty() && bfSet.contains(dto.getBusinessFormatCode()),
                                "此大区预算项目不属于当前业态");
                        this.validateIsTrue(!buSet.isEmpty() && buSet.contains(dto.getBusinessUnitCode()),
                                "此大区预算项目不属于当前业务单元");
                    } else {
                        this.validateIsTrue(false, "未查找到此大区预算项目");
                    }
                }
                if (StringUtils.isNotBlank(dto.getStoresCode())) {
                    if (termialMap.containsKey(dto.getStoresCode())) {
                        dto.setStoresName(termialMap.get(dto.getStoresCode()).getTerminalName());
                    }

                }
                if (StringUtils.isNotBlank(dto.getPersonIdCard()) && personIdMap.containsKey(dto.getPersonIdCard())) {
                    //                Validate.isTrue(, "未查找到此人员信息[%s]", dto.getPersonIdCard());
                    PromotersVo promotersVo = personIdMap.get(dto.getPersonIdCard());
                    dto.setPersonCode(promotersVo.getEmployeeCode());
                    dto.setPersonName(promotersVo.getName());
                }


                //            dto.setIndicatorName(varMap.get(dto.getVariableCode()));
                dto.setIndicatorName(dto.getVariableCode());
                String onlyKey = dto.getBusinessFormatCode() + dto.getBusinessUnitCode() + DateUtil.dateToStr(dto.getYearAndMonth(), DateUtil.date_yyyy_MM)
                        + dto.getCustomerCode() + dto.getVariableCode();
                if (StringUtils.isNotBlank(dto.getStoresCode())) {
                    onlyKey = onlyKey + dto.getStoresCode();
                }
                onlyKeys.add(onlyKey);
                String errInfo = this.validateGetErrorInfo();
                if (errInfo != null) {
                    resultErrorMap.put(index, errInfo);
                }
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                String msg = "第【" + index + "】行错误，错误信息为：" + e.getMessage();
                resultErrorMap.put(index, msg);
                return;
            }
        }

        //获取历史数据 id
        Map<String, String> oldMap = auditExecuteIndicatorService.findByOnlyKeys(onlyKeys);
        data.forEach((k, dto) -> {
            String onlyKey = dto.getBusinessFormatCode() + dto.getBusinessUnitCode() + DateUtil.dateToStr(dto.getYearAndMonth(), DateUtil.date_yyyy_MM)
                    + dto.getCustomerCode() + dto.getVariableCode();
            if (StringUtils.isNotBlank(dto.getStoresCode())) {
                onlyKey = onlyKey + dto.getStoresCode();
            }
            if (oldMap.containsKey(onlyKey)) {
                dto.setId(oldMap.get(onlyKey));
            }
        });
    }


    /**
     * 获取数据实体
     *
     * @return Class<AuditExecuteIndicatorImportVo>
     */
    @Override
    public Class<AuditExecuteIndicatorImportVo> findCrmExcelVoClass() {
        return AuditExecuteIndicatorImportVo.class;
    }

    /**
     * 获取业务编码
     *
     * @return String
     */
    @Override
    public String getBusinessCode() {
        return "TPM_AUDIT_EXECUTE_INDICATOR_IMPORT";
    }

    /**
     * 获取业务名称
     *
     * @return String
     */
    @Override
    public String getBusinessName() {
        return "TPM-结案核销执行指标表";
    }

    /**
     * 获取数据实体
     *
     * @return String
     */
    @Override
    public String getTemplateCode() {
        return "TPM_AUDIT_EXECUTE_INDICATOR_ONE_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return String
     */
    @Override
    public String getTemplateName() {
        return "TPM-结案核销执行指标表-主体";
    }
}
