package com.biz.crm.tpm.business.variable.local.executeIndicator.service.internal;

import com.alibaba.excel.util.Validate;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.retailer.sdk.service.CustomerRetailerVoService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.promoters.sdk.service.PromotersVoService;
import com.biz.crm.mdm.business.promoters.sdk.vo.PromotersVo;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.constant.AuditExecuteIndicatorConstant;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.dto.AuditExecuteIndicatorDto;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.service.AuditExecuteIndicatorService;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.vo.AuditExecuteIndicatorStaffSalaryImportVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;

/**
 * @author: chenlong
 * @date: 2022/11/15 15:29
 * @description: 结案核销-执行指标表(AuditExecuteIndicator)导入实现层
 */
@Slf4j
@Component
public class AuditExecuteIndicatorStaffSalaryImportProcess implements ImportProcess<AuditExecuteIndicatorStaffSalaryImportVo> {

    @Autowired(required = false)
    private AuditExecuteIndicatorService auditExecuteIndicatorService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;
    @Autowired(required = false)
    private CustomerRetailerVoService customerRetailerVoService;
    @Autowired(required = false)
    private TerminalVoService terminalVoService;
    @Autowired(required = false)
    private PromotersVoService promotersVoService;

    @Override
    public Integer getBatchCount() {
        return 5000;
    }

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, AuditExecuteIndicatorStaffSalaryImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        for (Map.Entry<Integer, AuditExecuteIndicatorStaffSalaryImportVo>  row : data.entrySet()) {
            int rowNum = row.getKey();
            AuditExecuteIndicatorStaffSalaryImportVo vo = row.getValue();

            this.validateIsTrue(StringUtils.isNotBlank(vo.getBusinessFormatCode()), "业态不能为空!");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getBusinessUnitCode()), "业务单元不能为空!");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getYearAndMonthStr()), "年月不能为空!");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getIndicatorName()), "执行指标名称不能为空!");
            this.validateIsTrue(vo.getIndicatorValue() != null, "指标值不能为空!");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getStoresCode()), "门店编码不能为空!");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getStoresName()), "门店名称不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getPersonCode()), "人员编码不能为空!");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getPersonName()), "人员名称不能为空!");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getEmployeeTypeName()), "员工类型名称不能为空");

            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }
        return errMap;
    }

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, AuditExecuteIndicatorStaffSalaryImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Validate.notNull(data, "导入数据不能为空！");
        Map<Integer, String> errMap = new HashMap<>();
        log.error("执行指标导入第一步");
        Map<Integer,String> resultErrorMap = new HashMap<>();
        List<AuditExecuteIndicatorDto> importList = this.validate(data, resultErrorMap);
        if (!resultErrorMap.isEmpty()) {
            return resultErrorMap;
        }
        //大批量保存
        auditExecuteIndicatorService.bulkImportSave(importList);
        log.error("执行指标导入第二步");
        return null;
    }

    /**
     * 校验数据
     *
     * @param data data
     **/
    private List<AuditExecuteIndicatorDto> validate(LinkedHashMap<Integer, AuditExecuteIndicatorStaffSalaryImportVo> data, Map<Integer,String> resultErrorMap) {
        List<AuditExecuteIndicatorStaffSalaryImportVo> importList = new ArrayList<>(data.values());

        //获取业态字典
        Map<String, String> formatMap = dictToolkitService.findConvertMapByDictTypeCode(AuditExecuteIndicatorConstant.AUDIT_EXECUTE_BUSINESS_FORMAT);
        //获取业务单元字典
        Map<String, String> unitMap = dictToolkitService.findConvertMapByDictTypeCode(AuditExecuteIndicatorConstant.AUDIT_EXECUTE_BUSINESS_UNIT);
        // MDM_CUSTOMIZE_ORG编码  大区数据字典
        Map<String, String> regionMap = dictToolkitService.findConvertMapByDictTypeCode(AuditExecuteIndicatorConstant.MDM_CUSTOMIZE_ORG);
        Validate.isTrue(ObjectUtils.isNotEmpty(regionMap), "未查询到零售商区域数据字典!");
        // 员工类型数据字典 MDM_EMPLOYEE_TYPE
        Map<String, String> employeeMap = dictToolkitService.findConvertMapByDictTypeCode(AuditExecuteIndicatorConstant.MDM_EMPLOYEE_TYPE);
        Validate.isTrue(ObjectUtils.isNotEmpty(employeeMap), "未查询到员工类型数据字典!");
        Map<String, String> indicatorsMap = dictToolkitService.findConvertMapByDictTypeCode(AuditExecuteIndicatorConstant.TPM_IMPLEMENTATION_INDICATORS);
        Validate.isTrue(ObjectUtils.isNotEmpty(indicatorsMap), "未查询到执行指标数据字典!");
        int i = 1;
        for (AuditExecuteIndicatorStaffSalaryImportVo item : importList) {
            i++;
            try {
                this.validateData(item, formatMap, unitMap, regionMap, employeeMap, indicatorsMap,i,resultErrorMap);
            } catch (Exception e) {
                log.error(e.getMessage(),e);
                String msg = "第【" + i + "】行错误，错误信息为：" + e.getMessage();
                resultErrorMap.put(i, msg);
                return new ArrayList<>();
            }
        }
        Collection<AuditExecuteIndicatorDto> dtoList = nebulaToolkitService.copyCollectionByWhiteList(importList, AuditExecuteIndicatorStaffSalaryImportVo.class, AuditExecuteIndicatorDto.class, LinkedHashSet.class, ArrayList.class);
        return new ArrayList<>(dtoList);
    }


    /**
     * 校验数据
     *
     * @param dto       导入数据
     * @param formatMap 业态map
     * @param unitMap   业务单元map
     **/
    private void validateData(AuditExecuteIndicatorStaffSalaryImportVo dto, Map<String, String> formatMap, Map<String, String> unitMap,
                              Map<String, String> regionMap, Map<String, String> employeeMap, Map<String, String> indicatorsMap,
                              Integer rowNum, Map<Integer, String> errMap) {
        validateIsTrue(StringUtils.isNotBlank(dto.getBusinessFormatCode()), "业态不能为空!");
        validateIsTrue(StringUtils.isNotBlank(dto.getBusinessUnitCode()), "业务单元不能为空!");
        validateIsTrue(StringUtils.isNotBlank(dto.getYearAndMonthStr()), "年月不能为空!");
        String indicatorName = dto.getIndicatorName();
        validateIsTrue(StringUtils.isNotBlank(indicatorName), "执行指标名称不能为空!");
        validateIsTrue(null != dto.getIndicatorValue(), "指标值不能为空!");
        String employeeCode = indicatorsMap.get(indicatorName);
        validateIsTrue(StringUtils.isNotBlank(employeeCode), "执行指标名称:" + indicatorName + "未匹配到执行指标编码!");
        dto.setIndicatorName(employeeCode);
        String personCode = dto.getPersonCode();
        PromotersVo promotersVo = promotersVoService.findByOutZwCode(personCode);
        if (ObjectUtils.isNotEmpty(promotersVo)) {
            dto.setPersonName(promotersVo.getName());
        } else {
            validateIsTrue(false, "人员编码" + personCode + "未匹配到，请检查！");
        }
        String employeeTypeCode = employeeMap.get(dto.getEmployeeTypeName());
        if (StringUtils.isNotBlank(employeeTypeCode)) {
            dto.setEmployeeTypeCode(employeeTypeCode);
        } else {
            validateIsTrue(false, "员工类型名称:" + dto.getEmployeeTypeName() + "未匹配到员工类型编码!");
        }
        List<String> terminalCodeList = new ArrayList<>(1);
        terminalCodeList.add(dto.getStoresCode());
        List<TerminalVo> terminalVos = terminalVoService.findDetailsByIdsOrTerminalCodes(null, terminalCodeList);
        if (CollectionUtils.isNotEmpty(terminalVos) && terminalVos.size() > 0) {
            dto.setStoresName(terminalVos.get(0).getTerminalName());
        } else {
            validateIsTrue(false, "门店编码" + dto.getStoresCode() + "未匹配到，请检查！");
        }
        String regionName = dto.getRegionName();
        if (StringUtils.isNotEmpty(regionName)) {
            String regionCode = regionMap.get(regionName);
            if (StringUtils.isNotBlank(regionCode)) {
                dto.setRegionCode(regionCode);
            } else {
                validateIsTrue(false, "所属大区名称:" + regionName + "未匹配到所属大区编码!");
            }
        }
        String retailerName = dto.getCustomerRetailerName();
        if (StringUtils.isNotEmpty(retailerName)) {
            List<String> customerRetailerVoList = customerRetailerVoService.findByName(retailerName);
            if (CollectionUtils.isNotEmpty(customerRetailerVoList)) {
                if (customerRetailerVoList.size() == 1) {
                    dto.setCustomerRetailerCode(customerRetailerVoList.get(0));
                } else {
                    validateIsTrue(false,  "所属零售商名称:" + regionName + "匹配到所属零售商编码不止一个!");
                }
            } else {
                validateIsTrue(false, "所属零售商名称:" + regionName + "未匹配到所属零售商编码!");
            }
        }
        if (formatMap.containsKey(dto.getBusinessFormatCode())) {
            dto.setBusinessFormatCode(formatMap.get(dto.getBusinessFormatCode()));
        } else {
            validateIsTrue(false, "业态未能识别");
        }
        if (unitMap.containsKey(dto.getBusinessUnitCode())) {
            dto.setBusinessUnitCode(unitMap.get(dto.getBusinessUnitCode()));
        } else {
            validateIsTrue(false, "业务单元未能识别");
        }
        try {
            dto.setYearAndMonth(DateUtil.date_yyyy_MM.parse(dto.getYearAndMonthStr()));
        } catch (Exception e) {
            validateIsTrue(false, "年月格式错误【yyyy-MM】");
        }
        if (StringUtils.isNotEmpty(dto.getPersonnelId())) {
            String personnelId = dto.getPersonnelId().trim();
            if (StringUtils.isNotEmpty(personnelId)) {
                validateIsTrue(personnelId.length() == 18, "身份证号必须等于18位");
            }
        }
        dto.setDockingSystem(AuditExecuteIndicatorConstant.AUDIT_EXECUTE_DOCKING_SYSTEM);

        String errInfo = this.validateGetErrorInfo();
        if (errInfo != null) {
            errMap.put(rowNum, errInfo);
        }
    }

    /**
     * 获取数据实体
     *
     * @return Class<AuditExecuteIndicatorImportVo>
     */
    @Override
    public Class<AuditExecuteIndicatorStaffSalaryImportVo> findCrmExcelVoClass() {
        return AuditExecuteIndicatorStaffSalaryImportVo.class;
    }

    /**
     * 获取业务编码
     *
     * @return String
     */
    @Override
    public String getBusinessCode() {
        return "TPM_AUDIT_EXECUTE_INDICATOR_IMPORT";
    }

    /**
     * 获取业务名称
     *
     * @return String
     */
    @Override
    public String getBusinessName() {
        return "TPM-结案核销执行指标表";
    }

    /**
     * 获取数据实体
     *
     * @return String
     */
    @Override
    public String getTemplateCode() {
        return "TPM_AUDIT_EXECUTE_INDICATOR_STAFF_SALARY_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return String
     */
    @Override
    public String getTemplateName() {
        return "TPM-结案核销执行指标表-人员工资";
    }
}
