package com.biz.crm.tpm.business.variable.local.register.ecommerce;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.distrib.close.policy.sdk.dto.TpmDistribClosePolicyDto;
import com.biz.crm.tpm.business.distrib.close.policy.sdk.service.TpmDistribClosePolicyService;
import com.biz.crm.tpm.business.distrib.close.policy.sdk.vo.TpmDistribClosePolicyVo;
import com.biz.crm.tpm.business.variable.sdk.dto.CalculateDto;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * @className: DistribClosePolicyCommonService
 * @description: 电商指标-分销结案政策表共用
 * @author: hb
 * @create: 2023-03-15 10:15
 */
@Slf4j
@Service
public abstract class DistribClosePolicyCommonService {

    @Autowired(required = false)
    private TpmDistribClosePolicyService closePolicyService;

    /**
     * 参数验证
     * @param calculate 参数信息
     */
    public void paramVariable(CalculateDto calculate){
        log.info("分销政策表查询参数:{}", JSONUtil.toJsonStr(calculate));

        Validate.notBlank(calculate.getProductCode(), "产品编码不能为空!");
        Validate.notNull(calculate.getStartTimeOrDate(), "活动年月不能为空!");

    }

    /**
     *
     * @param calculate
     * @param type 优先级：first second
     * @return
     */
    private TpmDistribClosePolicyDto distributionGetValueParams(CalculateDto calculate, String type, String activityType){
        /**
         * 取值参数封装 1、当入参1 其中一个不满足时,用参数2 去取值
         * 入参1：客户MDG编码+销售机构+业态+渠道+产品+活动年月，
         * 入参2：客户等级+产品+年月
         */

        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");

        TpmDistribClosePolicyDto distributionClosePolicy = new TpmDistribClosePolicyDto();

        distributionClosePolicy.setProductCode(calculate.getProductCode());
        //客户编码
        String customerErpCode = calculate.getCustomerErpCode();
        //销售机构
        String salesOrganizationErpCode = calculate.getSalesOrganizationErpCode();
        //业态
        String businessFormatCode = calculate.getBusinessFormatCode();
        //电商客户等级编码
        String estoreCustomerLevel = calculate.getEstoreCustomerLevel();

        Date startTimeOrDate = calculate.getStartTimeOrDate();
        Date endTimeOrDate = calculate.getEndTimeOrDate();

        if (StringUtils.equals(DistributionClosePolicyActivityTypeEnum.Last_Basic_Rebate_Monthly_Return.getKey(), activityType)
                || StringUtils.equals(DistributionClosePolicyActivityTypeEnum.LAST_BUNDLE_REBATES.getKey(), activityType)) {
            startTimeOrDate = this.getLastMonthDay(startTimeOrDate);
            endTimeOrDate = this.getLastMonthDay(endTimeOrDate);
        }

        distributionClosePolicy.setStartDate(format.format(startTimeOrDate));
        distributionClosePolicy.setEndDate(format.format(endTimeOrDate));
        distributionClosePolicy.setProductCode(calculate.getProductCode());

        //渠道
        String channel = calculate.getChannel();
        if (StringUtils.isNotEmpty(customerErpCode)
                && StringUtils.isNotEmpty(salesOrganizationErpCode)
                && StringUtils.isNotEmpty(businessFormatCode)
                && StringUtils.isNotEmpty(channel) && StringUtils.equals("first", type)){

            distributionClosePolicy.setCustomerErpCode(customerErpCode);
            distributionClosePolicy.setSalesInstitutionErpCode(salesOrganizationErpCode);
            distributionClosePolicy.setBusinessFormatCode(businessFormatCode);
            distributionClosePolicy.setChannelCode(channel);

            return distributionClosePolicy;

        }else if (StringUtils.isNotEmpty(estoreCustomerLevel) && StringUtils.equals("second", type)){

            distributionClosePolicy.setEstoreCustomerLevel(estoreCustomerLevel);

            return distributionClosePolicy;
        }else {

            return null;

        }

    }

    public BigDecimal getDistributionClosurePolicy(CalculateDto calculate, String activityType) {
        BigDecimal resultValue = BigDecimal.ZERO;
        DistributionClosePolicyActivityTypeEnum activityTypeEnum = DistributionClosePolicyActivityTypeEnum.getActivityType(activityType);
        TpmDistribClosePolicyDto distributionClosePolicy = distributionGetValueParams(calculate, "first", activityType);
        if (ObjectUtil.isNull(distributionClosePolicy)) {
            return BigDecimal.ZERO;
        }
        log.info("getDistributionClosurePolicy 1===>" + JSONObject.toJSONString(distributionClosePolicy));
        //第一条件
        List<TpmDistribClosePolicyVo> list = closePolicyService.findByCondition(distributionClosePolicy, calculate.getYearMonthLy());
        if (CollectionUtils.isEmpty(list)) {
            //第二条件
            distributionClosePolicy = distributionGetValueParams(calculate, "second", activityType);
            if (ObjectUtil.isNull(distributionClosePolicy)) {
                return BigDecimal.ZERO;
            }
            log.info("getDistributionClosurePolicy 2===>" + JSONObject.toJSONString(distributionClosePolicy));
            list = closePolicyService.findByCondition(distributionClosePolicy, calculate.getYearMonthLy());
        }
        if (!CollectionUtils.isEmpty(list)){
            TpmDistribClosePolicyVo distribClosePolicy = list.get(0);
            resultValue = getValue(distribClosePolicy, activityTypeEnum);

        }
        if (Objects.isNull(resultValue)) {
            return BigDecimal.ZERO;
        }
        return resultValue;
    }

    private BigDecimal getValue(TpmDistribClosePolicyVo closePolicy, DistributionClosePolicyActivityTypeEnum activityType){
        BigDecimal resultValue = BigDecimal.ZERO;
        switch (activityType){
            case Basic_Rebate_Ten_Return:
                resultValue = closePolicy.getTenDayReturn();
                break;
            case Basic_Rebate_Monthly_Return:
                resultValue = closePolicy.getLunarReturn();
                break;
            case Last_Basic_Rebate_Monthly_Return:
                resultValue = closePolicy.getLunarReturn();
                break;
            case MONTHLY_GOAL_RETURN:
                resultValue = closePolicy.getMonthlyTargetReturn();
                break;
            case LOT_PRICE_POLICY:
                resultValue = closePolicy.getTakeHomePricePolicy();
                break;
            case PERSONNEL_COSTS:
                resultValue = closePolicy.getStaffCosts();
                break;
            case DELIVERY_COSTS:
                resultValue = closePolicy.getCostOfPlacement();
                break;
            case LOGISTICS_SUPPORT:
                resultValue = closePolicy.getLogisticsSupport();
                break;
            case BUNDLE_REBATES:
                resultValue = closePolicy.getBundleReturnPoint();
                break;
            case LAST_BUNDLE_REBATES:
                resultValue = closePolicy.getBundleReturnPoint();
                break;
            case ADDITIONAL_SUPPORT:
                resultValue = closePolicy.getOtherCooperation();
                break;
            case ADDITIONAL_SUPPORT2:
                resultValue = closePolicy.getOtherCooperationTwo();
                break;
            case YEARLY_RETURN:
                resultValue = closePolicy.getAnnualReturn();
                break;
            case OTHER_SUPPORT_3:
                resultValue = closePolicy.getOtherCooperationThree();
                break;
            case OTHER_SUPPORT_4:
                resultValue = closePolicy.getOtherCooperationFour();
                break;
            case BASIC_REBATE_QUARTER_RETURN:
                resultValue = closePolicy.getQuarterReturn();
                break;
            default:
        }
        return resultValue;
    }

    public enum DistributionClosePolicyActivityTypeEnum{

        Basic_Rebate_Ten_Return("basic_rebate_ten_return", "基础返点-旬返"),

        Basic_Rebate_Monthly_Return("basic_rebate_monthly_return","基础返点-月返"),

        Last_Basic_Rebate_Monthly_Return("last_basic_rebate_monthly_return","上月基础返点-月返"),

        MONTHLY_GOAL_RETURN("monthly_goal_return","月度目标返"),

        LOT_PRICE_POLICY("lot_price_policy","到手价政策"),

        PERSONNEL_COSTS("personnel_costs","人员费用"),

        DELIVERY_COSTS("delivery_costs","投放费用"),

        LOGISTICS_SUPPORT("logistics_support","物流支持"),

        BUNDLE_REBATES("bundle_rebates","捆绑返点"),

        LAST_BUNDLE_REBATES("last_bundle_rebates","捆绑返点"),

        ADDITIONAL_SUPPORT("additional_support","其他支持"),

        ADDITIONAL_SUPPORT2("additional_support2","其他支持2"),

        YEARLY_RETURN("yearly_return","年返"),

        OTHER_SUPPORT_3("other_support_3","其他支持3"),

        OTHER_SUPPORT_4("other_support_4","其他支持4"),

        BASIC_REBATE_QUARTER_RETURN("basic_rebate_quarter_return", "季返"),
        ;

        @Getter
        private String key;

        @Getter
        private String value;

        DistributionClosePolicyActivityTypeEnum(String key, String value) {
            this.key = key;
            this.value = value;
        }

        public static DistributionClosePolicyActivityTypeEnum getActivityType(String activityType){

            DistributionClosePolicyActivityTypeEnum[] values = DistributionClosePolicyActivityTypeEnum.values();

            for (DistributionClosePolicyActivityTypeEnum value : values) {
                if (StringUtils.equals(value.getKey(), activityType)) {
                    return value;
                }
            }

            return null;

        }
    }

    public Date getLastMonthDay(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.MONTH, -1);
        return calendar.getTime();
    }
}
