package com.biz.crm.tpm.business.variable.local.register.ecommerce;

import com.biz.crm.tpm.business.promotion.plan.sdk.service.GeneralExpensesService;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.PromotionPlanService;
import com.biz.crm.tpm.business.promotion.plan.sdk.vo.GeneralExpensesVo;
import com.biz.crm.tpm.business.variable.local.utils.DateUtils;
import com.biz.crm.tpm.business.variable.sdk.dto.CalculateDto;
import com.biz.crm.tpm.business.withholding.detail.local.sdk.service.TpmWithholdingDetailService;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Date;
import java.util.List;

/**
 * @className: PromotionPlanCommonService
 * @description: 促销规划公共类
 * @author: hb
 * @create: 2023-03-17 15:16
 */
@Slf4j
@Service
public abstract class PromotionPlanCommonService {

    @Autowired(required = false)
    private PromotionPlanService promotionPlanService;

    @Autowired(required = false)
    private GeneralExpensesService generalExpensesService;

    @Autowired(required = false)
    private TpmWithholdingDetailService withholdingDetailService;


    public void promotionPlanValidate(CalculateDto calculate){

        Validate.notBlank(calculate.getDetailPlanItemCode(), "明细编码不能为空");
    }

    public BigDecimal getPromotionPlanValue(CalculateDto calculate, GeneralExpensesEnum generalExpensesEnum){

        if (StringUtils.equals(generalExpensesEnum.getKey(), GeneralExpensesEnum.ACTIVITY_ACTUAL_DAY.getKey())) {
            Validate.notNull(calculate.getWithHoldDate(), "预提时间不能为空");
        }
        BigDecimal resultValue = BigDecimal.ZERO;

        String detailPlanItemCode = calculate.getDetailPlanItemCode();

        List<String> codes = Lists.newArrayList();
        codes.add(detailPlanItemCode);

        List<GeneralExpensesVo> resultList = generalExpensesService.findByExpensesCode(codes);
        if (CollectionUtils.isEmpty(resultList)){
            return resultValue;
        }

        GeneralExpensesVo generalExpensesVo = resultList.get(0);

        switch (generalExpensesEnum){
            case ACTIVITY_END_DATE:
                Date endDate = generalExpensesVo.getEndDate();
                LocalDate localDate = DateUtils.date2LocalDate(endDate);
                resultValue = new BigDecimal(localDate.getDayOfMonth());
                break;
            case ACTIVITY_APPLY_AMOUNT:
                resultValue = generalExpensesVo.getApplyAmount();
                break;
            case ACTIVITY_ACTUAL_DAY:
                resultValue = withholdingDate(generalExpensesVo, calculate.getWithHoldDate());
                break;
            case ACTIVITY_TOTAL_DAY:
                Date startDate = generalExpensesVo.getStartDate();
                Date endDate1 = generalExpensesVo.getEndDate();
                Integer day = DateUtils.timeDifferDay(DateUtils.date2LocalDateTime(startDate), DateUtils.date2LocalDateTime(endDate1));
                resultValue = new BigDecimal(day);
                break;
            default:
        }

        return resultValue;

    }

    private BigDecimal withholdingDate(GeneralExpensesVo generalExpensesVo, Date withHole){
        BigDecimal resultDay;

        Date endDate1 = generalExpensesVo.getEndDate();

        //预提时间< 活动结束时间
        if (withHole.before(endDate1)) {

            Date startDate = generalExpensesVo.getStartDate();
            LocalDateTime startDateLocalDateTime = DateUtils.date2LocalDateTime(startDate);
            LocalDateTime createTimeLocalDateTime = DateUtils.date2LocalDateTime(withHole);
            Integer day = DateUtils.timeDifferDay(startDateLocalDateTime, createTimeLocalDateTime);
            resultDay = new BigDecimal(day);

        }else {

            Date beginDate = generalExpensesVo.getStartDate();
            Date endDate = generalExpensesVo.getEndDate();
            LocalDateTime beginLocalDateTime = DateUtils.date2LocalDateTime(beginDate);
            LocalDateTime endLocalDateTime = DateUtils.date2LocalDateTime(endDate);

            Integer day = DateUtils.timeDifferDay(beginLocalDateTime, endLocalDateTime);
            resultDay = new BigDecimal(day);
        }
        return resultDay;
    }

    public enum GeneralExpensesEnum {

        ACTIVITY_END_DATE("activity_end_date", "活动结束日期"),

        ACTIVITY_APPLY_AMOUNT("activity_apply_amount", "电商活动申请金额"),

        ACTIVITY_ACTUAL_DAY("activity_actual_day", "活动实际天数"),

        ACTIVITY_TOTAL_DAY("activity_total_day", "活动总天数");

        @Getter
        private String key;

        @Getter
        private String desc;

        GeneralExpensesEnum(String key, String desc) {
            this.key = key;
            this.desc = desc;
        }
    }

}
