package com.biz.crm.tpm.business.variable.local.register.subcompany;

import com.biz.crm.tpm.business.variable.local.register.auditExecuteIndicator.common.ExecutorIndicatorParamReflectEnum;
import com.biz.crm.tpm.business.variable.sdk.dto.CalculateDto;
import com.biz.crm.tpm.business.variable.sdk.dto.SourceCustomVariableDto;
import com.biz.crm.tpm.business.variable.sdk.enums.VariableFunctionEnum;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.dto.AuditExecuteIndicatorDto;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.service.AuditExecuteIndicatorService;
import com.biz.crm.tpm.business.variable.sdk.executeIndicator.vo.AuditExecuteIndicatorVo;
import com.biz.crm.tpm.business.variable.sdk.register.FormulaCustomVariableRegister;
import com.biz.crm.tpm.business.variable.sdk.register.FormulaVariableRegister;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import io.swagger.annotations.ApiModelProperty;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.util.*;

/**
 * <p>
 * 执行指标类通用变量
 * </p>
 *
 * @author chenshuang
 * @since 2023-04-25
 */
@Slf4j
@Service
public class ExecuteIndicatorCommonRegister implements FormulaCustomVariableRegister {

    @Autowired(required = false)
    private AuditExecuteIndicatorService auditExecuteIndicatorService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    private static final String detailPlanItemCode = "detailPlanItemCode";

    /**
     * 变量编码
     *
     * @return 变量编码
     */
    @Override
    public String getVariableCode() {
        return "executorIndicator";
    }

    /**
     * 变量名称
     *
     * @return 变量名称
     */
    @Override
    public String getVariableName() {
        return "执行指标类通用变量（分子）";
    }

    @Override
    public Integer getSort() {
        return 1;
    }

    @Override
    public List<VariableFunctionEnum> getFunctionEnumList() {
        List<VariableFunctionEnum> list = Lists.newArrayList();
        list.add(VariableFunctionEnum.AUDIT);
        return list;
    }

    @Override
    public Map<String, BigDecimal> calculateVariable(CalculateDto calculateDto, SourceCustomVariableDto sourceCustomVariableDto) {
        Map<String, BigDecimal> resultMap = new HashMap<>();
        resultMap.put(sourceCustomVariableDto.getVariableCode(), BigDecimal.ZERO);
        Validate.notBlank(sourceCustomVariableDto.getParamsJson(), "变量[%s]参数未配置！", sourceCustomVariableDto.getVariableName());
        List<String> params = Arrays.asList(sourceCustomVariableDto.getParamsJson().split(","));
        if (!CollectionUtils.isEmpty(calculateDto.getCustomerCodeList())) {
            params.add(ExecutorIndicatorParamReflectEnum.customerCodeList.getSrcProperty());
            params.remove(ExecutorIndicatorParamReflectEnum.customerCode.getSrcProperty());
        }
        List<AuditExecuteIndicatorDto> paramList = new ArrayList<>();
        if (params.contains(detailPlanItemCode)) {
            AuditExecuteIndicatorDto dto = this.buildDto(calculateDto, Lists.newArrayList(detailPlanItemCode));
            paramList.add(dto);
        } else {
            AuditExecuteIndicatorDto dto = this.buildDto(calculateDto, params);
            paramList = this.buildParamList(dto);
        }
        List<AuditExecuteIndicatorVo> list = new ArrayList<>();
        for (AuditExecuteIndicatorDto dto : paramList) {
            list = auditExecuteIndicatorService.findListForVariable(dto);
            if (!CollectionUtils.isEmpty(list)) {
                break;
            }
        }
        if (CollectionUtils.isEmpty(list)) {
            return resultMap;
        }
        BigDecimal value = list.stream().filter(e -> Objects.nonNull(e.getIndicatorValue()))
                .map(AuditExecuteIndicatorVo::getIndicatorValue).reduce(BigDecimal.ZERO, BigDecimal::add);
        resultMap.put(sourceCustomVariableDto.getVariableCode(), value);
        return resultMap;
    }

    public AuditExecuteIndicatorDto buildDto(CalculateDto calculateDto, List<String> params) {
        AuditExecuteIndicatorDto dto = new AuditExecuteIndicatorDto();
        Class<CalculateDto> srcClazz = CalculateDto.class;
        Class<AuditExecuteIndicatorDto> destClazz = AuditExecuteIndicatorDto.class;
        Set<String> set = new HashSet<>();
        for (String propertyName : params) {
            if (set.contains(propertyName)) {
                continue;
            }
            set.add(propertyName);
            Field srcFiled = null;
            Field destFiled = null;
            Object o = null;
            ExecutorIndicatorParamReflectEnum reflectEnum = null;
            try {
                srcFiled = srcClazz.getDeclaredField(propertyName);
                srcFiled.setAccessible(true);
                o = srcFiled.get(calculateDto);

                reflectEnum = ExecutorIndicatorParamReflectEnum.getBySrcProperty(propertyName);
                Validate.notNull(reflectEnum, "属性[%s]未定义目标映射属性！", propertyName);
                destFiled = destClazz.getDeclaredField(reflectEnum.getDestProperty());
                destFiled.setAccessible(true);
            } catch (Exception e) {
                log.error(e.getMessage(), e);
                throw new RuntimeException("公式计算参数[" + propertyName + "]未定义！");
            }
            ApiModelProperty annotation = srcFiled.getAnnotation(ApiModelProperty.class);
            if (Objects.nonNull(annotation)) {
                String desc = StringUtils.isNotEmpty(annotation.value()) ? annotation.value() : (StringUtils.isNotEmpty(annotation.name()) ? annotation.name() : annotation.notes());
                Validate.notNull(o, desc + "不能为空！");
            } else {
                Validate.notNull(o, "属性[%s]值不能为空！", propertyName);
            }

            try {
                if (reflectEnum.isSplit()) {
                    destFiled.set(dto, Arrays.asList(o.toString().split(",")));
                } else if (destFiled.getType().equals(Collection.class)) {
                    destFiled.set(dto, Lists.newArrayList(o));
                } else {
                    destFiled.set(dto, o);
                }
            } catch (IllegalAccessException e) {
                log.error(e.getMessage(), e);
                throw new RuntimeException("公式计算参数[" + propertyName + "]设置目标映射属性异常！");
            }
        }
        return dto;
    }

    public List<AuditExecuteIndicatorDto> buildParamList(AuditExecuteIndicatorDto dto) {
        List<AuditExecuteIndicatorDto> list = new ArrayList<>();

        if (StringUtils.isNotEmpty(dto.getCustomerCode()) || StringUtils.isNotEmpty(dto.getCustomerErpCode())) {
            AuditExecuteIndicatorDto customerDto = nebulaToolkitService.copyObjectByWhiteList(dto, AuditExecuteIndicatorDto.class, HashSet.class, ArrayList.class);
            customerDto.setStoresCodes(null);
            customerDto.setStoresCode(null);
            list.add(customerDto);
        }
        if (StringUtils.isNotEmpty(dto.getStoresCode())) {
            AuditExecuteIndicatorDto terminalDto = nebulaToolkitService.copyObjectByWhiteList(dto, AuditExecuteIndicatorDto.class, HashSet.class, ArrayList.class);
            terminalDto.setCustomerCode(null);
            terminalDto.setCustomerErpCode(null);
            list.add(terminalDto);
        }

        if (StringUtils.isEmpty(dto.getCustomerCode()) && StringUtils.isEmpty(dto.getCustomerErpCode()) && StringUtils.isEmpty(dto.getStoresCode())) {
            return Lists.newArrayList(dto);
        }

        return list;
    }
}
