package com.biz.crm.tpm.business.variable.local.service.impl;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.tpm.business.subsidiary.activity.detail.plan.sdk.vo.SubComActivityDetailPlanItemVo;
import com.biz.crm.tpm.business.variable.local.entity.ConVariableDetailEntity;
import com.biz.crm.tpm.business.variable.local.repository.ConVariableDetailRepository;
import com.biz.crm.tpm.business.variable.sdk.dto.ConVariableDetailDto;
import com.biz.crm.tpm.business.variable.sdk.service.ConVariableDetailSdkService;
import com.biz.crm.tpm.business.variable.sdk.vo.ConVariableDetailVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

/**
 * @Description 可配置变量明细接口实现
 * @Author YangWei
 * @Date 2023/3/17 上午10:42
 */
@Slf4j
@Service
public class ConVariableDetailSdkServiceImpl implements ConVariableDetailSdkService {

    @Resource
    private ConVariableDetailRepository conVariableDetailRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 删除明细
     * @param detailDto
     */
    @Override
    public void deleteByAuditFormulaCode(ConVariableDetailDto detailDto) {
        Assert.notNull(detailDto, "参数不能为空");
        Assert.hasText(detailDto.getAuditFormulaCode(), "核销公式编码不能为空");
        this.conVariableDetailRepository.deleteByAuditFormulaCode(detailDto.getAuditFormulaCode());
    }

    /**
     * 保存明细
     * @param detailDto
     */
    @Override
    public void batchSave(List<ConVariableDetailDto> detailDto) {
        if(CollectionUtils.isEmpty(detailDto)){
            return;
        }
        detailDto.forEach(o -> {
            o.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            o.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            o.setTenantCode(TenantUtils.getTenantCode());
        });
        Collection<ConVariableDetailEntity> entities = this.nebulaToolkitService.copyCollectionByBlankList(detailDto, ConVariableDetailDto.class,
            ConVariableDetailEntity.class, HashSet.class, ArrayList.class);
        this.conVariableDetailRepository.saveBatch(entities);
    }

    /**
     * 获取明细
     * @param detailDto
     * @return {@link List}<{@link ConVariableDetailVo}>
     */
    @Override
    public List<ConVariableDetailVo> getItemDetail(ConVariableDetailDto detailDto) {
        Assert.notNull(detailDto, "参数不能为空");
        Assert.hasText(detailDto.getAuditFormulaCode(), "核销公式编码不能为空");
        return this.conVariableDetailRepository.getItemDetail(detailDto);
    }

    /**
     * 通过核销公式编码查询可配置变量
     *
     * @param codes
     * @return
     */
    @Override
    public List<ConVariableDetailVo> findConVariableListByCodes(List<String> codes) {
        if (CollectionUtils.isEmpty(codes)) {
            return Lists.newArrayList();
        }
        List<ConVariableDetailEntity> list = this.conVariableDetailRepository.lambdaQuery()
                .eq(ConVariableDetailEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .in(ConVariableDetailEntity::getAuditFormulaCode, codes)
                .list();
        return (List<ConVariableDetailVo>) nebulaToolkitService.copyCollectionByWhiteList(list, ConVariableDetailEntity.class, ConVariableDetailVo.class, HashSet.class, ArrayList.class);
    }

    @Override
    public List<ConVariableDetailVo> findConVariableList(String formulaCode, String variableCode) {
        if (StringUtils.isEmpty(formulaCode) || StringUtils.isEmpty(variableCode)) {
            return Lists.newArrayList();
        }
        List<ConVariableDetailEntity> list = this.conVariableDetailRepository.lambdaQuery()
                .eq(ConVariableDetailEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(ConVariableDetailEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(ConVariableDetailEntity::getAuditFormulaCode, formulaCode)
                .eq(ConVariableDetailEntity::getVariableCode, variableCode)
                .list();
        return (List<ConVariableDetailVo>) nebulaToolkitService.copyCollectionByWhiteList(list, ConVariableDetailEntity.class, ConVariableDetailVo.class, HashSet.class, ArrayList.class);
    }
}