package com.biz.crm.tpm.business.variable.local.service.impl;

import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.tpm.business.variable.local.entity.SourceCustomVariableEntity;
import com.biz.crm.tpm.business.variable.local.repository.SourceCustomVariableRepository;
import com.biz.crm.tpm.business.variable.local.utils.SourceCustomVariableUtil;
import com.biz.crm.tpm.business.variable.sdk.dto.SourceCustomVariableDto;
import com.biz.crm.tpm.business.variable.sdk.dto.log.SourceCustomVariableEventDto;
import com.biz.crm.tpm.business.variable.sdk.enums.CustomVariableSourceEnum;
import com.biz.crm.tpm.business.variable.sdk.event.SourceCustomVariableEventListener;
import com.biz.crm.tpm.business.variable.sdk.service.SourceCustomVariableService;
import com.biz.crm.tpm.business.variable.sdk.vo.RegisterVariableVo;
import com.biz.crm.tpm.business.variable.sdk.vo.SourceCustomVariableVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 *
 * </p>
 *
 * @author chenshuang
 * @since 2023-04-24
 */
@Service
public class SourceCustomVariableServiceImpl implements SourceCustomVariableService {

    @Autowired(required = false)
    private SourceCustomVariableRepository sourceCustomVariableRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private SourceCustomVariableUtil sourceCustomVariableUtil;

    @Override
    public SourceCustomVariableVo findById(String id) {
        List<SourceCustomVariableEntity> entityList = sourceCustomVariableRepository.findByIds(Lists.newArrayList(id));
        Validate.notEmpty(entityList, "数据不存在");
        return nebulaToolkitService.copyObjectByWhiteList(entityList.get(0), SourceCustomVariableVo.class, HashSet.class, ArrayList.class);
    }

    @Transactional
    @Override
    public void create(SourceCustomVariableDto dto) {
        this.sourceCustomVariableUtil.valCreate(dto);
        SourceCustomVariableEntity entity = nebulaToolkitService.copyObjectByWhiteList(dto, SourceCustomVariableEntity.class, HashSet.class, ArrayList.class);
        this.sourceCustomVariableUtil.checkDuplicate(entity);
        this.sourceCustomVariableUtil.packageEntity(entity);
        this.sourceCustomVariableRepository.save(entity);

        //新增业务日志
        SourceCustomVariableEventDto logEventDto = new SourceCustomVariableEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<SourceCustomVariableEventListener, SourceCustomVariableEventDto> onCreate =
                SourceCustomVariableEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, SourceCustomVariableEventListener.class, onCreate);
    }

    @Transactional
    @Override
    public void update(SourceCustomVariableDto dto) {
//        this.sourceCustomVariableUtil.valUpdate(dto);
        SourceCustomVariableVo vo = this.findById(dto.getId());
        Validate.notNull(vo, "数据不存在！");
        SourceCustomVariableEntity entity = nebulaToolkitService.copyObjectByWhiteList(dto, SourceCustomVariableEntity.class, HashSet.class, ArrayList.class);
        this.sourceCustomVariableUtil.checkDuplicate(entity);
        this.sourceCustomVariableUtil.packageEntity(entity);
        this.sourceCustomVariableRepository.updateById(entity);

        //更新业务日志
        SourceCustomVariableEventDto logEventDto = new SourceCustomVariableEventDto();
        logEventDto.setOriginal(vo);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<SourceCustomVariableEventListener, SourceCustomVariableEventDto> onUpdate =
                SourceCustomVariableEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, SourceCustomVariableEventListener.class, onUpdate);
    }

    @Transactional
    @Override
    public void enable(List<String> ids) {
        Validate.notEmpty(ids, "请选择数据！");
        List<SourceCustomVariableEntity> list = this.sourceCustomVariableRepository.findByIds(ids);
        List<SourceCustomVariableEntity> enableList = list.stream().filter(e -> EnableStatusEnum.DISABLE.getCode().equals(e.getEnableStatus())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(enableList)) {
            return;
        }
        List<SourceCustomVariableVo> vos = (List<SourceCustomVariableVo>) nebulaToolkitService.copyCollectionByWhiteList(list, SourceCustomVariableEntity.class, SourceCustomVariableVo.class, HashSet.class, ArrayList.class);
        for (SourceCustomVariableEntity entity : enableList) {
            entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        }
        this.sourceCustomVariableRepository.updateBatchById(enableList);

        //启用业务日志
        SerializableBiConsumer<SourceCustomVariableEventListener, SourceCustomVariableEventDto> onEnable =
                SourceCustomVariableEventListener::onEnable;
        for (SourceCustomVariableVo vo : vos) {
            SourceCustomVariableEventDto logEventDto = new SourceCustomVariableEventDto();
            logEventDto.setOriginal(vo);
            SourceCustomVariableDto dto = nebulaToolkitService.copyObjectByWhiteList(vo, SourceCustomVariableDto.class, HashSet.class, ArrayList.class);
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            logEventDto.setNewest(dto);
            this.nebulaNetEventClient.publish(logEventDto, SourceCustomVariableEventListener.class, onEnable);
        }
    }

    @Transactional
    @Override
    public void disable(List<String> ids) {
        Validate.notEmpty(ids, "请选择数据！");
        List<SourceCustomVariableEntity> list = this.sourceCustomVariableRepository.findByIds(ids);
        List<SourceCustomVariableEntity> disableList = list.stream().filter(e -> EnableStatusEnum.ENABLE.getCode().equals(e.getEnableStatus())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(disableList)) {
            return;
        }
        List<SourceCustomVariableVo> vos = (List<SourceCustomVariableVo>) nebulaToolkitService.copyCollectionByWhiteList(list, SourceCustomVariableEntity.class, SourceCustomVariableVo.class, HashSet.class, ArrayList.class);
        for (SourceCustomVariableEntity entity : disableList) {
            entity.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
        }
        this.sourceCustomVariableRepository.updateBatchById(disableList);

        //禁用业务日志
        SerializableBiConsumer<SourceCustomVariableEventListener, SourceCustomVariableEventDto> onDisable =
                SourceCustomVariableEventListener::onDisable;
        for (SourceCustomVariableVo vo : vos) {
            SourceCustomVariableEventDto logEventDto = new SourceCustomVariableEventDto();
            logEventDto.setOriginal(vo);
            SourceCustomVariableDto dto = nebulaToolkitService.copyObjectByWhiteList(vo, SourceCustomVariableDto.class, HashSet.class, ArrayList.class);
            dto.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            logEventDto.setNewest(dto);
            this.nebulaNetEventClient.publish(logEventDto, SourceCustomVariableEventListener.class, onDisable);
        }
    }

    @Override
    public List<RegisterVariableVo> findAllVariableList() {
        return this.sourceCustomVariableRepository.findAllVariableList();
    }
}
