package com.biz.crm.tpm.business.variable.local.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.tpm.business.budget.controls.config.sdk.service.DimensionControlsService;
import com.biz.crm.tpm.business.budget.controls.config.sdk.service.RulesEnactService;
import com.biz.crm.tpm.business.budget.controls.config.sdk.vo.DimensionControlsVo;
import com.biz.crm.tpm.business.variable.local.entity.SourceCustomVariableEntity;
import com.biz.crm.tpm.business.variable.local.repository.SourceCustomVariableRepository;
import com.biz.crm.tpm.business.variable.local.utils.TestResultUtil;
import com.biz.crm.tpm.business.variable.local.utils.VariableUtil;
import com.biz.crm.tpm.business.variable.sdk.dto.CalculateDto;
import com.biz.crm.tpm.business.variable.sdk.dto.ControlsDto;
import com.biz.crm.tpm.business.variable.sdk.dto.FormulaInfoDto;
import com.biz.crm.tpm.business.variable.sdk.dto.SourceCustomVariableDto;
import com.biz.crm.tpm.business.variable.sdk.enums.VariableFunctionEnum;
import com.biz.crm.tpm.business.variable.sdk.enums.YesOrNoEnum;
import com.biz.crm.tpm.business.variable.sdk.register.FormulaCustomVariableRegister;
import com.biz.crm.tpm.business.variable.sdk.register.FormulaVariableRegister;
import com.biz.crm.tpm.business.variable.sdk.service.VariableService;
import com.biz.crm.tpm.business.variable.sdk.vo.CalculateVo;
import com.biz.crm.tpm.business.variable.sdk.vo.ControlsVo;
import com.biz.crm.tpm.business.variable.sdk.vo.RegisterVariableVo;
import com.biz.crm.tpm.business.variable.sdk.vo.VariableCalTestVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 公式服务
 *
 * @author ：dengwei
 * @date ：Created in 2022/11/18 11:05
 */
@Service
@Slf4j
public class VariableServiceImpl implements VariableService {

    @Autowired(required = false)
    private List<FormulaVariableRegister> formulaVariableRegisterList;

    @Autowired(required = false)
    private VariableUtil variableUtil;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Resource
    private DimensionControlsService dimensionControlsService;

    @Resource
    private RulesEnactService rulesEnactService;

    @Resource
    private SourceCustomVariableRepository sourceCustomVariableRepository;


    /**
     * 计算所有条件&公式
     *
     * @param calculateDtoList 计算条件
     * @return 计算结果
     */
    @Override
    public List<CalculateVo> allCalculateConditionAndExpression(List<CalculateDto> calculateDtoList) {
        Validate.notNull(calculateDtoList, "计算公式时，参数不能为空");
        List<CalculateVo> calculateVoList = new ArrayList<>();
        for (CalculateDto calculateDto : calculateDtoList) {
            //匹配的公式
            List<FormulaInfoDto> formulaInfoList = calculateDto.getFormulaInfoDtoList();
            Validate.notNull(formulaInfoList, "未匹配到公式结果");
            //获取变量值
            Map<String, BigDecimal> variableValueMap = getVariableValue(calculateDto, formulaInfoList);
            //计算
            List<CalculateVo> calculateResultList = variableUtil.allCalculateConditionAndExpression(calculateDto, variableValueMap, formulaInfoList);
            Validate.noNullElements(calculateResultList, "公式计算失败");
            String code = calculateDto.getCode();
            calculateResultList.forEach(calculateResult -> calculateResult.setCode(code));
            calculateVoList.addAll(calculateResultList);
        }
        return calculateVoList;
    }

    /**
     * 仅计算所有条件
     *
     * @param calculateDtoList 计算条件
     * @return 计算结果
     */
    @Override
    public List<CalculateVo> allCalculateCondition(List<CalculateDto> calculateDtoList) {
        Validate.notNull(calculateDtoList, "计算公式时，参数不能为空");
        List<CalculateVo> calculateVoList = new ArrayList<>();
        for (CalculateDto calculateDto : calculateDtoList) {
            //匹配的公式
            List<FormulaInfoDto> formulaInfoList = calculateDto.getFormulaInfoDtoList();
            Validate.notNull(formulaInfoList, "未匹配到公式结果");
            //获取变量值
            Map<String, BigDecimal> variableValueMap = getVariableValue(calculateDto, formulaInfoList);
            //计算
            List<CalculateVo> calculateResultList = variableUtil.allCalculateCondition(calculateDto, variableValueMap, formulaInfoList);
            Validate.noNullElements(calculateResultList, "公式计算失败");
            String code = calculateDto.getCode();
            calculateResultList.forEach(calculateResult -> calculateResult.setCode(code));
            calculateVoList.addAll(calculateResultList);
        }
        return calculateVoList;
    }

    /**
     * 计算所有公式
     *
     * @param calculateDtoList 计算条件
     * @return 计算结果
     */
    @Override
    public List<CalculateVo> allCalculateExpression(List<CalculateDto> calculateDtoList) {
        Validate.notNull(calculateDtoList, "计算公式时，参数不能为空");
        List<CalculateVo> calculateVoList = new ArrayList<>();
        for (CalculateDto calculateDto : calculateDtoList) {
            //匹配的公式
            List<FormulaInfoDto> formulaInfoList = calculateDto.getFormulaInfoDtoList();
            Validate.notNull(formulaInfoList, "未匹配到公式结果");
            //获取变量值
            Map<String, BigDecimal> variableValueMap = getVariableValue(calculateDto, formulaInfoList);
            //计算
            List<CalculateVo> calculateResultList = variableUtil.allCalculateExpression(calculateDto, variableValueMap, formulaInfoList);
            Validate.noNullElements(calculateResultList, "公式计算失败");
            String code = calculateDto.getCode();
            calculateResultList.forEach(calculateResult -> calculateResult.setCode(code));
            calculateVoList.addAll(calculateResultList);
        }
        return calculateVoList;
    }

    /**
     * 或计算条件&公式（一个CalculateDto只计算满足一条就返回）
     *
     * @param calculateDtoList 计算条件
     * @return 计算结果
     */
    @Override
    public List<CalculateVo> orCalculateConditionAndExpression(List<CalculateDto> calculateDtoList) {
        Validate.notNull(calculateDtoList, "计算公式时，参数不能为空");
        List<CalculateVo> calculateVoList = new ArrayList<>();
        for (CalculateDto calculateDto : calculateDtoList) {
            //匹配的公式
            List<FormulaInfoDto> formulaInfoList = calculateDto.getFormulaInfoDtoList();
            Validate.notNull(formulaInfoList, "未匹配到公式结果");
            //获取变量值
            Map<String, BigDecimal> variableValueMap = getVariableValue(calculateDto, formulaInfoList);
            //计算
            CalculateVo calculateVo = variableUtil.orCalculateConditionAndExpression(calculateDto, variableValueMap, formulaInfoList);
            if (Objects.isNull(calculateVo)) {
                calculateVo = nebulaToolkitService.copyObjectByWhiteList(calculateDto, CalculateVo.class, HashSet.class, ArrayList.class);
                calculateVo.setFormulaConditionValue(Boolean.FALSE);
            }
            calculateVo.setVariableValueMap(variableValueMap);
            calculateVoList.add(calculateVo);
        }
        return calculateVoList;
    }

    /**
     * 或计算条件（一个CalculateDto只计算满足一条就返回）
     *
     * @param calculateDtoList 计算条件
     * @return 计算结果
     */
    @Override
    public List<CalculateVo> orCalculateCondition(List<CalculateDto> calculateDtoList) {
        Validate.notNull(calculateDtoList, "计算公式时，参数不能为空");
        List<CalculateVo> calculateVoList = new ArrayList<>();
        for (CalculateDto calculateDto : calculateDtoList) {
            //匹配的公式
            List<FormulaInfoDto> formulaInfoList = calculateDto.getFormulaInfoDtoList();
            Validate.notNull(formulaInfoList, "未匹配到公式结果");
            //获取变量值
            Map<String, BigDecimal> variableValueMap = getVariableValue(calculateDto, formulaInfoList);
            //计算
            CalculateVo calculateVo = variableUtil.orCalculateCondition(calculateDto, variableValueMap, formulaInfoList);
            if (Objects.isNull(calculateVo)) {
                calculateVo = nebulaToolkitService.copyObjectByWhiteList(calculateDto, CalculateVo.class, HashSet.class, ArrayList.class);
                calculateVo.setFormulaConditionValue(Boolean.FALSE);
            }
            calculateVoList.add(calculateVo);
        }
        return calculateVoList;
    }

    /**
     * 或计算公式（一个CalculateDto只计算满足一条就返回）
     *
     * @param calculateDtoList 计算条件
     * @return 计算结果
     */
    @Override
    public List<CalculateVo> orCalculateExpression(List<CalculateDto> calculateDtoList) {
        Validate.notNull(calculateDtoList, "计算公式时，参数不能为空");
        List<CalculateVo> calculateVoList = new ArrayList<>();
        for (CalculateDto calculateDto : calculateDtoList) {
            //匹配的公式
            List<FormulaInfoDto> formulaInfoList = calculateDto.getFormulaInfoDtoList();
            Validate.notNull(formulaInfoList, "未匹配到公式结果");
            //获取变量值
            Map<String, BigDecimal> variableValueMap = getVariableValue(calculateDto, formulaInfoList);
            //计算
            CalculateVo calculateVo = variableUtil.orCalculateExpression(calculateDto, variableValueMap, formulaInfoList);
            if (Objects.isNull(calculateVo)) {
                calculateVo = nebulaToolkitService.copyObjectByWhiteList(calculateDto, CalculateVo.class, HashSet.class, ArrayList.class);
                calculateVo.setFormulaConditionValue(Boolean.FALSE);
            }
            calculateVoList.add(calculateVo);
        }
        return calculateVoList;
    }

    /**
     * 获取变量列表
     *
     * @param isConfigure 是否可配置
     * @return 变量列表
     */
    @Override
    public List<RegisterVariableVo> variableList(String isConfigure, String function, String name, Integer pageSize) {
        List<RegisterVariableVo> registerVariableVoList = new ArrayList<>();
        Validate.notNull(function, "功能不能为空！");
        if (!CollectionUtils.isEmpty(formulaVariableRegisterList)) {
            if (StringUtils.isEmpty(name)) {
                formulaVariableRegisterList.forEach(register -> {
                    if (register.isConfigurable() != (isConfigure.equals(BooleanEnum.TRUE.getNumStr()))) {
                        return;
                    }
                    List<String> functionStringList = register.getFunctionStringList();
                    if (CollectionUtils.isEmpty(functionStringList) || (!functionStringList.contains(function))) {
                        return;
                    }
                    RegisterVariableVo registerVariableVo = new RegisterVariableVo();
                    registerVariableVo.setCode(register.getVariableCode());
                    registerVariableVo.setName(register.getVariableName());
                    registerVariableVo.setType(String.join(",", register.getFunctionStringList()));
                    registerVariableVo.setSort(register.getSort());
                    registerVariableVoList.add(registerVariableVo);
                });
            } else {
                Pattern compile = Pattern.compile(name, Pattern.CASE_INSENSITIVE);
                formulaVariableRegisterList.forEach(register -> {
                    Matcher matcher = compile.matcher(register.getVariableName());
                    if (matcher.find()) {
                        if (register.isConfigurable() != (isConfigure.equals(BooleanEnum.TRUE.getNumStr()))) {
                            return;
                        }
                        List<String> functionStringList = register.getFunctionStringList();
                        if (CollectionUtils.isEmpty(functionStringList) || (!functionStringList.contains(function))) {
                            return;
                        }
                        RegisterVariableVo registerVariableVo = new RegisterVariableVo();
                        registerVariableVo.setCode(register.getVariableCode());
                        registerVariableVo.setName(register.getVariableName());
                        registerVariableVo.setType(String.join(",", register.getFunctionStringList()));
                        registerVariableVo.setSort(register.getSort());
                        registerVariableVoList.add(registerVariableVo);
                    }
                });
            }

        }
        int currSize = registerVariableVoList.size();
        if (!isConfigure.equals(BooleanEnum.TRUE.getNumStr()) && VariableFunctionEnum.AUDIT.getCode().equals(function)) {
            //补充读取单独配置的自定义变量，按分页数量来查，差多少查多少
            //目前只有核销变量
            if (Objects.isNull(pageSize)) {
                pageSize = 200;
            }
            if (currSize < pageSize) {
                List<RegisterVariableVo> customVariableList = sourceCustomVariableRepository.findForVariableList(name, pageSize - currSize);
                registerVariableVoList.addAll(customVariableList);
            }
        }
        return registerVariableVoList.stream().sorted(Comparator.comparingInt(RegisterVariableVo::getSort)).collect(Collectors.toList());
    }

    @Override
    public List<RegisterVariableVo> variableAuthList(String isConfigure, String function, String name, Integer pageSize) {
        Validate.notNull(function, "功能不能为空！");
        return this.sourceCustomVariableRepository.variableAuthList(
                StringUtils.equals(BooleanEnum.TRUE.getNumStr(), isConfigure) ? YesOrNoEnum.YES.getCode() : YesOrNoEnum.NO.getCode(),
                function,
                name,
                pageSize);
    }

    /**
     * 变量编码-名称映射
     *
     * @return
     */
    @Override
    public Map<String, String> getVariableMap(String function, List<String> variableCodeList) {
        List<RegisterVariableVo> variableVos = new ArrayList<>();
        formulaVariableRegisterList.forEach(register -> {
            List<String> functionStringList = register.getFunctionStringList();
            if (CollectionUtils.isEmpty(functionStringList) || (!functionStringList.contains(function))) {
                return;
            }
            if (!CollectionUtils.isEmpty(variableCodeList) && !variableCodeList.contains(register.getVariableCode())) {
                return;
            }
            variableCodeList.remove(register.getVariableCode());
            RegisterVariableVo registerVariableVo = new RegisterVariableVo();
            registerVariableVo.setCode(register.getVariableCode());
            registerVariableVo.setName(register.getVariableName());
            registerVariableVo.setSort(register.getSort());
            variableVos.add(registerVariableVo);
        });
        Map<String, String> result = variableVos.stream().collect(Collectors.toMap(RegisterVariableVo::getCode, RegisterVariableVo::getName, (v1, v2) -> v2));
        if (!CollectionUtils.isEmpty(variableCodeList)) {
            List<SourceCustomVariableEntity> customVariableList = sourceCustomVariableRepository.findByVariableCodes(variableCodeList);
            Map<String, String> map = customVariableList.stream().collect(Collectors.toMap(SourceCustomVariableEntity::getVariableCode, SourceCustomVariableEntity::getVariableName, (v1, v2) -> v2));
            result.putAll(map);
        }
        return result;
    }

    /**
     * 测试计算
     *
     * @param dto
     * @return
     */
    @Override
    public VariableCalTestVo calTest(CalculateDto dto) {
        List<CalculateVo> calculateVos = this.orCalculateConditionAndExpression(Lists.newArrayList(dto));
        //核销变量编码-名称映射
        List<String> variableCodeList = new ArrayList<>();
        for (CalculateVo calculateVo : calculateVos) {
            if (Objects.nonNull(calculateVo.getVariableValueMap())) {
                variableCodeList.addAll(calculateVo.getVariableValueMap().keySet());
            }
        }
        Map<String, String> variableNameMap = this.getVariableMap(VariableFunctionEnum.AUDIT.getCode(), variableCodeList);
        return TestResultUtil.build(dto, calculateVos, variableNameMap);
    }

    /**
     * 获取变量值
     *
     * @param calculateDto    calculateDto
     * @param formulaInfoList formulaInfoList
     * @return 变量值
     */
    private Map<String, BigDecimal> getVariableValue(CalculateDto calculateDto, List<FormulaInfoDto> formulaInfoList) {
        Map<String, BigDecimal> variableValueMap = new HashMap<>();
        Set<String> variableNamesSet = variableUtil.getAllVariableNameSet(formulaInfoList);
        log.info("自定义变量,variableNamesSet:{}",JSONObject.toJSONString(variableNamesSet));
        log.info("计算参数:{}",JSONObject.toJSONString(calculateDto));
        Validate.notEmpty(variableNamesSet, "自定义变量%s不存在！", formulaInfoList);
        variableNamesSet.forEach(variableName -> {
            List<FormulaVariableRegister> list = formulaVariableRegisterList.stream().filter(register -> variableName.equals(register.getVariableCode())).collect(Collectors.toList());
            if (CollectionUtils.isEmpty(list)) {
                List<SourceCustomVariableEntity> customVariableList = this.sourceCustomVariableRepository.findByVariableCodes(Lists.newArrayList(variableName));
                Validate.notEmpty(customVariableList, "自定义变量[%s]不存在！", variableName);
                FormulaCustomVariableRegister formulaCustomVariableRegister = variableUtil.getCustomRegister(customVariableList);
                Validate.notNull(formulaCustomVariableRegister, "报错信息自定义变量[%s]所属公用变量注册器不存在", variableName);
//                Validate.notNull(formulaCustomVariableRegister, "核销基准数据没有符号连接", variableName);
                SourceCustomVariableDto sourceCustomVariableDto = nebulaToolkitService.copyObjectByWhiteList(customVariableList.get(0), SourceCustomVariableDto.class, HashSet.class, ArrayList.class);
                variableValueMap.putAll(formulaCustomVariableRegister.calculateVariable(calculateDto, sourceCustomVariableDto));
            } else {
                FormulaVariableRegister formulaVariableRegister = list.get(0);
                variableValueMap.putAll(formulaVariableRegister.calculateVariable(calculateDto));
            }
        });
        return variableValueMap;
    }

    /**
     * 预算整体管控
     *
     * @param controlsDto
     * @return {@link DimensionControlsVo}
     */
    @Override
    public ControlsVo overallControl(ControlsDto controlsDto) {
        this.verifyParams(controlsDto);
        ControlsVo controlsVo = new ControlsVo();
        return controlsVo;
    }

    /**
     * 测试公式配置是否合法
     *
     * @param formulaInfoList
     */
    @Override
    public void testFormula(List<FormulaInfoDto> formulaInfoList) {
        if (CollectionUtils.isEmpty(formulaInfoList)) {
            log.info("测试公式配置是否合法======>待验证数据为空");
            return;
        }
        log.info("测试公式配置是否合法======>{}", JSONObject.toJSONString(formulaInfoList));
        Map<String, BigDecimal> variableValueMap = new HashMap<>();
        Set<String> variableNamesSet = variableUtil.getAllVariableNameSet(formulaInfoList);
        List<SourceCustomVariableEntity> customVariableEntityList = this.sourceCustomVariableRepository.findByVariableCodes(Lists.newArrayList(variableNamesSet));
        Map<String, SourceCustomVariableEntity> customVariableEntityMap = customVariableEntityList.stream().collect(Collectors.toMap(SourceCustomVariableEntity::getVariableCode, v -> v, (v1, v2) -> v2));
        variableNamesSet.forEach(variableName -> {
            List<FormulaVariableRegister> list = formulaVariableRegisterList.stream().filter(register -> variableName.equals(register.getVariableCode())).collect(Collectors.toList());
            if (CollectionUtils.isEmpty(list)) {
                //Validate.isTrue(customVariableEntityMap.containsKey(variableName), "自定义变量[%s]注册器不存在", variableName);
                Validate.isTrue(customVariableEntityMap.containsKey(variableName), "核销基准数据没有符号连接", variableName);
                variableValueMap.put(customVariableEntityMap.get(variableName).getVariableCode(), BigDecimal.ONE);
                return;
            }
            Validate.notEmpty(list, "[%s]变量注册器不存在", variableName);
            FormulaVariableRegister formulaVariableRegister = list.get(0);
            variableValueMap.put(formulaVariableRegister.getVariableCode(), BigDecimal.ONE);
        });
        formulaInfoList.forEach(e -> {
            variableUtil.computeFormulaCondition(variableValueMap, e, true);
            variableUtil.computeFormulaExpression(variableValueMap, e, true);
        });
    }

    private void verifyParams(ControlsDto controlsDto) {
        Assert.hasText(controlsDto.getBusinessFormatCode(), "业态不能为空");
        Assert.hasText(controlsDto.getBusinessUnitCode(), "业务单元不能为空");
    }
}
