package com.biz.crm.tpm.business.variable.local.service.impl;

import com.biz.crm.tpm.business.variable.local.utils.VariableUtil;
import com.biz.crm.tpm.business.variable.sdk.dto.CalculateDto;
import com.biz.crm.tpm.business.variable.sdk.dto.FormulaInfoDto;
import com.biz.crm.tpm.business.variable.sdk.register.FormulaVariableUsingRedisRegister;
import com.biz.crm.tpm.business.variable.sdk.service.VariableUsingRedisService;
import com.biz.crm.tpm.business.variable.sdk.vo.CalculateVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 公式服务
 *
 * @author ：duyiran
 * @date ：2022/12/09 15:05
 */
@Service
@Slf4j
public class VariableUsingRedisServiceImpl implements VariableUsingRedisService {

    @Autowired(required = false)
    private List<FormulaVariableUsingRedisRegister> formulaVariableUsingRedisRegisterList;

    @Autowired(required = false)
    private VariableUtil variableUtil;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 计算所有条件&公式
     *
     * @param calculateDtoList 计算条件
     * @return 计算结果
     */
    @Override
    public List<CalculateVo> allCalculateConditionAndExpression(List<CalculateDto> calculateDtoList) {

        Validate.notNull(calculateDtoList, "计算公式时，参数不能为空");

        //根据变量名称汇总计算Dto
        Map<String, List<CalculateDto>> variableNameToCalculateDtoListMap = this.groupCalculateDtoByVariableName(calculateDtoList);

        //上传变量值到redis，并得到变量与注册器的映射关系
        Map<String, FormulaVariableUsingRedisRegister> variableNameToVariableRegister = this.upLoadVariableValuesToRedis(variableNameToCalculateDtoListMap);

        List<CalculateVo> calculateVoList = new ArrayList<>();
        for (CalculateDto calculateDto : calculateDtoList) {
            //匹配的公式
            List<FormulaInfoDto> formulaInfoList = calculateDto.getFormulaInfoDtoList();
            //获取变量值
            Map<String, BigDecimal> variableValueMap = getVariableValue(calculateDto, formulaInfoList, variableNameToVariableRegister);
            //计算
            List<CalculateVo> calculateResultList = variableUtil.allCalculateConditionAndExpression(calculateDto,variableValueMap, formulaInfoList);
            Validate.noNullElements(calculateResultList, "公式计算失败");
            String code = calculateDto.getCode();
            calculateResultList.forEach(calculateResult -> calculateResult.setCode(code));
            calculateVoList.addAll(calculateResultList);
        }
        return calculateVoList;
    }

    /**
     * 仅计算所有条件
     *
     * @param calculateDtoList 计算条件
     * @return 计算结果
     */
    @Override
    public List<CalculateVo> allCalculateCondition(List<CalculateDto> calculateDtoList) {

        Validate.notNull(calculateDtoList, "计算公式时，参数不能为空");

        //根据变量名称汇总计算Dto
        Map<String, List<CalculateDto>> variableNameToCalculateDtoListMap = this.groupCalculateDtoByVariableName(calculateDtoList);

        //上传变量值到redis，并得到变量与注册器的映射关系
        Map<String, FormulaVariableUsingRedisRegister> variableNameToVariableRegister = this.upLoadVariableValuesToRedis(variableNameToCalculateDtoListMap);

        List<CalculateVo> calculateVoList = new ArrayList<>();
        for (CalculateDto calculateDto : calculateDtoList) {
            //匹配的公式
            List<FormulaInfoDto> formulaInfoList = calculateDto.getFormulaInfoDtoList();
            //获取变量值
            Map<String, BigDecimal> variableValueMap = getVariableValue(calculateDto, formulaInfoList, variableNameToVariableRegister);
            //计算
            List<CalculateVo> calculateResultList = variableUtil.allCalculateCondition(calculateDto,variableValueMap, formulaInfoList);
            Validate.noNullElements(calculateResultList, "公式计算失败");
            String code = calculateDto.getCode();
            calculateResultList.forEach(calculateResult -> calculateResult.setCode(code));
            calculateVoList.addAll(calculateResultList);
        }
        return calculateVoList;
    }

    /**
     * 计算所有公式
     *
     * @param calculateDtoList 计算条件
     * @return 计算结果
     */
    @Override
    public List<CalculateVo> allCalculateExpression(List<CalculateDto> calculateDtoList) {

        Validate.notNull(calculateDtoList, "计算公式时，参数不能为空");

        //根据变量名称汇总计算Dto
        Map<String, List<CalculateDto>> variableNameToCalculateDtoListMap = this.groupCalculateDtoByVariableName(calculateDtoList);

        //上传变量值到redis，并得到变量与注册器的映射关系
        Map<String, FormulaVariableUsingRedisRegister> variableNameToVariableRegister = this.upLoadVariableValuesToRedis(variableNameToCalculateDtoListMap);

        List<CalculateVo> calculateVoList = new ArrayList<>();
        for (CalculateDto calculateDto : calculateDtoList) {
            //匹配的公式
            List<FormulaInfoDto> formulaInfoList = calculateDto.getFormulaInfoDtoList();
            //获取变量值
            Map<String, BigDecimal> variableValueMap = getVariableValue(calculateDto, formulaInfoList, variableNameToVariableRegister);
            //计算
            List<CalculateVo> calculateResultList = variableUtil.allCalculateExpression(calculateDto,variableValueMap, formulaInfoList);
            Validate.noNullElements(calculateResultList, "公式计算失败");
            String code = calculateDto.getCode();
            calculateResultList.forEach(calculateResult -> calculateResult.setCode(code));
            calculateVoList.addAll(calculateResultList);
        }
        return calculateVoList;
    }

    /**
     * 或计算条件（一个CalculateDto只计算满足一条就返回）
     *
     * @param calculateDtoList 计算条件
     * @return 计算结果
     */
    @Override
    public List<CalculateVo> orCalculateConditionAndExpression(List<CalculateDto> calculateDtoList) {

        Validate.notNull(calculateDtoList, "计算公式时，参数不能为空");

        //根据变量名称汇总计算Dto
        Map<String, List<CalculateDto>> variableNameToCalculateDtoListMap = this.groupCalculateDtoByVariableName(calculateDtoList);

        //上传变量值到redis，并得到变量与注册器的映射关系
        Map<String, FormulaVariableUsingRedisRegister> variableNameToVariableRegister = this.upLoadVariableValuesToRedis(variableNameToCalculateDtoListMap);

        List<CalculateVo> calculateVoList = new ArrayList<>();
        for (CalculateDto calculateDto : calculateDtoList) {
            //匹配的公式
            List<FormulaInfoDto> formulaInfoList = calculateDto.getFormulaInfoDtoList();
            //获取变量值
            Map<String, BigDecimal> variableValueMap = getVariableValue(calculateDto, formulaInfoList, variableNameToVariableRegister);
            //计算
            CalculateVo calculateVo = variableUtil.orCalculateConditionAndExpression(calculateDto, variableValueMap, formulaInfoList);
            if (Objects.isNull(calculateVo)) {
                calculateVo = nebulaToolkitService.copyObjectByWhiteList(calculateDto, CalculateVo.class, HashSet.class, ArrayList.class);
                calculateVo.setFormulaConditionValue(Boolean.FALSE);
            }
            calculateVoList.add(calculateVo);
        }
        return calculateVoList;
    }

    /**
     * 仅计算所有条件
     *
     * @param calculateDtoList 计算条件
     * @return 计算结果
     */
    @Override
    public List<CalculateVo> orCalculateCondition(List<CalculateDto> calculateDtoList) {

        Validate.notNull(calculateDtoList, "计算公式时，参数不能为空");

        //根据变量名称汇总计算Dto
        Map<String, List<CalculateDto>> variableNameToCalculateDtoListMap = this.groupCalculateDtoByVariableName(calculateDtoList);

        //上传变量值到redis，并得到变量与注册器的映射关系
        Map<String, FormulaVariableUsingRedisRegister> variableNameToVariableRegister = this.upLoadVariableValuesToRedis(variableNameToCalculateDtoListMap);

        List<CalculateVo> calculateVoList = new ArrayList<>();
        for (CalculateDto calculateDto : calculateDtoList) {
            //匹配的公式
            List<FormulaInfoDto> formulaInfoList = calculateDto.getFormulaInfoDtoList();
            //获取变量值
            Map<String, BigDecimal> variableValueMap = getVariableValue(calculateDto, formulaInfoList, variableNameToVariableRegister);
            //计算
            CalculateVo calculateVo = variableUtil.orCalculateCondition(calculateDto,variableValueMap, formulaInfoList);
            if (Objects.isNull(calculateVo)) {
                calculateVo = nebulaToolkitService.copyObjectByWhiteList(calculateDto, CalculateVo.class, HashSet.class, ArrayList.class);
                calculateVo.setFormulaConditionValue(Boolean.FALSE);
            }
            calculateVoList.add(calculateVo);
        }
        return calculateVoList;
    }

    /**
     * 计算所有公式
     *
     * @param calculateDtoList 计算条件
     * @return 计算结果
     */
    @Override
    public List<CalculateVo> orCalculateExpression(List<CalculateDto> calculateDtoList) {

        Validate.notNull(calculateDtoList, "计算公式时，参数不能为空");

        //根据变量名称汇总计算Dto
        Map<String, List<CalculateDto>> variableNameToCalculateDtoListMap = this.groupCalculateDtoByVariableName(calculateDtoList);

        //上传变量值到redis，并得到变量与注册器的映射关系
        Map<String, FormulaVariableUsingRedisRegister> variableNameToVariableRegister = this.upLoadVariableValuesToRedis(variableNameToCalculateDtoListMap);

        List<CalculateVo> calculateVoList = new ArrayList<>();
        for (CalculateDto calculateDto : calculateDtoList) {
            //匹配的公式
            List<FormulaInfoDto> formulaInfoList = calculateDto.getFormulaInfoDtoList();
            //获取变量值
            Map<String, BigDecimal> variableValueMap = getVariableValue(calculateDto, formulaInfoList, variableNameToVariableRegister);
            //计算
            CalculateVo calculateVo = variableUtil.orCalculateExpression(calculateDto,variableValueMap, formulaInfoList);
            if (Objects.isNull(calculateVo)) {
                calculateVo = nebulaToolkitService.copyObjectByWhiteList(calculateDto, CalculateVo.class, HashSet.class, ArrayList.class);
                calculateVo.setFormulaConditionValue(Boolean.FALSE);
            }
            calculateVoList.add(calculateVo);
        }
        return calculateVoList;
    }

    /**
     * 根据变量名称汇总计算Dto
     *
     * @param calculateDtoList 计算条件
     * @return 汇总结果
     */
    private Map<String, List<CalculateDto>> groupCalculateDtoByVariableName(List<CalculateDto> calculateDtoList) {
        Map<String, List<CalculateDto>> variableNameToCalculateDtoListMap = Maps.newConcurrentMap();
        calculateDtoList.forEach(calculateDto -> {
            List<FormulaInfoDto> formulaInfoDtoList = calculateDto.getFormulaInfoDtoList();
            Validate.notNull(formulaInfoDtoList, "计算公式时，公式不能为空");
            Set<String> variableNamesSet = variableUtil.getAllVariableNameSet(formulaInfoDtoList);
            variableNamesSet.forEach(variableName -> {
                if (variableNameToCalculateDtoListMap.containsKey(variableName)) {
                    List<CalculateDto> variableNameToCalculateDtoList = variableNameToCalculateDtoListMap.get(variableName);
                    variableNameToCalculateDtoList.add(calculateDto);
                } else {
                    List<CalculateDto> variableNameToCalculateDtoList = Lists.newArrayList();
                    variableNameToCalculateDtoList.add(calculateDto);
                    variableNameToCalculateDtoListMap.put(variableName, variableNameToCalculateDtoList);
                }
            });
        });
        return variableNameToCalculateDtoListMap;
    }

    /**
     * 上传变量值到redis，并返回变量与注册器的映射关系
     *
     * @param variableNameToCalculateDtoListMap variableNameToCalculateDtoListMap
     * @return 变量与注册器的映射关系
     */
    private Map<String, FormulaVariableUsingRedisRegister> upLoadVariableValuesToRedis(Map<String, List<CalculateDto>> variableNameToCalculateDtoListMap) {
        Map<String, FormulaVariableUsingRedisRegister> variableNameToVariableRegister = new HashMap<>(variableNameToCalculateDtoListMap.size() * 2);
        variableNameToCalculateDtoListMap.forEach((key, calculateDtoList) -> {
            List<FormulaVariableUsingRedisRegister> list = formulaVariableUsingRedisRegisterList.stream()
                    .filter(register -> key.equals(register.getVariableCode()))
                    .collect(Collectors.toList());
            Validate.noNullElements(list, "变量注册器不存在");
            FormulaVariableUsingRedisRegister formulaVariableRegister = list.get(0);
            formulaVariableRegister.upLoadValuesToRedis(calculateDtoList);
            variableNameToVariableRegister.put(key, formulaVariableRegister);
        });
        return variableNameToVariableRegister;
    }

    /**
     * 获取变量值
     *
     * @param calculateDto                   calculateDto
     * @param formulaInfoList                formulaInfoList
     * @param variableNameToVariableRegister variableNameToVariableRegister
     * @return 变量值
     */
    private Map<String, BigDecimal> getVariableValue(CalculateDto calculateDto, List<FormulaInfoDto> formulaInfoList, Map<String, FormulaVariableUsingRedisRegister> variableNameToVariableRegister) {
        Set<String> variableNamesSet = variableUtil.getAllVariableNameSet(formulaInfoList);
        Map<String, BigDecimal> variableValueMap = new HashMap<>(variableNamesSet.size() * 2);
        variableNamesSet.forEach(variableName -> {
            FormulaVariableUsingRedisRegister formulaVariableUsingRedisRegister = variableNameToVariableRegister.get(variableName);
            Validate.notNull(formulaVariableUsingRedisRegister, "变量注册器不存在");
            variableValueMap.putAll(formulaVariableUsingRedisRegister.calculateVariableUsingRedis(calculateDto));
        });
        return variableValueMap;
    }

}
