package com.biz.crm.tpm.business.variable.local.service.impl;

import com.biz.crm.tpm.business.variable.local.utils.VariableUtil;
import com.biz.crm.tpm.business.variable.sdk.dto.CalculateDto;
import com.biz.crm.tpm.business.variable.sdk.dto.CalculateWarningDto;
import com.biz.crm.tpm.business.variable.sdk.dto.FormulaInfoDto;
import com.biz.crm.tpm.business.variable.sdk.register.FormulaWarningVariableUsingRedisRegister;
import com.biz.crm.tpm.business.variable.sdk.service.WarningVariableUsingRedisService;
import com.biz.crm.tpm.business.variable.sdk.vo.CalculateVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 预警专用公式服务，其他模块禁止使用
 *
 * @author ：duyiran
 * @date ： 2022/12/09 15:05
 */
@Service
@Slf4j
public class WarningVariableUsingRedisServiceImpl implements WarningVariableUsingRedisService {

    @Autowired(required = false)
    private List<FormulaWarningVariableUsingRedisRegister> formulaWarningVariableUsingRedisRegisterList;

    @Autowired(required = false)
    private VariableUtil variableUtil;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 预警专用公式计算，其他功能禁止使用
     *
     * @param calculateDtoList 计算条件
     * @return 计算结果
     */
    @Override
    public List<CalculateVo> allCalculateForWarning(List<CalculateWarningDto> calculateDtoList) {

        Validate.notNull(calculateDtoList, "计算公式时，参数不能为空");

        //根据变量名称汇总计算Dto
        Map<String, List<CalculateWarningDto>> variableNameToCalculateDtoListMap = this.groupCalculateDtoByVariableName(calculateDtoList);

        //得到变量与注册器的映射关系
        Map<String, FormulaWarningVariableUsingRedisRegister> variableNameToVariableRegisterMap = this.getVariableNameToVariableRegister(variableNameToCalculateDtoListMap);

        //上传变量值到redis,并得到具体执行预警的calculateDto集合
        Set<CalculateWarningDto> completeCalculateWarningDtoList = this.upLoadVariableValuesToRedis(variableNameToCalculateDtoListMap,variableNameToVariableRegisterMap);

        Collection<CalculateDto> completeCalculateDtoList = nebulaToolkitService.copyCollectionByWhiteList(completeCalculateWarningDtoList, CalculateWarningDto.class,
                CalculateDto.class, HashSet.class, ArrayList.class);

        List<CalculateVo> calculateVoList = new ArrayList<>();
        for (CalculateDto calculateDto : completeCalculateDtoList) {
            //匹配的公式
            List<FormulaInfoDto> formulaInfoList = calculateDto.getFormulaInfoDtoList();
            //获取变量值
            Map<String, BigDecimal> variableValueMap = getVariableValue(calculateDto, formulaInfoList, variableNameToVariableRegisterMap);
            //计算
            List<CalculateVo> calculateResultList = variableUtil.allCalculateCondition(calculateDto, variableValueMap, formulaInfoList);
            Validate.noNullElements(calculateResultList, "公式计算失败");
            String code = calculateDto.getCode();
            calculateResultList.forEach(calculateResult -> calculateResult.setCode(code));
            calculateVoList.addAll(calculateResultList);
        }
        return calculateVoList;
    }

    /**
     * 根据变量名称汇总计算Dto
     *
     * @param calculateDtoList 计算条件
     * @return 汇总结果
     */
    private Map<String, List<CalculateWarningDto>> groupCalculateDtoByVariableName(List<CalculateWarningDto> calculateDtoList) {
        Map<String, List<CalculateWarningDto>> variableNameToCalculateDtoListMap = Maps.newConcurrentMap();
        calculateDtoList.forEach(calculateDto -> {
            List<FormulaInfoDto> formulaInfoDtoList = calculateDto.getFormulaInfoDtoList();
            Validate.notNull(formulaInfoDtoList, "计算公式时，公式不能为空");
            Set<String> variableNamesSet = variableUtil.getAllVariableNameSet(formulaInfoDtoList);
            variableNamesSet.forEach(variableName -> {
                if (variableNameToCalculateDtoListMap.containsKey(variableName)) {
                    List<CalculateWarningDto> variableNameToCalculateDtoList = variableNameToCalculateDtoListMap.get(variableName);
                    variableNameToCalculateDtoList.add(calculateDto);
                } else {
                    List<CalculateWarningDto> variableNameToCalculateDtoList = Lists.newArrayList();
                    variableNameToCalculateDtoList.add(calculateDto);
                    variableNameToCalculateDtoListMap.put(variableName, variableNameToCalculateDtoList);
                }
            });
        });
        return variableNameToCalculateDtoListMap;
    }

    /**
     * 得到变量与注册器的映射关系
     * @param variableNameToCalculateDtoListMap variableNameToCalculateDtoListMap
     * @return 变量与注册器的映射关系
     */
    private Map<String, FormulaWarningVariableUsingRedisRegister> getVariableNameToVariableRegister(Map<String, List<CalculateWarningDto>> variableNameToCalculateDtoListMap) {
        Map<String, FormulaWarningVariableUsingRedisRegister> variableNameToVariableRegister = new HashMap<>(variableNameToCalculateDtoListMap.size() * 2);
        variableNameToCalculateDtoListMap.forEach((key, calculateDtoList) -> {
            List<FormulaWarningVariableUsingRedisRegister> list = formulaWarningVariableUsingRedisRegisterList.stream()
                    .filter(register -> key.equals(register.getVariableCode()))
                    .collect(Collectors.toList());
            Validate.noNullElements(list, "变量注册器不存在");
            FormulaWarningVariableUsingRedisRegister formulaWarningVariableUsingRedisRegister = list.get(0);
            variableNameToVariableRegister.put(key, formulaWarningVariableUsingRedisRegister);
        });
        return variableNameToVariableRegister;
    }

    /**
     * 上传变量值到redis，并返回变量与注册器的映射关系
     *
     * @param variableNameToCalculateDtoListMap variableNameToCalculateDtoListMap
     * @param variableNameToVariableRegisterMap variableNameToVariableRegisterMap
     * @return 变量与注册器的映射关系
     */
    private Set<CalculateWarningDto> upLoadVariableValuesToRedis(Map<String, List<CalculateWarningDto>> variableNameToCalculateDtoListMap, Map<String, FormulaWarningVariableUsingRedisRegister> variableNameToVariableRegisterMap) {
        Set<CalculateWarningDto> calculateWarningDtoSet = Sets.newConcurrentHashSet();
        variableNameToCalculateDtoListMap.forEach((key, calculateDtoList) -> {
            FormulaWarningVariableUsingRedisRegister formulaWarningVariableUsingRedisRegister = variableNameToVariableRegisterMap.get(key);
            Validate.notNull(formulaWarningVariableUsingRedisRegister, "变量注册器不存在");
            List<CalculateWarningDto> calculateWarningDtoList = formulaWarningVariableUsingRedisRegister.upLoadWarningValuesToRedis(calculateDtoList);
            calculateWarningDtoSet.addAll(calculateWarningDtoList);
        });
        return calculateWarningDtoSet;
    }

    /**
     * 获取变量值
     *
     * @param calculateDto                   calculateDto
     * @param formulaInfoList                formulaInfoList
     * @param variableNameToVariableRegister variableNameToVariableRegister
     * @return 变量值
     */
    private Map<String, BigDecimal> getVariableValue(CalculateDto calculateDto, List<FormulaInfoDto> formulaInfoList, Map<String, FormulaWarningVariableUsingRedisRegister> variableNameToVariableRegister) {
        Set<String> variableNamesSet = variableUtil.getAllVariableNameSet(formulaInfoList);
        Map<String, BigDecimal> variableValueMap = new HashMap<>(variableNamesSet.size() * 2);
        variableNamesSet.forEach(variableName -> {
            FormulaWarningVariableUsingRedisRegister formulaVariableUsingRedisRegister = variableNameToVariableRegister.get(variableName);
            Validate.notNull(formulaVariableUsingRedisRegister, "变量注册器不存在");
            variableValueMap.putAll(formulaVariableUsingRedisRegister.calculateVariableUsingRedis(calculateDto));
        });
        return variableValueMap;
    }

}
