package com.biz.crm.tpm.business.variable.local.utils;

import com.alibaba.excel.util.StringUtils;
import com.biz.crm.tpm.business.variable.local.entity.SourceCustomVariableEntity;
import com.biz.crm.tpm.business.variable.sdk.dto.CalculateDto;
import com.biz.crm.tpm.business.variable.sdk.dto.FormulaInfoDto;
import com.biz.crm.tpm.business.variable.sdk.register.FormulaCustomVariableRegister;
import com.biz.crm.tpm.business.variable.sdk.vo.CalculateVo;
import com.biz.crm.tpm.business.variable.sdk.vo.VariableConditionResultVo;
import com.biz.crm.tpm.business.variable.sdk.vo.VariableResultVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author duyiran
 * @date 2022-12-09 10:53
 */

@Component
public class VariableUtil {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private List<FormulaCustomVariableRegister> formulaCustomVariableRegisters;

    /**
     * 或计算条件&公式（满足一条就返回）
     *
     * @param calculateDto     calculateDto
     * @param variableValueMap variableValueMap
     * @param formulaInfoList  formulaInfoList
     * @return 计算结果
     */
    public CalculateVo orCalculateConditionAndExpression(CalculateDto calculateDto, Map<String, BigDecimal> variableValueMap, List<FormulaInfoDto> formulaInfoList) {
        Validate.isTrue(!CollectionUtils.isEmpty(formulaInfoList), "公式不能为空");
        for (FormulaInfoDto formulaInfo : formulaInfoList) {
            VariableConditionResultVo variableConditionResultVo = computeFormulaCondition(variableValueMap, formulaInfo, false);
            if (variableConditionResultVo.getValue()) {
                VariableResultVo variableResultVo = computeFormulaExpression(variableValueMap, formulaInfo, false);
                CalculateVo calculateVo = nebulaToolkitService.copyObjectByWhiteList(calculateDto, CalculateVo.class, HashSet.class, ArrayList.class);
                calculateVo.setFormulaConditionName(variableConditionResultVo.getFormulaConditionName());
                calculateVo.setFormulaConditionValue(variableConditionResultVo.getValue());
                calculateVo.setFormulaValue(variableResultVo.getValue());
                calculateVo.setFormulaName(variableResultVo.getFormulaName());
                calculateVo.setVariableValueMap(variableValueMap);
                return calculateVo;
            }
        }
        return null;
    }

    /**
     * 计算全部条件&公式
     *
     * @param calculateDto     calculateDto
     * @param variableValueMap variableValueMap
     * @param formulaInfoList  formulaInfoList
     * @return 计算结果
     */
    public List<CalculateVo> allCalculateConditionAndExpression(CalculateDto calculateDto, Map<String, BigDecimal> variableValueMap, List<FormulaInfoDto> formulaInfoList) {
        Validate.isTrue(!CollectionUtils.isEmpty(formulaInfoList), "公式不能为空");
        List<CalculateVo> calculateVoList = Lists.newArrayList();
        for (FormulaInfoDto formulaInfo : formulaInfoList) {
            VariableConditionResultVo variableConditionResultVo = computeFormulaCondition(variableValueMap, formulaInfo, false);
            if (variableConditionResultVo.getValue()) {
                VariableResultVo variableResultVo = computeFormulaExpression(variableValueMap, formulaInfo, false);
                CalculateVo calculateVo = nebulaToolkitService.copyObjectByWhiteList(calculateDto, CalculateVo.class, HashSet.class, ArrayList.class);
                calculateVo.setFormulaConditionName(variableConditionResultVo.getFormulaConditionName());
                calculateVo.setFormulaConditionValue(variableConditionResultVo.getValue());
                calculateVo.setFormulaValue(variableResultVo.getValue());
                calculateVo.setFormulaName(variableResultVo.getFormulaName());
                calculateVo.setVariableValueMap(variableValueMap);
                calculateVoList.add(calculateVo);
            } else {
                CalculateVo calculateVo = nebulaToolkitService.copyObjectByWhiteList(calculateDto, CalculateVo.class, HashSet.class, ArrayList.class);
                calculateVo.setFormulaConditionName(variableConditionResultVo.getFormulaConditionName());
                calculateVo.setFormulaConditionValue(variableConditionResultVo.getValue());
                calculateVo.setVariableValueMap(variableValueMap);
                calculateVoList.add(calculateVo);
            }
        }
        return calculateVoList;
    }

    /**
     * 或计算条件（满足一条就返回）
     *
     * @param calculateDto     calculateDto
     * @param variableValueMap variableValueMap
     * @param formulaInfoList  formulaInfoList
     * @return 计算结果
     */
    public CalculateVo orCalculateCondition(CalculateDto calculateDto, Map<String, BigDecimal> variableValueMap, List<FormulaInfoDto> formulaInfoList) {
        Validate.isTrue(!CollectionUtils.isEmpty(formulaInfoList), "公式不能为空");
        for (FormulaInfoDto formulaInfo : formulaInfoList) {
            VariableConditionResultVo variableConditionResultVo = computeFormulaCondition(variableValueMap, formulaInfo, false);
            CalculateVo calculateVo = nebulaToolkitService.copyObjectByWhiteList(calculateDto, CalculateVo.class, HashSet.class, ArrayList.class);
            calculateVo.setFormulaConditionName(variableConditionResultVo.getFormulaConditionName());
            calculateVo.setFormulaConditionValue(variableConditionResultVo.getValue());
            calculateVo.setVariableValueMap(variableValueMap);
            return calculateVo;
        }
        return null;
    }

    /**
     * 计算全部条件
     *
     * @param calculateDto     calculateDto
     * @param variableValueMap variableValueMap
     * @param formulaInfoList  formulaInfoList
     * @return 计算结果
     */
    public List<CalculateVo> allCalculateCondition(CalculateDto calculateDto, Map<String, BigDecimal> variableValueMap, List<FormulaInfoDto> formulaInfoList) {
        Validate.isTrue(!CollectionUtils.isEmpty(formulaInfoList), "公式不能为空");
        List<CalculateVo> calculateVoList = Lists.newArrayList();
        for (FormulaInfoDto formulaInfo : formulaInfoList) {
            VariableConditionResultVo variableConditionResultVo = computeFormulaCondition(variableValueMap, formulaInfo, false);
            CalculateVo calculateVo = nebulaToolkitService.copyObjectByWhiteList(calculateDto, CalculateVo.class, HashSet.class, ArrayList.class);
            calculateVo.setFormulaConditionName(variableConditionResultVo.getFormulaConditionName());
            calculateVo.setFormulaConditionValue(variableConditionResultVo.getValue());
            calculateVo.setVariableValueMap(variableValueMap);
            calculateVoList.add(calculateVo);
        }
        return calculateVoList;
    }

    /**
     * 或计算公式（满足一条就返回）
     *
     * @param calculateDto     calculateDto
     * @param variableValueMap variableValueMap
     * @param formulaInfoList  formulaInfoList
     * @return 计算结果
     */
    public CalculateVo orCalculateExpression(CalculateDto calculateDto, Map<String, BigDecimal> variableValueMap, List<FormulaInfoDto> formulaInfoList) {
        Validate.isTrue(!CollectionUtils.isEmpty(formulaInfoList), "公式不能为空");
        for (FormulaInfoDto formulaInfo : formulaInfoList) {
            VariableResultVo variableResultVo = computeFormulaExpression(variableValueMap, formulaInfo, false);
            CalculateVo calculateVo = nebulaToolkitService.copyObjectByWhiteList(calculateDto, CalculateVo.class, HashSet.class, ArrayList.class);
            calculateVo.setFormulaValue(variableResultVo.getValue());
            calculateVo.setFormulaName(variableResultVo.getFormulaName());
            calculateVo.setVariableValueMap(variableValueMap);
            return calculateVo;
        }
        return null;
    }

    /**
     * 计算全部公式
     *
     * @param calculateDto     calculateDto
     * @param variableValueMap variableValueMap
     * @param formulaInfoList  formulaInfoList
     * @return 计算结果
     */
    public List<CalculateVo> allCalculateExpression(CalculateDto calculateDto, Map<String, BigDecimal> variableValueMap, List<FormulaInfoDto> formulaInfoList) {
        Validate.isTrue(!CollectionUtils.isEmpty(formulaInfoList), "公式不能为空");
        List<CalculateVo> calculateVoList = Lists.newArrayList();
        for (FormulaInfoDto formulaInfo : formulaInfoList) {
            VariableResultVo variableResultVo = computeFormulaExpression(variableValueMap, formulaInfo, false);
            CalculateVo calculateVo = nebulaToolkitService.copyObjectByWhiteList(calculateDto, CalculateVo.class, HashSet.class, ArrayList.class);
            calculateVo.setFormulaValue(variableResultVo.getValue());
            calculateVo.setFormulaName(variableResultVo.getFormulaName());
            calculateVo.setVariableValueMap(variableValueMap);
            calculateVoList.add(calculateVo);
        }
        return calculateVoList;
    }

    /**
     * 计算公式
     *
     * @param variableValueMap variableValueMap
     * @param formulaInfo      formulaInfo
     * @return 公式结果
     */
    public VariableResultVo computeFormulaExpression(Map<String, BigDecimal> variableValueMap, FormulaInfoDto formulaInfo, boolean test) {
        Validate.notBlank(formulaInfo.getFormula(), "表达式不能为空");
        //条件中的变量
        Set<String> variables = MathUtil.getFormulaReplace(formulaInfo.getFormula());
        String expressValue = formulaInfo.getFormula();
        //替换
        expressValue = replaceExpression(expressValue, variables, variableValueMap);
        BigDecimal computeResult = MathUtil.computeFormula(expressValue, formulaInfo.getFormula(), test);
        Validate.notNull(computeResult, "公式【%s】计算失败，请检查公式是否正确", formulaInfo.getFormula());
        VariableResultVo variableResultVo = new VariableResultVo();
        variableResultVo.setValue(computeResult);
        variableResultVo.setFormulaName(formulaInfo.getFormulaName());
        return variableResultVo;
    }

    /**
     * 计算公式条件
     *
     * @param variableValueMap variableValueMap
     * @param formulaInfo      formulaInfo
     * @return 公式条件结果
     */
    public VariableConditionResultVo computeFormulaCondition(Map<String, BigDecimal> variableValueMap, FormulaInfoDto formulaInfo, boolean test) {
        if (StringUtils.isBlank(formulaInfo.getFormulaCondition())) {
            VariableConditionResultVo variableConditionResultVo = new VariableConditionResultVo();
            variableConditionResultVo.setValue(true);
            variableConditionResultVo.setFormulaConditionName(formulaInfo.getFormulaConditionName());
            return variableConditionResultVo;
        }
        //条件中的变量
        Set<String> variables = MathUtil.getFormulaReplace(formulaInfo.getFormulaCondition());
        //最终表达式
        String expressValue = formulaInfo.getFormulaCondition();

        //替换
        expressValue = replaceExpression(expressValue, variables, variableValueMap);
        //执行计算
        Boolean computeResult = MathUtil.computeCondition(expressValue, formulaInfo.getFormulaCondition(), test);
        if (null == computeResult){
            computeResult = false;
        }
        VariableConditionResultVo variableConditionResultVo = new VariableConditionResultVo();
        variableConditionResultVo.setValue(computeResult);
        variableConditionResultVo.setFormulaConditionName(formulaInfo.getFormulaConditionName());
        return variableConditionResultVo;
    }

    /**
     * 将表达式里变量替换为可计算的数字
     *
     * @param expression       expression
     * @param variables        variables
     * @param variableValueMap variableValueMap
     */
    private String replaceExpression(String expression, Set<String> variables, Map<String, BigDecimal> variableValueMap) {
        for (String variable : variables) {
            BigDecimal value = variableValueMap.get(variable);
            Validate.notNull(value, "计算时检测到变量【%s】值为空", variable);
            //替换实际的值
            expression = expression.replace(variable, value.toString());
        }
        return expression;
    }

    /**
     * 提取变量set
     *
     * @param formulaInfoList formulaInfoList
     * @return 变量set
     */
    public Set<String> getAllVariableNameSet(List<FormulaInfoDto> formulaInfoList) {
        Set<String> conditionAndFormula = new HashSet<>();
        for (FormulaInfoDto formulaInfo : formulaInfoList) {
            if (!StringUtils.isBlank(formulaInfo.getFormulaCondition())) {
                Set<String> formulaConditions = MathUtil.getFormulaReplace(formulaInfo.getFormulaCondition());
                conditionAndFormula.addAll(formulaConditions);
            }
            Set<String> formulas = MathUtil.getFormulaReplace(formulaInfo.getFormula());
            conditionAndFormula.addAll(formulas);
        }
        return conditionAndFormula;
    }

    public FormulaCustomVariableRegister getCustomRegister(List<SourceCustomVariableEntity> customVariableList) {
        if (CollectionUtils.isEmpty(customVariableList)) {
            return null;
        }
        SourceCustomVariableEntity sourceCustomVariableEntity = customVariableList.get(0);
        List<FormulaCustomVariableRegister> filterList = formulaCustomVariableRegisters.stream().filter(e -> e.getVariableCode().equals(sourceCustomVariableEntity.getDataSource())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(filterList)) {
            return null;
        }
        return filterList.get(0);
    }
}
