package com.biz.crm.workflow.client.observer;

import com.alibaba.fastjson.JSON;
import com.biz.crm.workflow.client.config.ApplicationConfig;
import com.biz.crm.workflow.client.listener.ProcessStatusListener;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessMappingDto;
import com.biz.crm.workflow.sdk.dto.ProcessStatusDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.listener.ProcessCompleteListener;
import com.biz.crm.workflow.sdk.service.ProcessBusinessMappingService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessMappingVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 描述：</br>流程状态变更回调
 *
 * @author keller
 * @date 2022/8/25
 */
@Component
@Slf4j
class ProcessStatusEventListener implements ProcessStatusListener {

  @Value("${spring.application.name}")
  private String applicationName;
  @Autowired
  private ProcessBusinessMappingService processBusinessMappingService;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private List<ProcessCompleteListener> processCompleteListeners;

  @Override
  @Transactional
  public void onProcessStatusChanged(ProcessStatusDto dto) {
    log.info("消息通知参数：：{}::{}", JSON.toJSONString(dto),applicationName);
    log.warn("消息通知参数：：{}", JSON.toJSONString(dto));
    // nebula 消息通知 每个终端均会收到消息，隔离引用
    if (dto == null || !dto.getApplicationName().equals(applicationName)) {
      return;
    }
    log.info("消息通知参数processCompleteListeners：：{}", JSON.toJSONString(processCompleteListeners));
    ProcessBusinessMappingDto processBusinessMappingDto = new ProcessBusinessMappingDto();
    processBusinessMappingDto.setProcessNo(dto.getProcessNo());
    processBusinessMappingDto.setProcessStatus(ProcessStatusEnum.COMMIT.getDictCode());
    List<ProcessBusinessMappingVo> processBusinessMappingVoList = this.processBusinessMappingService.findMultiByByConditions(processBusinessMappingDto);
    if (CollectionUtils.isEmpty(processBusinessMappingVoList)){
      processBusinessMappingDto.setProcessStatus(ProcessStatusEnum.START.getDictCode());
      processBusinessMappingVoList = this.processBusinessMappingService.findMultiByByConditions(processBusinessMappingDto);
      if (CollectionUtils.isEmpty(processBusinessMappingVoList)) {
        log.warn("业务编码{}获取流程信息状态无效，可能是多次回调", dto.getBusinessNo());
        return;
      }
    }
    log.info("消息通知参数processBusinessMappingVoList：：{}", JSON.toJSONString(processBusinessMappingVoList));
    try {
      List<String> ids = processBusinessMappingVoList.stream().map(ProcessBusinessMappingVo::getId).collect(Collectors.toList());
      this.processBusinessMappingService.updateProcessStatusById(ids, dto.getProcessStatus());
      if (!CollectionUtils.isEmpty(processCompleteListeners)) {
        List<String> businessCodeList = processBusinessMappingVoList.stream().map(ProcessBusinessMappingVo::getBusinessNo).collect(Collectors.toList());
        dto.setBusinessNoList(businessCodeList);
        processCompleteListeners.stream().filter(item -> item.getBusinessCode().equals(dto.getBusinessCode())).forEach(item -> {
          item.onProcessComplete(dto);
        });
      }
    } catch (Exception e) {
      log.error(e.getMessage(), e);
      throw new RuntimeException(e.getMessage(), e);
    }
  }

  @Override
  public void onProcessInfoChanged(ProcessStatusDto dto) {
    // nebula 消息通知 每个终端均会收到消息，隔离引用
    if (dto == null || !dto.getApplicationName().equals(applicationName)) {
      return;
    }
    if (StringUtils.isBlank(dto.getProcessNo())) {
      log.warn("流程状态更新失败，更新流程编号为空，请检查");
      return;
    }
    ProcessBusinessMappingDto processBusinessMappingDto = new ProcessBusinessMappingDto();
    processBusinessMappingDto.setProcessNo(dto.getProcessNo());
    List<ProcessBusinessMappingVo> processBusinessMappingVoList = this.processBusinessMappingService.findMultiByByConditions(processBusinessMappingDto);
    if (!CollectionUtils.isEmpty(processBusinessMappingVoList)) {
      boolean canEdit = BooleanUtils.isTrue(dto.getCanEdit());
      List<String> ids = processBusinessMappingVoList.stream().map(ProcessBusinessMappingVo::getId).collect(Collectors.toList());
      this.processBusinessMappingService.updateCanEditByIds(ids, canEdit, dto.getTaskId(), dto.getTaskName());
    }
  }
}
