package com.biz.crm.workflow.client.repository;

import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.workflow.client.entity.ProcessBusinessMapping;
import com.biz.crm.workflow.client.mapper.ProcessBusinessMappingMapper;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessMappingDto;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessMappingVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * 业务流程关联表;(bpm_process_business_mapping)表数据库访问层
 *
 * @author : Keller
 * @date : 2022-8-24
 */
@Component
public class ProcessBusinessMappingRepository extends ServiceImpl<ProcessBusinessMappingMapper, ProcessBusinessMapping> {
  @Autowired
  private ProcessBusinessMappingMapper processBusinessMappingMapper;

  /**
   * 分页查询数据
   *
   * @param pageable 分页对象
   * @param dto      实体对象
   */
  public Page<ProcessBusinessMappingVo> findByConditions(Pageable pageable, ProcessBusinessMappingDto dto) {
    if (pageable == null) {
      pageable = PageRequest.of(0, 50);
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    Page<ProcessBusinessMappingVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return processBusinessMappingMapper.findByConditions(page, dto);
  }

  /**
   * 通过条件查询
   *
   * @param dto
   * @return
   */
  public ProcessBusinessMappingVo findSignalByConditions(ProcessBusinessMappingDto dto) {
    if (dto == null) {
      return null;
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    return processBusinessMappingMapper.findSignalByConditions(dto);
  }

  public List<ProcessBusinessMappingVo> findMultiByByConditions(ProcessBusinessMappingDto dto) {
    if (dto == null) {
      return Collections.emptyList();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    return processBusinessMappingMapper.findMultiByByConditions(dto);
  }

  /**
   * 根据id集合获取详情集合
   *
   * @param ids ID集合
   * @return List<ProcessBusinessMapping>
   */
  public List<ProcessBusinessMapping> findByIds(Collection<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return Collections.emptyList();
    }
    String tenantCode = TenantUtils.getTenantCode();
    return this.lambdaQuery()
            .in(ProcessBusinessMapping::getId, ids)
            .eq(ProcessBusinessMapping::getTenantCode, tenantCode)
            .eq(ProcessBusinessMapping::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
            .list();
  }

  /**
   * 根据业务编号获取数据
   *
   * @param businessNo 业务编号
   * @return List<ProcessBusinessMapping>
   */
  public List<ProcessBusinessMapping> findByBusinessNo(String businessNo) {
    if (!StringUtils.hasText(businessNo)) {
      return Collections.emptyList();
    }
    String tenantCode = TenantUtils.getTenantCode();
    return this.lambdaQuery()
            .eq(ProcessBusinessMapping::getBusinessNo, businessNo)
            .eq(ProcessBusinessMapping::getTenantCode, tenantCode)
            .eq(ProcessBusinessMapping::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
            .list();
  }

  /**
   * 根据业务编号获取数据
   *
   * @param businessNo    业务编号
   * @param processStatus 流程状态
   * @return ProcessBusinessMapping
   */
  public ProcessBusinessMapping findByBusinessNoAndProcessStatus(String businessNo, String processStatus) {
    if (!StringUtils.hasText(businessNo)) {
      return null;
    }
    String tenantCode = TenantUtils.getTenantCode();
    return this.lambdaQuery()
            .eq(ProcessBusinessMapping::getBusinessNo, businessNo)
            .eq(ProcessBusinessMapping::getProcessStatus, processStatus)
            .eq(ProcessBusinessMapping::getTenantCode, tenantCode)
            .eq(ProcessBusinessMapping::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
            .one();
  }

  /**
   * 根据业务编号获取数据
   *
   * @param businessNo    业务编号
   * @param processStatus 流程状态
   * @return ProcessBusinessMapping
   */
  public ProcessBusinessMapping findByBusinessNoAndProcessStatus(String businessNo, String processKey, String processStatus) {
    if (!StringUtils.hasText(businessNo)) {
      return null;
    }
    String tenantCode = TenantUtils.getTenantCode();
    return this.lambdaQuery()
            .eq(ProcessBusinessMapping::getBusinessNo, businessNo)
            .eq(ProcessBusinessMapping::getProcessKey, processKey)
            .eq(ProcessBusinessMapping::getProcessStatus, processStatus)
            .eq(ProcessBusinessMapping::getTenantCode, tenantCode)
            .eq(ProcessBusinessMapping::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
            .one();
  }

  /**
   * 根据业务编号获取数据
   *
   * @param businessNo    业务编号
   * @param businessCode  业务编号
   * @param processKey    流程key
   * @param processStatus 流程状态
   * @return ProcessBusinessMapping
   */
  public ProcessBusinessMapping findByBusinessNoAndBusinessCodeAndProcessStatus(String businessNo, String businessCode, String processKey, String processStatus) {
    if (!StringUtils.hasText(businessNo)) {
      return null;
    }
    String tenantCode = TenantUtils.getTenantCode();
    return this.lambdaQuery()
            .eq(ProcessBusinessMapping::getBusinessNo, businessNo)
            .eq(ProcessBusinessMapping::getBusinessCode, businessCode)
            .eq(ProcessBusinessMapping::getProcessKey, processKey)
            .eq(ProcessBusinessMapping::getProcessStatus, processStatus)
            .eq(ProcessBusinessMapping::getTenantCode, tenantCode)
            .eq(ProcessBusinessMapping::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
            .one();
  }

  public void deleteByBusinessNoAndBusinessCodeAndProcessKey(String businessNo, String businessCode, String processKey) {
    if (org.apache.commons.lang3.StringUtils.isAnyBlank(businessNo, businessCode, processKey)) {
      return;
    }
    String tenantCode = TenantUtils.getTenantCode();
    this.lambdaUpdate()
            .eq(ProcessBusinessMapping::getBusinessNo, businessNo)
            .eq(ProcessBusinessMapping::getBusinessCode, businessCode)
            .eq(ProcessBusinessMapping::getProcessKey, processKey)
            .eq(ProcessBusinessMapping::getTenantCode, tenantCode)
            .set(ProcessBusinessMapping::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
            .update();
  }

  public ProcessBusinessMapping findByIdAndTenantCode(String id, String tenantCode) {
    return this.lambdaQuery()
        .eq(ProcessBusinessMapping::getTenantCode,tenantCode)
        .in(ProcessBusinessMapping::getId,id)
        .one();
  }

  public void updateByIdAndTenantCode(ProcessBusinessMapping processBusinessMapping, String tenantCode) {
    LambdaUpdateWrapper<ProcessBusinessMapping>lambdaUpdateWrapper= Wrappers.lambdaUpdate();
    lambdaUpdateWrapper.eq(ProcessBusinessMapping::getTenantCode,tenantCode);
    lambdaUpdateWrapper.in(ProcessBusinessMapping::getId,processBusinessMapping.getId());
    this.baseMapper.update(processBusinessMapping,lambdaUpdateWrapper);
  }
}
