package com.biz.crm.workflow.client.service.internal;

import com.biz.crm.workflow.client.listener.ProcessInfoListener;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessMappingDto;
import com.biz.crm.workflow.sdk.dto.ProcessStatusDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.listener.ProcessCompleteListener;
import com.biz.crm.workflow.sdk.register.ProcessBusinessRegister;
import com.biz.crm.workflow.sdk.service.ProcessBusinessMappingService;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessMappingVo;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.biz.crm.workflow.sdk.vo.ProcessInfoResponse;
import com.biz.crm.workflow.sdk.vo.ProcessInstanceVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Sets;

import java.util.Objects;

import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.stereotype.Service;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.util.CollectionUtils;

import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;
import javax.transaction.Transactional;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 描述：</br>业务流程发起服务实现
 * nebula event 事件通知的方式调用流程引擎发起流程信息记录信息至业务流程管理表中
 *
 * @author keller
 * @date 2022/8/24
 */
@Service
@Slf4j
public class ProcessBusinessServiceImpl implements ProcessBusinessService {
  @Autowired
  private ProcessBusinessMappingService processBusinessMappingService;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Value("${spring.application.name}")
  private String applicationName;
  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private List<ProcessBusinessRegister> processBusinessRegisters;
  @Autowired(required = false)
  private List<ProcessCompleteListener> processCompleteListeners;
  @Autowired
  private ProcessBusinessService processBusinessService;

  @Override
  public ProcessBusinessVo processStart(ProcessBusinessDto dto) {
    /*
     * 1、检查流程客户端注册器是否已经配置
     * 2、检查参数合规
     * 3、检查模块是否开启流程
     * 4、检查业务编号是否有存在审批中的流程
     * 5、发起流程
     * 6、本地存储流程信息
     */
    this.validateProcessStart(dto);
    Collection<String> processBusinessCodes = null;
    if (CollectionUtils.isEmpty(processBusinessRegisters)) {
      processBusinessCodes = Sets.newHashSet();
    } else {
      processBusinessCodes = processBusinessRegisters.stream().map(ProcessBusinessRegister::getBusinessCode).collect(Collectors.toSet());
    }

    ProcessBusinessVo processBusinessVo = null;
    try {
      if (processBusinessCodes.contains(dto.getBusinessCode())) {
        // 创建流程
        processBusinessVo = this.processBusinessService.create(dto);
        dto.setProcessNo(processBusinessVo.getProcessNo());
        // 发起流程
        String processStatus = this.processBusinessService.startProcess(dto);
        processBusinessVo.setProcessStatus(processStatus);
      } else {
        dto.setSkipped(true);
        this.skip(dto);
        processBusinessVo = new ProcessBusinessVo();
        processBusinessVo.setSkipped(true);
        processBusinessVo.setBusinessNo(dto.getBusinessNo());
        processBusinessVo.setBusinessCode(dto.getBusinessCode());
        processBusinessVo.setApplicationName(dto.getApplicationName());
      }
    } catch (Exception e) {
      log.error(e.getMessage(), e);
      throw new RuntimeException(String.format("发起流程错误:%s", e.getMessage()), e);
    }
    return processBusinessVo;
  }

  /**
   * 创建流程
   *
   * @param dto
   * @return
   */
  @Transactional
  @Override
  public ProcessBusinessVo create(ProcessBusinessDto dto) {
    dto.setApplicationName(this.applicationName);
    dto.setTenantCode(TenantUtils.getTenantCode());
    ProcessStatusDto processStatusDto = new ProcessStatusDto();
    ProcessBusinessMappingVo processBusinessMaVo = this.processBusinessMappingService.findProcessByBusinessNoAndBusinessCodeAndProcessKeyAndProcessStatus(dto.getBusinessNo(), dto.getBusinessCode(), dto.getProcessKey(), ProcessStatusEnum.START.getDictCode());
    if (Objects.nonNull(processBusinessMaVo)) {
      processStatusDto.setProcessTitle(dto.getProcessTitle());
      processStatusDto.setBusinessNo(dto.getBusinessNo());
      processStatusDto.setProcessKey(dto.getProcessKey());
      processStatusDto.setProcessNo(processBusinessMaVo.getProcessNo());
      processStatusDto.setProcessStatus(ProcessStatusEnum.START.getDictCode());
      processStatusDto.setApplicationName(applicationName);
      processStatusDto.setBusinessCode(dto.getBusinessCode());
      processStatusDto.setRemark(dto.getRemark());
      ProcessBusinessVo processBusinessVo = this.nebulaToolkitService.copyObjectByWhiteList(processStatusDto, ProcessBusinessVo.class, HashSet.class, ArrayList.class);
      return processBusinessVo;
    }
    ProcessBusinessMappingVo processBusinessMappingVo = this.processBusinessMappingService.findProcessByBusinessNoAndBusinessCodeAndProcessKeyAndProcessStatus(dto.getBusinessNo(), dto.getBusinessCode(), dto.getProcessKey(), ProcessStatusEnum.COMMIT.getDictCode());
    Validate.isTrue(processBusinessMappingVo == null, "业务编号[%s]已经存在待审批流程,请检查！", dto.getBusinessNo());

    processBusinessMappingVo = this.processBusinessMappingService.findProcessByBusinessNoAndBusinessCodeAndProcessKeyAndProcessStatus(dto.getBusinessNo(), dto.getBusinessCode(), dto.getProcessKey(), ProcessStatusEnum.PASS.getDictCode());
    Validate.isTrue(processBusinessMappingVo == null, "业务编号[%s]已经存在审批通过记录,请检查！", dto.getBusinessNo());
    ProcessInfoResponse response = null;
    SerializableBiConsumer<ProcessInfoListener, ProcessBusinessDto> sf = ProcessInfoListener::createProcess;
    response = (ProcessInfoResponse) this.nebulaNetEventClient.directPublish(dto, ProcessInfoListener.class, sf);
    Validate.notNull(response, "创建流程失败，请检查流程图是否正常");
    Validate.notBlank(response.getProcessNo(), "流程[%s]创建失败", response.getProcessKey());
    log.debug("创建流程-业务编号：{},流程编号:{}", response.getBusinessNo(), response.getProcessNo());
    dto.setProcessNo(response.getProcessNo());
    // 记录流程发起信息与业务关联表信息
    ProcessBusinessMappingDto processBusinessMappingDto = new ProcessBusinessMappingDto();
    processBusinessMappingDto.setBusinessNo(response.getBusinessNo());
    processBusinessMappingDto.setProcessKey(response.getProcessKey());
    processBusinessMappingDto.setProcessNo(response.getProcessNo());
    processBusinessMappingDto.setApplicationName(applicationName);
    processBusinessMappingDto.setBusinessCode(dto.getBusinessCode());
    processBusinessMappingDto.setProcessStatus(ProcessStatusEnum.START.getDictCode());
    // 旧数据标识为删除
    this.processBusinessMappingService.deleteByBusinessNoAndBusinessCodeAndProcessKey(dto.getBusinessNo(), dto.getBusinessCode(), dto.getProcessKey());
    processBusinessMappingVo = this.processBusinessMappingService.create(processBusinessMappingDto);
    processStatusDto.setProcessTitle(dto.getProcessTitle());
    processStatusDto.setBusinessNo(dto.getBusinessNo());
    processStatusDto.setProcessKey(dto.getProcessKey());
    processStatusDto.setProcessNo(response.getProcessNo());
    processStatusDto.setProcessStatus(ProcessStatusEnum.START.getDictCode());
    processStatusDto.setApplicationName(applicationName);
    processStatusDto.setBusinessCode(dto.getBusinessCode());
    processStatusDto.setRemark(dto.getRemark());
    ProcessBusinessVo processBusinessVo = this.nebulaToolkitService.copyObjectByWhiteList(processStatusDto, ProcessBusinessVo.class, HashSet.class, ArrayList.class);
    return processBusinessVo;
  }

  /**
   * 发起流程
   *
   * @param dto
   * @return
   */
  @Transactional
  @Override
  public void start(ProcessBusinessDto dto) {
    this.initiateProcess(dto);
  }

  @Override
  @Transactional
  public String startProcess(ProcessBusinessDto dto) {
    return this.initiateProcess(dto);
  }

  /**
   * 跳过工作流
   *
   * @param dto
   */
  private void skip(ProcessBusinessDto dto) {
    if (!CollectionUtils.isEmpty(processCompleteListeners)) {
      processCompleteListeners.stream().filter(item -> item.getBusinessCode().equals(dto.getBusinessCode())).forEach(item -> {
        ProcessStatusDto processStatusDto = new ProcessStatusDto();
        processStatusDto.setBusinessCode(dto.getBusinessCode());
        processStatusDto.setBusinessNo(dto.getBusinessNo());
        processStatusDto.setApplicationName(dto.getApplicationName());
        processStatusDto.setTenantCode(TenantUtils.getTenantCode());
        processStatusDto.setProcessStatus(ProcessStatusEnum.PASS.getDictCode());
        log.debug("跳过流程-业务编号：{},流程编号:{}", dto.getBusinessNo(), dto.getProcessNo());
        item.onProcessComplete(processStatusDto);
      });
    }
  }

  /**
   * 验证启动流程参数
   *
   * @param dto
   */
  private void validateProcessStart(ProcessBusinessDto dto) {
    Validate.notNull(dto, "发起流程参数不能为空，请检查！");
    Validate.notBlank(dto.getBusinessNo(), "发起流程业务编号不能为空，请检查！");
    Validate.notBlank(dto.getProcessTitle(), "流程主题不能为空，请检查！");
    Validate.notBlank(dto.getProcessKey(), "流程key不能为空，请检查！");
  }

  private String initiateProcess(ProcessBusinessDto dto) {
    Validate.notNull(dto, "启动流程参数为空");
    Validate.notBlank(dto.getProcessNo(), "启动流程流程编号为空，请检查");
    ProcessInfoResponse response = null;
    SerializableBiConsumer<ProcessInfoListener, ProcessBusinessDto> sf = ProcessInfoListener::startProcess;
    response = (ProcessInfoResponse) this.nebulaNetEventClient.directPublish(dto, ProcessInfoListener.class, sf);
    Validate.notNull(response, "发起流程失败，请检查工作流服务是否正常启动");
    Validate.notBlank(response.getProcessNo(), "流程[%s]发起流程失败", response.getProcessKey());
    log.debug("发起流程-业务编号：{},流程编号:{}", response.getBusinessNo(), response.getProcessNo());
    ProcessBusinessMappingVo processBusinessMappingVo = this.processBusinessMappingService.findProcessByBusinessNoAndBusinessCodeAndProcessKeyAndProcessStatus(response.getBusinessNo(), dto.getBusinessCode(), dto.getProcessKey(), ProcessStatusEnum.START.getDictCode());
    if (processBusinessMappingVo != null) {
      this.processBusinessMappingService.updateProcessStatusById(processBusinessMappingVo.getId(), response.getProcessStatus());
    }
    return  response.getProcessStatus();
  }
}
