package com.biz.eisp.base.core.dao;

import com.biz.eisp.base.core.page.Page;

import org.hibernate.Session;
import org.hibernate.criterion.Criterion;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * <p>Dao 操作基类 接口<p>
 * 提供对数据库的基本操作接口：save,update,delete,find的CURD操作。
 * <li>提供Hibernate方式的CURD操作</li>
 * <li>提供JDBC方式的CURD操作</li>
 * @author liukai
 *
 */
public interface BaseDao {
	
	/**
	 * 获取Session对象.
	 * <p>
	 * @return
	 */
	public Session getSession();
	
	/**
	 * 保存实体
	 * @param entity Hibernate POJO对象
	 * @return T-泛型实体对象
	 */
	public <T> Serializable save(T entity);

	/**
	 * 保存修改实体
	 * @param entity Hibernate POJO对象
	 */
	public <T> void saveOrUpdate(T entity);

	/**
	 * 批量保存实体
	 * @param entities Hibernate POJO对象列表
	 */
	public <T> void batchSave(List <T> entities);

	/**
	 * 删除实体
	 * @param entity Hibernate POJO对象
	 */
	public <T> void delete(T entity);
	
	/**
	 * 删除实体主键删除
	 * @param clazz 实体Class
	 * @param id 实体主键ID
	 */
	public <T> void deleteEntityById(Class <T> clazz, Serializable id);

	/**
	 * 删除实体集合
	 * @param entities 实体集合
	 */
	public <T> void deleteAllEntity(Collection <T> entities);

	/**
	 * 修改实体
	 * @param entity Hibernate POJO对象
	 */
	public <T> void updateEntity(T entity);
	
	/**
	 * 修改实体
	 * @param sql 语句
	 * @param param 参数 
	 */
	public  void updateBySql(String sql,Object...param);

	
	/**
	 * 根据实体类型和主键获取实体
	 * @param <T>
	 * @param clazz 实体Class
	 * @param id 实体主键ID
	 * @return T-泛型实体对象
	 */
	public <T> T get(Class <T> clazz, Serializable id);

	/**
	 * 通过属性获取实体<br>
	 * 例如：Student实体，存在code属性，则可以使用findUniqueByProperty(Student.class, "code", "0001")
	 * @param clazz 实体Class
	 * @param propertyName 实体属性
	 * @param value 值
	 * @return T-泛型实体对象列表
	 */
	public <T> List<T> findByProperty(Class<T> clazz,
			String propertyName, Object value);
	
	/**
	 * 通过属性获取实体带排序<br>
	 * 例如：Student实体，存在code属性，则可以使用findUniqueByProperty(Student.class, "code", "0001", true)
	 * @param clazz 实体Class
	 * @param propertyName 实体属性
	 * @param value 值
	 * @param isAsc 是否升序   true 升序， false 降序
	 * @param isAscForpropertyName 排序字段
	 * @return T-泛型实体对象列表
	 */
	public <T> List<T> findByPropertyisOrder(Class<T> clazz,
			String propertyName, Object value, boolean isAsc,String isAscForpropertyName);
	
	/**
	 * 根据属性获取实体唯一记录<br>
	 * 例如：Student实体，存在code属性，则可以使用findUniqueByProperty(Student.class, "code", "0001")
	 * @param clazz 实体Class
	 * @param propertyName 实体属性
	 * @param value 值
	 * @return T-泛型实体对象
	 */
	public <T> T findUniqueByProperty(Class<T> clazz,
			String propertyName, Object value);
	
	/**
	 * 加载全部实体
	 * @param clazz 实体Class
	 * @return T-泛型实体对象列表
	 */
	public <T> List<T> loadAll(final Class<T> clazz);

	/**
	 * jdbc查询
	 * @param sql
	 * @param objs
	 * @return map集合
	 */
	public List<Map<String, Object>> findForMapList(String sql, Object... objs) ;
	
	/**
	 * jdbc查询
	 * @param sql
	 * @param objs
	 * @return map
	 */
	public Map<String, Object> findForMap(String sql, Object... objs) ;
	
	/**
	 * QBC排序查询
	 * @param clazz 返回值类型
	 * @param isAsc 是否降序
	 * @param criterions 查询条件
	 * @return 对象集合
	 */
	public <T>List<T> findByCriteria(Class<T> clazz,boolean isAsc,String isAscForpropertyName,Criterion... criterions);
	
	/**
	 * QBC查询
	 * @param clazz
	 * @param criterions
	 * @return 对象集合
	 */
	public <T>List<T> findByCriteria(Class<T> clazz,Criterion... criterions);

    /**
     * QBC分页查询
     *
     * @param clazz
     * @param page
     * @param criterions
     * @param <T>
     * @return
     */
    public <T> List<T> findByCriteria(Class<T> clazz, Page page, List<Criterion> criterions);

    /**
	 * 执行sql语句
	 * @param sql sql语句
	 * @param params 动态查询参数， 此参数sql使用?做为占位符
	 * @return 返回执行条数
	 */
	public Integer executeSql(String sql, Object... params);
	
	/**
	 * jdbc查询结果条数 
	 * @param sql
	 * @param objs
	 * @return 返回查询条数
	 */
	public Long getCountForJdbcParam(String sql,Object... params) ;
	
	/**
	 * 通过hql 查询语句查找对象
	 * @param hql hql语句
	 * @param params 动态查询参数， 此参数hql使用?做为占位符
	 * @return T-泛型实体对象列表
	 */
	public <T> List<T> findByHql(String hql, Object ... params);
	
	
	/**
	 * 通过hql 查询语句查找对象
	 * @param hql hql语句
	 * @param page 页面对象
	 * @param params 动态查询参数， 此参数hql使用?做为占位符
	 * @return 对象集合
	 */
	public <T> List<T> findByHql(String hql, Page page, Object ... params);
	
	
	/**
	 * 通过hql 查询语句查找对象
	 * @param clazz 返回对象类
	 * @param hql hql语句
	 * @param params 动态查询参数， 此参数hql使用?做为占位符
	 * @return 对象
	 */
	public <T> T getUniqueByHql(Class<T> clazz,String hql,  Object ... params);
	
	/**
	 * 通过Sql查询语句查询单个对象
	 * @param clazz 返回对象类
	 * @param sql 查询sql语句
	 * @param params 动态查询参数
	 * @return clazz对象
	 */
	public <T> T getUniqueBySql(Class<T> clazz,String sql, Object ... params);
	
	/**
	 * 通过sql 查询语句查找对象
	 * @param clazz 返回值对象
	 * @param sql sql语句
	 * @param params 动态查询参数， 此参数hql使用?做为占位符
	 * @return T-泛型实体对象列表
	 */
	public <T> List<T> findBySql(Class <T> clazz, String sql, Object ... params);
	

	/**
	 * 通过sql 查询语句查找对象
	 * @param clazz 返回值对象
	 * @param sql sql语句
	 * @param page 页面对象
	 * @param params 动态查询参数， 此参数hql使用?做为占位符
	 * @return 对象集合
	 */
	public <T> List<T> findBySql(Class <T> clazz, String sql, Page page, Object ... params);
	
	/**
	 * 调用存储过程.
	 * <p>
	 * @param proc 存储过程名称
	 * @param list 存储过程参数值列表
	 */
	public int callableStatementByName(String proc, List<String> list);
}
