package com.biz.eisp.base.interfacedao.aop;

import com.biz.eisp.base.common.exception.BusinessException;
import com.biz.eisp.base.common.util.StringUtil;
import com.biz.eisp.base.core.page.Page;
import com.biz.eisp.base.interfacedao.annotation.Arguments;
import com.biz.eisp.base.interfacedao.annotation.ResultType;
import com.biz.eisp.base.interfacedao.annotation.Sql;
import com.biz.eisp.base.interfacedao.common.FreemarkerParseFactory;
import com.biz.eisp.base.interfacedao.common.InterfaceDaoUtil;
import com.biz.eisp.base.interfacedao.interceptor.EmptyInterceptor;
import com.biz.eisp.mdm.authobj.util.AuthObjHandler;
import ognl.Ognl;
import ognl.OgnlException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.ColumnMapRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 动态代理处理器工具
 * @author Sean
 */
public class InterfaceDaoHandler implements InvocationHandler {
	
	private static final Logger logger = LoggerFactory.getLogger(InterfaceDaoHandler.class);
	
	/**
	 * 接口方法定义规则 添加：insert,add,create 添加：update,modify,store 刪除：delete,remove
	 * 检索：以上各单词之外
	 */
	public static final String INFO_METHOD_MODIFY = "insert,add,create,update,modify,store,delete,remove";

	public static final String INFO_METHOD_BATCH = "batch";
	
	private  static final  String PAGE="page";
	
	private  static final  String SQL="sql";
	
	
	//springJdbc数据库操作
	@Autowired
	private JdbcTemplate jdbcTemplate;

	//具名参数数据库操作
	@Autowired
	private NamedParameterJdbcTemplate namedParameterJdbcTemplate;

	//map的关键字类型 三个值
	private String keyType = "origin";
	
	//格式化sql
	private boolean formatSql = false;
	//输出sql
	private boolean showSql = false;
	//数据库类型
	private String dbType;
	
	// interfaceDao拦截器
	private EmptyInterceptor emptyInterceptor;

	@Override
	@SuppressWarnings("all")
	public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
		// 返回结果
		Object returnObj = null;
		// SQL模板
		String templateSql = null;
		// SQL模板参数
		Map<String, Object> sqlParamsMap = new HashMap<String, Object>();
		
		//装载SQL模板参数
		templateSql=this.installDaoMetaData( method, sqlParamsMap, args);
		//解析SQL模板
		String executeSql = parseSqlTemplate(method, templateSql, sqlParamsMap);
		//组装占位符参数
		Map<String, Object> sqlMap = installPlaceholderSqlParam(executeSql, sqlParamsMap);
		//获取SQL执行返回值
		try {
			Page page=(Page) sqlParamsMap.get(PAGE);
			returnObj = getReturnResultType(dbType, page, method, executeSql, sqlMap);
		} catch (Exception e) {
			returnObj = null;
			if(e instanceof EmptyResultDataAccessException){
				//数据查询为空，不抛出Spring异常
//				logger.info("SQL:" + executeSql+"执行结果为空");
			}else{
				e.printStackTrace();
				throw e;
			}
		}
		if (showSql) {
//			logger.info("SQL:\n\n" + executeSql);
		}
		return returnObj;
	}

	/**
	 * 装载SQL模板参数
	 * @param pageSetting
	 * @param method
	 * @param sqlParamsMap
	 * @param args 方法参数
	 * @return
	 * @throws Exception
	 */
	@SuppressWarnings("all")
	private String installDaoMetaData( Method method,
			Map<String, Object> sqlParamsMap, Object[] args) throws Exception {
		
		String templateSql = null;
		if(emptyInterceptor!=null && args!= null && args.length==1){
			String methodName = method.getName();
			Object obj = args[0];
			Field[] fields = obj.getClass().getDeclaredFields();
			if(methodName.startsWith("insert")){
				if(emptyInterceptor!=null){
					emptyInterceptor.onInsert(fields, obj);
				}
			}
			if(methodName.startsWith("update")){
				if(emptyInterceptor!=null){
					emptyInterceptor.onUpdate(fields, obj);
				}
			}
		}
		//判断是否存在Arguments类型注解
		boolean hasArguments = method.isAnnotationPresent(Arguments.class);
		if(hasArguments){
			Arguments arguments = method.getAnnotation(Arguments.class);
			if (arguments.value().length != args.length) {
				// 校验机制-如果注释标签参数数目大于方法的参数，则抛出异常
				throw new BusinessException("注释标签@Arguments参数数目，与方法参数数目不相等~");
			}
			
			int argsNum = 0;
			for (String v : arguments.value()) {
				//判断参数page是否是Page对象实例
				if (v.equalsIgnoreCase(PAGE)) {
					Class[] params = method.getParameterTypes();
					 for (int j = 0; j < params.length; j++) {
						 String paramName=params[j].getName();
						 if(paramName.indexOf(PAGE)!=-1){
						    Class clazz = Class.forName(paramName);
			                Class supClazz=clazz.getSuperclass();
			                if (!clazz.isAssignableFrom(Page.class)&&!supClazz.isAssignableFrom(Page.class)) {
								throw new BusinessException("参数page必须是com.biz.eisp.base.core.page.Page对象实例");
			                }
						 }
					}
	             }
				 
				//判断是否传有sql参数 有就直接
				if(v.equalsIgnoreCase(SQL)){
					templateSql=(String) args[argsNum];
				}
				
				sqlParamsMap.put(v, args[argsNum]);
				argsNum++;
			}
			 
		}else{
			if (args != null && args.length >= 0) {
				String[] params=InterfaceDaoUtil.getMethodParameterNamesByAnnotation(method);
				if(params==null || params.length==0){
					throw new BusinessException("方法参数数目>=2，必须使用：方法标签@Arguments 或  参数标签@param");
				}
				if (params.length != args.length) {
					throw new BusinessException("方法参数数目与配置参数数目不相等");
				}
				
				int args_num = 0;
				for (String v : params) {
					if(v==null){
						throw new BusinessException("Dao接口定义，所有参数必须使用@param标签~");
					}
					
					//判断是否传有sql参数 有就直接
					if(StringUtil.equals(v, SQL)){
						templateSql=(String) args[args_num];
					}
					
					//判断参数page是否是Page对象实例
					Class[] param = method.getParameterTypes();
					 for (int j = 0; j < param.length; j++) {
						 if (v.equalsIgnoreCase(PAGE)) {
							 Class obj = Class.forName(param[j].getName());
			                Class supClazz=obj.getSuperclass();
			                if (!obj.isAssignableFrom(Page.class)&&!supClazz.isAssignableFrom(Page.class)) {
								throw new BusinessException("参数page必须是com.biz.eisp.base.core.page.Page对象实例");
			                }
						 }
		             }
					 
					sqlParamsMap.put(v, args[args_num]);
					args_num++;
				}
			}
		}
		
		//获取Sql 找到Sql标签 则忽略Sql文件
		if(method.isAnnotationPresent(Sql.class)){
			Sql sql=method.getAnnotation(Sql.class);
			if(StringUtil.isNotEmpty(sql.value())){
				templateSql=sql.value();
			}
		}
		
		return templateSql;
	}
	
	/**
	 * 解析SQL模板
	 * @param method
	 * @param templateSql
	 * @param sqlParamsMap
	 * @return 可执行SQL
	 */
	@SuppressWarnings("all")
	private String parseSqlTemplate(Method method, String templateSql, Map<String, Object> sqlParamsMap) {
		// step.1.根据命名规范[接口名_方法名.sql]，获取SQL模板文件的路径
		String executeSql = null;
		
		// .获取SQL模板内容
		if (StringUtil.isNotEmpty(templateSql)) {
			executeSql = FreemarkerParseFactory.parseTemplateContent(templateSql, sqlParamsMap);
		} else {
			// 通过模板引擎给SQL模板装载参数,解析生成可执行SQL
			String sqlTempletPath = method.getDeclaringClass().getName().
					replace(".", "/").replace("/dao/", "/sql/") + "_" + method.getName() + ".sql";
			if (!FreemarkerParseFactory.isExistTemplate(sqlTempletPath)) {
				sqlTempletPath = method.getDeclaringClass().getName().replace(".", "/") + "_" + method.getName() + ".sql";
			}
//			logger.debug("InterfaceDao-SQL-Path:" + sqlTempletPath);
			executeSql = FreemarkerParseFactory.parseTemplate(sqlTempletPath, sqlParamsMap);
		}
		//权限对象判定
		executeSql=AuthObjHandler.executeAuthHandle(executeSql);
		return executeSql;
	}
	
	/**
	 * 组装占位符参数
	 * @param executeSql
	 * @param sqlParamsMap
	 * @return
	 * @throws OgnlException
	 */
	@SuppressWarnings("all")
	private Map<String, Object> installPlaceholderSqlParam(String executeSql, 
			Map sqlParamsMap) throws OgnlException{
		
		Map<String, Object> map = new HashMap<String, Object>();
		String regex = ":[ tnx0Bfr]*[0-9a-z.A-Z_]+";
		Pattern pattern=Pattern.compile(regex);
		Matcher marcher=pattern.matcher(executeSql);
		while(marcher.find()){
			String ognlKey=marcher.group().replace(":", "").trim();
			//结合解析表达，方便调用GetValue方法。
			map.put(ognlKey,  Ognl.getValue(ognlKey, sqlParamsMap));
		}
		return map;
	}
	
	/**
	 * 处理返回结果
	 * @param dbType
	 * @param pageSetting
	 * @param method
	 * @param executeSql
	 * @param paramMap
	 * @return
	 */
	@SuppressWarnings("all")
	private Object getReturnResultType(String dbType, Page page,
			Method method, String executeSql, Map<String, Object> paramMap) throws Exception {
		String methodName=method.getName();
		
		//判断是非查询 
		if(checkIsNotSearch(methodName,this.INFO_METHOD_MODIFY)
				||checkIsNotSearch(executeSql,this.INFO_METHOD_MODIFY)){
			if (paramMap != null) {
				return namedParameterJdbcTemplate.update(executeSql, paramMap);
			} else {
				return jdbcTemplate.update(executeSql);
			}
			//判断是否是批Sql处理
		}else if (checkIsNotSearch(methodName,this.INFO_METHOD_BATCH)) {
			return batchUpdate(executeSql);
			//查询操作
		}else{
			Class<?> returnType=method.getReturnType();
			
			//Page对象不能直接返回
			if (returnType.isAssignableFrom(Page.class)&&!returnType.isAssignableFrom(Map.class)) {
				throw new BusinessException("Page对象不能直接返回，分页请使用list返回");
			}
			
			//确定指定的类对象是否代表一个原始类型
			if (returnType.isPrimitive()) {
				Number number = namedParameterJdbcTemplate.queryForObject(executeSql, paramMap, BigDecimal.class);
				//返回规范名称的底层阶级的java语言规范定义。
				if ("int".equals(returnType.getCanonicalName())) {
					return number.intValue();
				} else if ("long".equals(returnType.getCanonicalName())) {
					return number.longValue();
				} else if ("double".equals(returnType.getCanonicalName())) {
					return number.doubleValue();
				}
				//如果返回为List并判断是否添加分页
			}else if (returnType.isAssignableFrom(List.class)){
				if(page!=null){
					int pages = page.getPageNo();
					int rows = page.getInt(page.ROWS);
					if (pages >= 0 && rows != 0) {
						if (paramMap != null) {
							page.put(Page.TOTAL, namedParameterJdbcTemplate.
									queryForObject(getCountSql(executeSql), paramMap, String.class));
						} else {
							page.put(Page.TOTAL,jdbcTemplate.queryForObject(getCountSql(executeSql), String.class));
						}
						executeSql = InterfaceDaoUtil.createPageSql(dbType, executeSql, pages, rows);
					}
				}
				//获取方法返回类型
				RowMapper<?> resultType = getListRealType(method);
				List list;
				if (paramMap != null) {
					list = namedParameterJdbcTemplate.query(executeSql, paramMap, resultType);
				} else {
					list = jdbcTemplate.query(executeSql, resultType);
				}
				return list;
				// Map类型
			} else if (returnType.isAssignableFrom(Map.class)) {
				if (paramMap != null) {
					return namedParameterJdbcTemplate.queryForObject(executeSql, paramMap, new ColumnMapRowMapper());
				} else {
					return jdbcTemplate.queryForObject(executeSql,new ColumnMapRowMapper());
				}
				//String类型
			} else if (returnType.isAssignableFrom(String.class)) {
				if (paramMap != null) {
					return namedParameterJdbcTemplate.queryForObject(executeSql, paramMap, String.class);
				} else {
					return jdbcTemplate.queryForObject(executeSql, String.class);
				}
				//判断是基本数据类型
			} else if (InterfaceDaoUtil.isWrapClass(returnType)) {
				if (paramMap != null) {
					return namedParameterJdbcTemplate.queryForObject(executeSql, paramMap, returnType);
				} else {
					return jdbcTemplate.queryForObject(executeSql, returnType);
				}
			} else {
				// 对象类型
				RowMapper<?> rm = BeanPropertyRowMapper.newInstance(returnType);
				if (paramMap != null) {
					return namedParameterJdbcTemplate.queryForObject(executeSql, paramMap, rm);
				} else {
					return jdbcTemplate.queryForObject(executeSql, rm);
				}
			}
		}
		return null;
	}
	
	/**
	 * 判断方法操作类型
	 * @param head 操作开头
	 * @param target 目标类型
	 * @return
	 */
	private boolean checkIsNotSearch(String head,String target) {
		String keys[] =target.split(",");
		for (String s : keys) {
			if (head.startsWith(s))
				return true;
		}
		return false;
	}
	
	/**
	 * 装载批处理执行结果
	 * @param result
	 * @param index
	 * @param arr
	 */
	private void addResultArray(int[] result, int index, int[] arr) {
		int length = arr.length;
		for (int i = 0; i < length; i++) {
			result[index - length + i] = arr[i];
		}
	}
	
	/**
	 * 多条Sql批量处理
	 * @param executeSql
	 * @return
	 */
	private int[] batchUpdate(String executeSql) {
		String[] sqls = executeSql.split(";");
		if (sqls.length < 100) {
			return jdbcTemplate.batchUpdate(sqls);
		}
		int[] result = new int[sqls.length];
		List<String> sqlList = new ArrayList<String>();
		for (int i = 0; i < sqls.length; i++) {
			sqlList.add(sqls[i]);
			if (i % 100 == 0) {
				addResultArray(result, i + 1, jdbcTemplate.batchUpdate(sqlList.toArray(new String[0])));
				sqlList.clear();
			}
		}
		addResultArray(result, sqls.length, jdbcTemplate.batchUpdate(sqlList.toArray(new String[0])));
		return result;
	}
	
	/**
	 * 获取总数sql 
	 * @param sql
	 * @return
	 */
	private String getCountSql(String sql) {
		return "select count(1) from (" + sql + ") tmp_count";
	}
	
	/**
	 * 获取方法返回类型
	 * @param method
	 * @return
	 */
	private RowMapper<?> getListRealType(Method method) {
		ResultType resultType = method.getAnnotation(ResultType.class);
		if (resultType != null) {
			if (resultType.value().equals(Map.class)) {
				return new ColumnMapRowMapper();
			}
			return BeanPropertyRowMapper.newInstance(resultType.value());
		}
		String genericReturnType = method.getGenericReturnType().toString();
		String realType = genericReturnType.replace("java.util.List", "").replace("<", "").replace(">", "");
		if (realType.contains("java.util.Map")) {
			return new ColumnMapRowMapper();
		} else if (realType.length() > 0) {
			try {
				return BeanPropertyRowMapper.newInstance(Class.forName(realType));
			} catch (ClassNotFoundException e) {
				logger.error(e.getMessage(), e.fillInStackTrace());
				throw new RuntimeException("获取返回类型错误 ,class:" + realType);
			}
		}
		return new ColumnMapRowMapper();
	}
	

	public JdbcTemplate getJdbcTemplate() {
		return jdbcTemplate;
	}

	public void setJdbcTemplate(JdbcTemplate jdbcTemplate) {
		this.jdbcTemplate = jdbcTemplate;
	}

	public NamedParameterJdbcTemplate getNamedParameterJdbcTemplate() {
		return namedParameterJdbcTemplate;
	}

	public void setNamedParameterJdbcTemplate(NamedParameterJdbcTemplate namedParameterJdbcTemplate) {
		this.namedParameterJdbcTemplate = namedParameterJdbcTemplate;
	}

	public String getKeyType() {
		return keyType;
	}

	public void setKeyType(String keyType) {
		this.keyType = keyType;
	}

	public boolean isFormatSql() {
		return formatSql;
	}

	public void setFormatSql(boolean formatSql) {
		this.formatSql = formatSql;
	}

	public boolean isShowSql() {
		return showSql;
	}

	public void setShowSql(boolean showSql) {
		this.showSql = showSql;
	}

	public String getDbType() {
		return dbType;
	}

	public void setDbType(String dbType) {
		this.dbType = dbType;
	}

	public EmptyInterceptor getEmptyInterceptor() {
		return emptyInterceptor;
	}

	public void setEmptyInterceptor(EmptyInterceptor emptyInterceptor) {
		this.emptyInterceptor = emptyInterceptor;
	}
	
	

}
