package com.biz.eisp.mdm.role.service.impl;

import com.biz.eisp.base.common.constant.Globals;
import com.biz.eisp.base.common.exception.BusinessException;
import com.biz.eisp.base.common.jsonmodel.AjaxJson;
import com.biz.eisp.base.common.jsonmodel.ValidForm;
import com.biz.eisp.base.common.util.CollectionUtil;
import com.biz.eisp.base.common.util.ResourceConfigUtils;
import com.biz.eisp.base.common.util.StringUtil;
import com.biz.eisp.base.core.page.Page;
import com.biz.eisp.base.core.service.impl.BaseServiceImpl;
import com.biz.eisp.mdm.dict.util.DictUtil;
import com.biz.eisp.mdm.dict.vo.TmDictDataVo;
import com.biz.eisp.mdm.role.dao.TmRoleDao;
import com.biz.eisp.mdm.role.entity.TmRoleEntity;
import com.biz.eisp.mdm.role.service.TmRoleExtendService;
import com.biz.eisp.mdm.role.service.TmRoleService;
import com.biz.eisp.mdm.role.transform.TmRoleEntityToTmRoleVo;
import com.biz.eisp.mdm.role.transform.TmRoleVoToTmRoleEntity;
import com.biz.eisp.mdm.role.vo.TmRoleInfoExportVo;
import com.biz.eisp.mdm.role.vo.TmRoleVo;
import com.biz.eisp.mdm.web.pojo.Client;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;

/** 角色接口实现.
 * <p>角色相关功能接口的实现类
 * @author grover
 * @version v1.0
 */
@Service("roleService")
@Transactional
public class TmRoleServiceImpl extends BaseServiceImpl implements TmRoleService{
	/**
	 * 注入dao实现
	 */
	@Autowired
	private TmRoleDao tmRoleDao;
	/**
	 * 注入角色扩展
	 */
	@Autowired(required=false)
	private TmRoleExtendService tmRoleExtendService;
	
	/*校验编码代号*/
	private static final int VALIDATE_ROLECODE = 0;
	
	/*校验名称代号*/
	private static final int VALIDATE_ROLENAME = 1;
	
	@Override
	public List<TmRoleEntity> getRoleListByCurrentPosId(TmRoleVo tmRole,Page page)
			throws BusinessException {
		return tmRoleDao.getRoleListByCurrentPosId(tmRole, page);
	}
	
	@Override
	public List<TmRoleEntity> findRoleList(TmRoleVo tmRole,Page page)
			throws BusinessException {
		return tmRoleDao.findRoleList(tmRole, page);
	}
	
	/**
	 * 分页查询实现
	 * @param page 分页对象
	 * @param tmRole 模糊查询条件
	 * @return 返回查询结果的list集合
	 */
	@Override
	public List<TmRoleEntity> findPage(Page page, TmRoleEntity tmRole) {
		StringBuffer hql = new StringBuffer("from TmRoleEntity t");
		List<TmRoleEntity> list = null;
		if (tmRole == null) {
			
			// 如果没有查询条件，则查询全部
			list = this.findByHql(hql.toString(), page);
		} else {

			// 条件查询
			hql.append(" where t.roleCode like ? and t.roleName like ?");
			String code = tmRole.getRoleCode();
			String name = tmRole.getRoleName();
			Object[] obj = new Object[2];
			if (code != null && !code.equals("undefind")) {
				obj[0] = "%" + tmRole.getRoleCode().trim() + "%";
			} else {
				obj[0] = "%%";
			}
			if (name != null && !name.equals("undefind")) {
				obj[1] = "%" + tmRole.getRoleName().trim() + "%";
			} else {
				obj[1] = "%%";
			}
			list = this.findByHql(hql.toString(), page, obj);
		}
		return list;
	}

	/**
	 * 新增权限角色
	 * @return 返回增加成功的boolean值
	 */
	@Override
	public void createTmRole(TmRoleVo tmRoleVo, Page page) {
		TmRoleEntity tmRoleEntity = null;
		//保存前扩展 此时entity还未通过vo转换
		if(StringUtil.isNotEmpty(tmRoleExtendService)){
			//验证
			try {
				tmRoleExtendService.validate(tmRoleVo, page);
			} catch (Exception e) {
				throw new BusinessException("验证数据失败:"+e.getMessage());
			}
			//扩展
			tmRoleEntity = tmRoleExtendService.saveBefore(tmRoleVo, page);
		}
		
		//如果entity为null 则通过vo转换
		if (StringUtil.isEmpty(tmRoleEntity)) {
			tmRoleEntity = new TmRoleVoToTmRoleEntity(this).apply(tmRoleVo);
		}
		//保存前扩展 此时entity已经通过vo转换
		if(StringUtil.isNotEmpty(tmRoleExtendService)){
			tmRoleEntity = tmRoleExtendService.saveBefore(tmRoleEntity, tmRoleVo, page);
		}
		//保存
		saveOrUpdate(tmRoleEntity);
		//保存后扩展
		if(StringUtil.isNotEmpty(tmRoleExtendService)){
			tmRoleEntity = tmRoleExtendService.savePost(tmRoleEntity, tmRoleVo, page);
		}
	}

	public AjaxJson startOrStopRole(TmRoleVo tmRoleVo){
		AjaxJson j = new AjaxJson();
		TmRoleEntity roleEntity = get(TmRoleEntity.class, tmRoleVo.getId());
		if(StringUtil.isNotEmpty(roleEntity)){
			if(roleEntity.getEnableStatus().equals(tmRoleVo.getEnableStatus())){
				j.setSuccess(false);
				String fg = Globals.ZERO == tmRoleVo.getEnableStatus()?"启用":"停用";
				
				j.setMsg("该数据状态已经为"+fg+"，无需再次操作");
			}else{
				roleEntity.setEnableStatus(tmRoleVo.getEnableStatus());
				updateEntity(roleEntity);
			}
		}else{
			throw new BusinessException("未找到该纪录");
		}
		return j;
	}
	
	/**
	 * 删除权限角色
	 * @param id 需要删除的权限角色的ID
	 * @return 返回删除成功的boolean值
	 */
	@Override
	public boolean deleteTmRole(String id) {
		TmRoleEntity roleEntity = get(TmRoleEntity.class, id);
		if(StringUtil.isNotEmpty(roleEntity)){
			if(Globals.Admin_Name.equalsIgnoreCase(roleEntity.getRoleName())){
				return false;
			}else{
				//删除关联数据
				this.deleteRelationData(id);

				super.deleteEntityById(TmRoleEntity.class, id);
			}
		}else{
			return false;
		}
		
		return true;
	}

	/**
	 * 删除角色关联的数据
	 */
	private void deleteRelationData(String roleId) {
		//删除角色与菜单权限对象的关系, 注意 必须在执行了此方法后，才能调用deleteRelationFunction， 注意先后顺序
		this.deleteRelationFunctionAuth(roleId);

		//删除角色与菜单的关系数据
		this.deleteRelationFunction(roleId);

		//删除角色与职位的相关
		this.deleteRelationPosition(roleId);
	}

	/**
	 * 删除角色与职位的相关
	 * @param roleId
	 */
	private void deleteRelationPosition(String roleId) {
		String sql = "delete from tm_r_position_role where role_id = ?";
		this.executeSql(sql, roleId);
	}

	/**
	 * 删除角色与菜单权限对象的关系
	 * @param roleId
	 */
	private void deleteRelationFunctionAuth(String roleId) {
		String sql = "DELETE tm_r_role_func_authobj WHERE role_function_id IN (SELECT ID FROM tm_r_role_function WHERE role_id = ? )";
		this.executeSql(sql, roleId);
	}

	/**
	 * 删除角色与菜单的关系数据
	 * @param roleId
	 */
	private void deleteRelationFunction(String roleId) {
		String sql = "delete from tm_r_role_function where role_id = ?";
		this.executeSql(sql, roleId);
	}

	/**
	 * 修改权限角色信息
	 * @param tmRole 修改之后的权限角色信息
	 * @return 返回修改成功的boolean值
	 */
	@Override
	public Map<String, String> updateTmRole(TmRoleEntity tmRole) {
		Map<String, String> map = new HashMap<String, String>();
		// 校验角色编码
		List<TmRoleEntity> li1 = this.validate(VALIDATE_ROLECODE, tmRole.getRoleCode());
		if (li1.size() == 0 || li1.get(0).getId().equals(tmRole.getId())) {
			// 校验角色名称
			li1 = this.validate(VALIDATE_ROLENAME, tmRole.getRoleName());
			if (li1.size() == 0 || li1.get(0).getId().equals(tmRole.getId())) {
				// 查询出对应权限角色信息
				TmRoleEntity role = super.get(TmRoleEntity.class, tmRole.getId());
				// 将修改过的字段赋值给查询出的对象
				role.setRoleCode(tmRole.getRoleCode());
				role.setRoleName(tmRole.getRoleName());
				// 添加修改时间
				role.setUpdateDate(new Date());
				// 添加修改人
				Client client = ResourceConfigUtils.getClient();
				String name = client.getUser().getUserName();
				role.setUpdateName(name);
				super.updateEntity(role);
				map.put("success", "true");
			} else {
				map.put("success", "false");
				map.put("msg", "角色名称重复");
			}
		} else {
			map.put("success", "false");
			map.put("msg", "角色编码重复");
		}
		return map;

	}

	/**
	 * 校验角色编码或者名称是否重复
	 * @param flag 区分校验编码与校验名称
	 * @param str 需要校验的值
	 * @return 校验结果
	 */
	public List<TmRoleEntity> validate(int flag, String str) {
		String hql = null;
		Object[] obj = { str };
		if (flag == VALIDATE_ROLENAME) {
			hql = "from TmRoleEntity t where t.roleName = ?";
		} else if (flag == VALIDATE_ROLECODE) {
			hql = "from TmRoleEntity t where t.roleCode = ?";
		}
		return super.findByHql(hql, obj);
	}

	@Override
	public ValidForm validateTmRoleForm(TmRoleEntity tmRole) {
		ValidForm validForm = new ValidForm();
		String hql = null;
		if(StringUtil.isEmpty(tmRole.getId())){
			//新增验证
			if (StringUtil.isNotEmpty(tmRole.getRoleCode())) {
				hql = "from TmRoleEntity t where t.roleCode = ?";
				if(!CollectionUtils.isEmpty(super.findByHql(hql, tmRole.getRoleCode()))){
					validForm.setInfo("角色编码已存在");
					validForm.setStatus("n");
				}
			}
			if (StringUtil.isNotEmpty(tmRole.getRoleName())) {
				hql = "from TmRoleEntity t where t.roleName = ?";
				if(!CollectionUtils.isEmpty(super.findByHql(hql, tmRole.getRoleName()))){
					validForm.setInfo("角色名称已存在");
					validForm.setStatus("n");
				}
			}
		}else{
			//修改验证
			if (StringUtil.isNotEmpty(tmRole.getRoleCode())) {
				hql = "from TmRoleEntity t where t.roleCode = ? and id <> ?";
				if(!CollectionUtils.isEmpty(super.findByHql(hql, tmRole.getRoleCode(), tmRole.getId()))){
					validForm.setInfo("角色编码已存在");
					validForm.setStatus("n");
				}
			}
			if (StringUtil.isNotEmpty(tmRole.getRoleName())) {
				hql = "from TmRoleEntity t where t.roleName = ? and id <> ?";
				if(!CollectionUtils.isEmpty(super.findByHql(hql, tmRole.getRoleName(), tmRole.getId()))){
					validForm.setInfo("角色名称已存在");
					validForm.setStatus("n");
				}
			}
		}
		
		return validForm;
	}

	@Override
	public List<TmRoleVo> findTmRoleByPostionId(String positionId) {
		String sql = "select t1.* from tm_role t1 where t1.id in "
				+ "(select t2.role_id from tm_r_position_role t2 where t2.position_id = ?)";
		List<TmRoleEntity> roles = findBySql(TmRoleEntity.class, sql, positionId);
		List<TmRoleVo> result = new ArrayList<>();
		for(TmRoleEntity e : roles){
			TmRoleVo v = new TmRoleEntityToTmRoleVo().apply(e);
			result.add(v);
		}
		return result;
	}

	@Override
	public List<TmRoleVo> findTmRoleByUserId(String userId) {
		String sql = "select t1.* from tm_role t1 where t1.id in "
				+ "(select t2.role_id from tm_r_position_role t2 where t2.position_id "
				+ " in (select position_id from tm_r_user_position where user_id=?"
				+ " and sysdate between  start_date and end_date))";
		List<TmRoleEntity> roles = findBySql(TmRoleEntity.class, sql, userId);
		List<TmRoleVo> result = new ArrayList<>();
		for(TmRoleEntity e : roles){
			TmRoleVo v = new TmRoleEntityToTmRoleVo().apply(e);
			result.add(v);
		}
		return result;
	}

	 /**检测角色通用权限*/
	@Override
    public boolean checkRoleAu(String userId){
		boolean b=false;
    	if (StringUtils.isNotBlank(userId)) {
    		List<TmRoleVo> roleusers=findTmRoleByUserId(userId);
    		HashSet<String> set=new HashSet<String>();
    		if (roleusers!=null) {
    			for (TmRoleVo ru : roleusers) {
    				if (StringUtils.isBlank(ru.getRoleCode())) {
    					set.add(ru.getRoleCode());
    				}
    			}
    		}
    		List<TmDictDataVo> datas=DictUtil.getDictList("role_au");
    		if (CollectionUtil.listNotEmptyNotSizeZero(datas)) {
    			for (TmDictDataVo dt : datas) {
    				if (set.contains(dt.getDictCode())) {
    					b=true;
    					break;
    				}
    			}
    		}
        	
		}
		return b;
    }

    @Override
	public List <TmRoleVo> findNoSelfRoleList(TmRoleVo tmRoleVo, String selfRoleId, Page page) {
		List <TmRoleVo> voList = tmRoleDao.findNoSelfRoleList(tmRoleVo, selfRoleId, page);
		return voList;
	}

	@Override
	public List <TmRoleInfoExportVo> findExportRoleInfoList(TmRoleInfoExportVo tmRoleInfoExportVo) {
		List <TmRoleInfoExportVo> voList = tmRoleDao.findExportRoleInfoList(tmRoleInfoExportVo);
		return voList;
	}
}
