package com.biz.eisp.mdm.terminal.service;

import com.biz.eisp.base.common.exception.BusinessException;
import com.biz.eisp.base.common.util.CollectionUtil;
import com.biz.eisp.base.importer.ImpEventHandler;
import com.biz.eisp.base.importer.validator.ValidateException;
import com.biz.eisp.base.utils.ApplicationContextUtils;
import com.biz.eisp.generatednum.num.util.TbNumRuleProvider;
import com.biz.eisp.mdm.customer.entity.TmCustomerEntity;
import com.biz.eisp.mdm.terminal.entity.TmRTermCustPosBGEntity;
import com.biz.eisp.mdm.terminal.entity.TmTerminalEntity;
import com.biz.eisp.mdm.terminal.vo.TmTermCustRelImportVo;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * Created by 76315 on 2017/10/27.
 * 终端-客户对应关系导入处理器
 */
public class TmTermCustRelHandler extends ImpEventHandler<TmTermCustRelImportVo> {

    /**
     * 要导入的关系数据
     */
    private List<TmRTermCustPosBGEntity> importDataList = new ArrayList<TmRTermCustPosBGEntity>();
    private TmTermCustPostService tmTermCustPostService;
    private TmTerminalBackExtendService tmTerminalBackExtendService;
    private TbNumRuleProvider tbNumRuleProvider;
    private int succNum = 0;
    private int errNum = 0;
    private StringBuilder errors = new StringBuilder();
    private List<String> saveIdList;

    public TmTermCustRelHandler() {
        tmTermCustPostService = ApplicationContextUtils.getContext().getBean(TmTermCustPostService.class);
        tbNumRuleProvider = ApplicationContextUtils.getContext().getBean(TbNumRuleProvider.class);

        String[] beanNamesForBackExtendService = ApplicationContextUtils.getContext().getBeanNamesForType(TmTerminalBackExtendService.class);
        if (beanNamesForBackExtendService.length > 0){
            tmTerminalBackExtendService = ApplicationContextUtils.getContext().getBean(beanNamesForBackExtendService[0],TmTerminalBackExtendService.class);
        }

        saveIdList = Collections.synchronizedList(new ArrayList<>());
    }

    @Override
    public void endRow(Map<String, Object> row, TmTermCustRelImportVo bean) throws ValidateException {
        String terminalCode = bean.getTerminalCode();
        String customerCode = bean.getCustomerCode();

        TmTerminalEntity terminalEntity = tmTermCustPostService
                .findUniqueByProperty(TmTerminalEntity.class, "terminalCode", terminalCode);
        TmCustomerEntity customerEntity = tmTermCustPostService
                .findUniqueByProperty(TmCustomerEntity.class, "customerCode", customerCode);

        if(terminalEntity == null) {
            this.addErrorMsg("终端编码：" + terminalCode + "不存在");
        }

        if(customerEntity == null) {
            this.addErrorMsg("客户编码：" + customerCode + "不存在");
        }

        if(terminalEntity != null && customerEntity != null) {
            if(isExistTmRTermCustPosBGEntity(terminalEntity.getId(), customerEntity.getId())) {
                this.addErrorMsg("终端：" + terminalCode + "客户" + "关系已存在");
            } else {
                //把数据添加到保存数据列表中
                TmRTermCustPosBGEntity entity = new TmRTermCustPosBGEntity();
                entity.setTmCustomer(customerEntity);
                entity.setTmTerminal(terminalEntity);
                tbNumRuleProvider.getMaxNum("terminal_business_group");
                importDataList.add(entity);

                //设置导入成功信息
                succNum++;
                this.setSuccNum(succNum);
            }
        }

    }

    /**
     * 是否存在终端也客户的对应关系数据
     * @param terminalId
     * @param customerId
     * @return
     */
    private boolean isExistTmRTermCustPosBGEntity(String terminalId, String customerId) {
        String hql = "from TmRTermCustPosBGEntity where tmTerminal.id = ? and tmCustomer.id = ?";
        List<TmRTermCustPosBGEntity> tmRTermCustPosBGEntities = tmTermCustPostService.findByHql(hql,terminalId,customerId);
        if(CollectionUtil.listNotEmptyNotSizeZero(tmRTermCustPosBGEntities)){
            return true;
        }
        return false;
    }

    /**
     * 添加错误信息提示
     * @param error
     * @throws ValidateException
     */
    private void addErrorMsg(String error) throws ValidateException {
        this.errors.append("【第" +this.getRowNumber()+"行】:" + error).append("<br/>");
        this.errNum += 1;
        if (this.errNum > 200)
            throw new ValidateException(this.errors.toString());
    }

    public void end(){
        if(errNum > 0) {
            //返回导入验证失败的数据
            throw new BusinessException(this.errors.toString());
        }

        //保存导入数据
        this.saveData();
        //导入是分批次提交，end()方法会多次调用，保存完了把数据清掉，不然会重复保存
        importDataList.clear();
    }

    /**
     * 保存导入数据到数据库
     */
    private void saveData() {
        if(CollectionUtil.listNotEmptyNotSizeZero(importDataList)) {
            for(TmRTermCustPosBGEntity entity : importDataList) {
                tmTermCustPostService.save(entity);
                saveIdList.add(entity.getId());
            }
        }
    }

    @Override
    public void impEnd() {
        super.impEnd();
        if (null != tmTerminalBackExtendService){
            tmTerminalBackExtendService.afterImportTmTermCustRel(saveIdList);
        }
    }
}
