package com.biz.eisp.mdm.timeTask.service;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URI;
import java.util.List;
import java.util.Set;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.SAXReader;
import org.dom4j.io.XMLWriter;
import org.quartz.CronTrigger;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.TriggerKey;
import org.quartz.impl.StdScheduler;
import org.quartz.impl.matchers.GroupMatcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import com.biz.eisp.base.common.util.StringUtil;
import com.biz.eisp.mdm.timeTask.util.QuartzManagerUtil;

/**
 * 动态任务,用以动态调整Spring的任务.
 * @author grover
 * @version v1.0
 */
@Service(value = "dynamicTask")
public class DynamicTask {

    private static Logger logger = LoggerFactory.getLogger(DynamicTask.class);
    @Resource
    private StdScheduler schedulerFactory;

    /**
     * 更新定时任务的触发表达式
     * @author grover
     * @param triggerName
     *            触发器名字
     * @param start
     *            触发表达式
     * @return 成功则返回true，否则返回false
     */
    public boolean startOrStop(String triggerName, boolean start) {
        try {
            CronTrigger trigger = (CronTrigger) getTrigger(triggerName, Scheduler.DEFAULT_GROUP);
            if (start) {
            	schedulerFactory.resumeTrigger(trigger.getKey());
                logger.info("trigger the start successfully!!");
            } else {
            	schedulerFactory.pauseTrigger(trigger.getKey());
                logger.info("trigger the pause successfully!!");
            }
            return true;
        } catch (SchedulerException e) {
            logger.error("Fail to reschedule. " + e);
            return false;
        }
    }

    /**
     * 更新定时任务的触发表达式。
     * @author grover
     * @param triggerName
     *            触发器名字
     * @param cronExpression
     *            触发表达式
     * @return 成功则返回true，否则返回false
     */
    public boolean updateCronExpression(String triggerName, String cronExpression) {
        try {
            CronTrigger trigger = (CronTrigger) getTrigger(triggerName, Scheduler.DEFAULT_GROUP);
            if (trigger == null) {
                return false;
            }
            if (StringUtils.equals(trigger.getCronExpression(), cronExpression)) {
                logger.info("与当前运行任务的触发器表达式一致，无需修改.");
                return true;
            }
            QuartzManagerUtil.changeCronExpByName(triggerName, cronExpression);
            updateSpringMvcTaskXML(trigger, cronExpression);
            logger.info("修改触发器表达式成功!!");
            return true;
        } catch (Exception e) {
        	e.printStackTrace();
            logger.error("触发器表达式修改失败. " + e);
            return false;
        }
    }

    /**
     * 获取触发器。
     * @author grover
     * @param triggerName
     *            触发器名字
     * @param groupName
     *            触发器组名字
     * @return 对应Trigger
     */
    private Trigger getTrigger(String triggerName, String groupName) {
        Trigger trigger = null;
        if (StringUtils.isBlank(groupName)) {
            logger.warn("Schedule Job Group is empty!");
            return null;
        }
        if (StringUtils.isBlank(triggerName)) {
            logger.warn("Schedule trigger Name is empty!");
            return null;
        }
        try {
        	Set<TriggerKey> triggerKeys = schedulerFactory.getTriggerKeys(GroupMatcher.triggerGroupEquals(groupName));
        	if(triggerKeys != null && triggerKeys.size() > 0){
        		for (TriggerKey triggerKey : triggerKeys) {
        			if(StringUtil.isNotBlank(triggerKey.getName()) && triggerKey.getName().equals(triggerName)){
        				trigger = schedulerFactory.getTrigger(triggerKey);
        				break;
        			}
				}
        	}
        } catch (SchedulerException e) {
            logger.warn("Fail to get the trigger (triggerName: " + triggerName + ", groupName : " + groupName + ")");
            return null;
        }
        if (trigger == null) {
            logger.warn("Can not found the trigger of triggerName: " + triggerName + ", groupName : " + groupName);
        }
        return trigger;
    }

    /**
     * 更新spring-quartz.xml 配置文件
     * @author grover
     * @param trigger 触发器
     * @param cronExpression 新cron表达式
     */
    @SuppressWarnings("unchecked")
    public synchronized static void updateSpringMvcTaskXML(CronTrigger trigger, String cronExpression) {
        Document document = null;
        File file = null;
        SAXReader saxReader = new SAXReader();
        try {
            URI url = DynamicTask.class.getClassLoader().getResource("spring/spring-quartz.xml").toURI();
            file = new File(url.getPath());
            document = saxReader.read(new FileInputStream(file));
        } catch (Exception e) {
            logger.error("读取系统中用到的SQL 语句XML出错");
            throw new RuntimeException("---------读取spring-quartz.xml文件出错:" + e.getMessage());
        }
        Element root = document.getRootElement();
        List<Element> beans = root.elements();
        for (Element bean : beans) {
            if (bean.attribute("id") != null && bean.attribute("id").getValue().equals(trigger.getKey().getName())) {
                beans = bean.elements();
                for (Element temp : beans) {
                    if (temp.attribute("name") != null && temp.attribute("name").getValue().equals("cronExpression")) {
                        temp.attribute("value").setValue(cronExpression);
                        break;
                    }
                }
                break;
            }
        }
        XMLWriter fileWriter = null;
        try {
            OutputFormat xmlFormat = OutputFormat.createPrettyPrint();
            xmlFormat.setEncoding("utf-8");
            fileWriter = new XMLWriter(new FileOutputStream(file), xmlFormat);
            fileWriter.write(document);
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {
                fileWriter.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

    }

}
