/* Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.activiti.engine.impl.bpmn.behavior;

import com.biz.eisp.activiti.designer.processconf.entity.TaProcessNodeProEntity;
import com.biz.eisp.activiti.designer.processconf.service.TaProcessNodeService;
import com.biz.eisp.activiti.runtime.entity.TaProcessApprovalLogEntity;
import com.biz.eisp.activiti.runtime.service.TaTaskService;
import com.biz.eisp.base.common.exception.BusinessException;
import com.biz.eisp.base.common.util.CollectionUtil;
import com.biz.eisp.base.common.util.ResourceConfigUtils;
import com.biz.eisp.base.common.util.StringUtil;
import com.biz.eisp.base.utils.ApplicationContextUtils;
import com.biz.eisp.mdm.user.entity.TmUserEntity;
import org.activiti.engine.ActivitiException;
import org.activiti.engine.ActivitiIllegalArgumentException;
import org.activiti.engine.delegate.Expression;
import org.activiti.engine.delegate.TaskListener;
import org.activiti.engine.impl.calendar.BusinessCalendar;
import org.activiti.engine.impl.calendar.DueDateBusinessCalendar;
import org.activiti.engine.impl.context.Context;
import org.activiti.engine.impl.persistence.entity.ExecutionEntity;
import org.activiti.engine.impl.persistence.entity.IdentityLinkEntity;
import org.activiti.engine.impl.persistence.entity.TaskEntity;
import org.activiti.engine.impl.pvm.delegate.ActivityExecution;
import org.activiti.engine.impl.task.TaskDefinition;
import org.apache.commons.lang.StringUtils;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;

import java.util.*;

/**
 * activity implementation for the user task.
 * 复写， 加入创建任务时，处理重复节点和空节点自动跳过功能。
 */
public class UserTaskActivityBehavior extends TaskActivityBehavior {

    protected TaskDefinition taskDefinition;

    public UserTaskActivityBehavior(TaskDefinition taskDefinition) {
        this.taskDefinition = taskDefinition;
    }

    public void execute(ActivityExecution execution) throws Exception {
        TaskEntity task = TaskEntity.createAndInsert(execution);
        task.setExecution(execution);
        task.setTaskDefinition(taskDefinition);

        if (taskDefinition.getNameExpression() != null) {
            String name = (String) taskDefinition.getNameExpression().getValue(execution);
            task.setName(name);
        }

        if (taskDefinition.getDescriptionExpression() != null) {
            String description = (String) taskDefinition.getDescriptionExpression().getValue(execution);
            task.setDescription(description);
        }

        if(taskDefinition.getDueDateExpression() != null) {
            Object dueDate = taskDefinition.getDueDateExpression().getValue(execution);
            if(dueDate != null) {
                if (dueDate instanceof Date) {
                    task.setDueDate((Date) dueDate);
                } else if (dueDate instanceof String) {
                    BusinessCalendar businessCalendar = Context
                            .getProcessEngineConfiguration()
                            .getBusinessCalendarManager()
                            .getBusinessCalendar(DueDateBusinessCalendar.NAME);
                    task.setDueDate(businessCalendar.resolveDuedate((String) dueDate));
                } else {
                    throw new ActivitiIllegalArgumentException("Due date expression does not resolve to a Date or Date string: " +
                            taskDefinition.getDueDateExpression().getExpressionText());
                }
            }
        }

        if (taskDefinition.getPriorityExpression() != null) {
            final Object priority = taskDefinition.getPriorityExpression().getValue(execution);
            if (priority != null) {
                if (priority instanceof String) {
                    try {
                        task.setPriority(Integer.valueOf((String) priority));
                    } catch (NumberFormatException e) {
                        throw new ActivitiIllegalArgumentException("Priority does not resolve to a number: " + priority, e);
                    }
                } else if (priority instanceof Number) {
                    task.setPriority(((Number) priority).intValue());
                } else {
                    throw new ActivitiIllegalArgumentException("Priority expression does not resolve to a number: " +
                            taskDefinition.getPriorityExpression().getExpressionText());
                }
            }
        }

        if (taskDefinition.getCategoryExpression() != null) {
            final Object category = taskDefinition.getCategoryExpression().getValue(execution);
            if (category != null) {
                if (category instanceof String) {
                    task.setCategory((String) category);
                } else {
                    throw new ActivitiIllegalArgumentException("Category expression does not resolve to a string: " +
                            taskDefinition.getCategoryExpression().getExpressionText());
                }
            }
        }

        handleAssignments(task, execution);

        // All properties set, now firing 'create' event
        task.fireEvent(TaskListener.EVENTNAME_CREATE);

        //查询查询任务指派人
        String assignPositionCodes = this.findAssignPositionCodes(task);
        String version=task.getProcessDefinitionId();

        TaProcessNodeService taProcessNodeService = ApplicationContextUtils.getContext()
                .getBean(TaProcessNodeService.class);
        TaProcessNodeProEntity processNodeEntity = taProcessNodeService.getTaProcessNodeByVersionKey(task.getProcessDefinitionId(),task.getProcessInstance().getProcessDefinition().getKey());

        if(StringUtils.isBlank(assignPositionCodes)) {
            //没有找到审批人，自动完成节点
            if (processNodeEntity!=null) {
                String nullJumpNode = processNodeEntity.getNullJumpNode();
                if (StringUtils.isNotBlank(nullJumpNode)
                        && "Y".equals(nullJumpNode)) {
                    saveApproveLog(task, "空节点跳过");
                    task.complete();
                } else {
                    throw new BusinessException("流程节点'" +
                            processNodeEntity.getProcessNodeName() + "'没有找到任务指派人，请配置");
                }
            }
        } else {
            boolean isDuplicateApprove = isDuplicateApprove(task, assignPositionCodes);
            if(isDuplicateApprove && processNodeEntity!=null) { //重复自动跳过
                String duplicateJumpNode = processNodeEntity.getDuplicateJumpNode();
                if (StringUtil.isNotBlank(duplicateJumpNode)
                        && "Y".equals(duplicateJumpNode)){
                    saveApproveLog(task, "重复通过");
                    task.complete();
                }
            }
        }
    }

    /**
     * 是否重复审批
     * @param task
     * @param assignPositionCodes
     * @return
     */
    private boolean isDuplicateApprove(TaskEntity task, String assignPositionCodes) {
        String procInstId = task.getProcessInstanceId();
        String strCodes[] = assignPositionCodes.split(",");

        String sql = "SELECT * FROM ACT_HI_TASKINST WHERE PROC_INST_ID_ = :procInstId AND OWNER_ IN (:assignPositionCodes)";

        NamedParameterJdbcTemplate named = ApplicationContextUtils.getContext().getBean(NamedParameterJdbcTemplate.class);

        MapSqlParameterSource paramMap = new MapSqlParameterSource();
        paramMap.addValue("procInstId", procInstId);
        paramMap.addValue("assignPositionCodes", Arrays.asList(strCodes));

        List<Map<String, Object>> list = named.queryForList(sql, paramMap);

        if(CollectionUtil.listNotEmptyNotSizeZero(list)) {
            return true;
        }

        return false;
    }

    public void signal(ActivityExecution execution, String signalName, Object signalData) throws Exception {
        if (((ExecutionEntity) execution).getTasks().size() != 0)
            throw new ActivitiException("UserTask should not be signalled before complete");
        leave(execution);
    }

    @SuppressWarnings({ "unchecked", "rawtypes" })
    protected void handleAssignments(TaskEntity task, ActivityExecution execution) {
        if (taskDefinition.getAssigneeExpression() != null) {
            task.setAssignee((String) taskDefinition.getAssigneeExpression().getValue(execution), true, false);
        }

        if (taskDefinition.getOwnerExpression() != null) {
            task.setOwner((String) taskDefinition.getOwnerExpression().getValue(execution));
        }

        if (!taskDefinition.getCandidateGroupIdExpressions().isEmpty()) {
            for (Expression groupIdExpr : taskDefinition.getCandidateGroupIdExpressions()) {
                Object value = groupIdExpr.getValue(execution);
                if (value instanceof String) {
                    List<String> candiates = extractCandidates((String) value);
                    task.addCandidateGroups(candiates);
                } else if (value instanceof Collection) {
                    task.addCandidateGroups((Collection) value);
                } else {
                    throw new ActivitiIllegalArgumentException("Expression did not resolve to a string or collection of strings");
                }
            }
        }

        if (!taskDefinition.getCandidateUserIdExpressions().isEmpty()) {
            for (Expression userIdExpr : taskDefinition.getCandidateUserIdExpressions()) {
                Object value = userIdExpr.getValue(execution);
                if (value instanceof String) {
                    List<String> candiates = extractCandidates((String) value);
                    task.addCandidateUsers(candiates);
                } else if (value instanceof Collection) {
                    task.addCandidateUsers((Collection) value);
                } else {
                    throw new ActivitiException("Expression did not resolve to a string or collection of strings");
                }
            }
        }
    }

    /**
     * 查询任务指派人 以逗号分隔
     * @param task
     * @return
     */
    private String findAssignPositionCodes(TaskEntity task) {
        String assignPositionCodes = "";
        if (StringUtils.isNotBlank(task.getAssignee())) {
            assignPositionCodes = task.getAssignee();
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(task.getIdentityLinks())) {
            for(IdentityLinkEntity identityLinkEntity : task.getIdentityLinks()) {
                assignPositionCodes += identityLinkEntity.getUserId() + ",";
            }
            assignPositionCodes = assignPositionCodes.substring(0, assignPositionCodes.length() - 1);
        }
        return assignPositionCodes;
    }

    /**
     * 保存审批日志信息
     * @param task
     * @param content
     */
    private void saveApproveLog(TaskEntity task, String content) {
        TmUserEntity user = ResourceConfigUtils.getClient().getUser();

        TaProcessApprovalLogEntity log = new TaProcessApprovalLogEntity();
        log.setCreateDate(new Date());
        log.setAccount(user.getUserName());
        log.setContent(content);
        log.setName(user.getFullName());
        log.setProcessInstId(task.getProcessInstanceId());
        log.setPositionCode(ResourceConfigUtils.getCurrPosition().getPositionCode());
        log.setPositionName(ResourceConfigUtils.getCurrPosition().getPositionName());
        log.setTaskDefKey(this.taskDefinition.getKey());
        log.setType(1);

        TaTaskService taTaskService = ApplicationContextUtils.getContext().getBean(TaTaskService.class);

        taTaskService.save(log);
    }

    /**
     * Extract a candidate list from a string.
     *
     * @param str
     * @return
     */
    protected List<String> extractCandidates(String str) {
        return Arrays.asList(str.split("[\\s]*,[\\s]*"));
    }

    // getters and setters //////////////////////////////////////////////////////

    public TaskDefinition getTaskDefinition() {
        return taskDefinition;
    }

}
