package com.ec.primus.commons.jackson.serialize;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;

import java.io.IOException;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class SensitiveFieldJsonSerializer
        extends JsonSerializer<String> {
    private static final Logger log = LoggerFactory.getLogger(SensitiveFieldJsonSerializer.class);

    private static final String MOBILE_REGEX = "^\\d{7,11}$";

    private static final String ID_CARD_REGEX = "^\\d{17}[0-9a-zA-Z]$";


    @Override
    public void serialize(String value, JsonGenerator gen, SerializerProvider serializers)
            throws IOException {
        String trimmedValue = StringUtils.trim(value);
        if (StringUtils.isBlank(trimmedValue)) {
            gen.writeString("");
            return;
        }
        if (trimmedValue.matches("^\\d{7,11}$")) {
            gen.writeString(maskMobileProcess(trimmedValue));
        } else if (trimmedValue.matches("^\\d{17}[0-9a-zA-Z]$")) {
            gen.writeString(maskIdCardProcess(trimmedValue));
        } else {
            gen.writeString(generalMaskProcess(value));
        }
    }


    public static String maskMobile(String mobileString) {
        String trimmedValue = StringUtils.trim(mobileString);
        if (StringUtils.isBlank(trimmedValue)) {
            return mobileString;
        }
        return trimmedValue.matches("^\\d{7,11}$") ? maskMobileProcess(trimmedValue) : mobileString;
    }


    public static String maskIdCard(String idCard) {
        String trimmedValue = StringUtils.trim(idCard);
        if (StringUtils.isBlank(trimmedValue)) {
            return idCard;
        }
        return trimmedValue.matches("^\\d{17}[0-9a-zA-Z]$") ? maskIdCardProcess(trimmedValue) : idCard;
    }


    public static String generalMask(String rawString) {
        String trimmedValue = StringUtils.trim(rawString);
        if (StringUtils.isBlank(trimmedValue)) {
            return rawString;
        }
        return generalMaskProcess(trimmedValue);
    }

    private static String maskMobileProcess(String mobileString) {
        int length = mobileString.length();
        int prefixRawStringLength = Math.max(1, Math.min(length - 8, 3));
        StringBuilder sb = new StringBuilder(mobileString.substring(0, prefixRawStringLength));
        int maskLength = length - prefixRawStringLength - 4;
        int maxMaskLength = 4;
        for (int i = 0; i < Math.min(maskLength, maxMaskLength); i++) {
            sb.append("*");
        }
        sb.append(mobileString.substring(mobileString.length() - 4));
        return sb.toString();
    }

    private static String maskIdCardProcess(String idCard) {
        return String.format("%s*******%s****", idCard
                .substring(0, 3), idCard
                .substring(10, idCard.length() - 4));
    }

    private static String generalMaskProcess(String rawString) {
        switch (rawString.length()) {
            case 1:
                return "*";
            default:
        }
        int maskLength = rawString.length() - 1;
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < maskLength; i++) {
            sb.append("*");
        }
        sb.append(rawString.substring(rawString.length() - 1));
        return sb.toString();
    }
}