package com.ec.primus.commons.support.rest;

import com.ec.primus.commons.utils.ValueUtils;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Enumeration;
import java.util.Map;


public class HttpServletHelper {
    public static final String X_FORWARDED_FOR = "X-FORWARDED-FOR";
    public static final String X_REAL_IP = "X-Real-IP";

    public static String getCookieValue(HttpServletRequest request, String key) {
        Cookie[] cookies = request.getCookies();
        if (ArrayUtils.isEmpty(cookies)) {
            return null;
        }
        String value = null;
        for (Cookie cookie : cookies) {
            if (cookie.getName().equals(key)) {
                value = cookie.getValue();
                break;
            }
        }
        return value;
    }


    public static void putSessionValue(HttpServletRequest request, String key, Object value) {
        request.getSession().setAttribute(key, value);
    }


    public static <T> T getSessionValue(HttpServletRequest request, String key) {
        T value = (T) request.getSession().getAttribute(key);
        return value;
    }


    public static boolean getBooleanValueFromSession(HttpServletRequest request, String key) {
        Boolean booleanValue = (Boolean) getSessionValue(request, key);
        return ValueUtils.getValue(booleanValue, Boolean.FALSE);
    }


    public static void setCookie(HttpServletResponse response, String key, String value, int expireSeconds, String path, Boolean httpOnly) {
        Cookie cookie = new Cookie(key, value);
        cookie.setMaxAge(expireSeconds);
        cookie.setPath(path);
        cookie.setHttpOnly(ValueUtils.getValue(httpOnly, Boolean.TRUE));
        response.addCookie(cookie);
    }


    public static void setCookie(HttpServletResponse response, String key, String value, int expireSeconds, String path) {
        setCookie(response, key, value, expireSeconds, path, Boolean.TRUE);
    }


    public static String getClientIP(HttpServletRequest request) {
        String ipAddress = request.getHeader("X-FORWARDED-FOR");
        if (ipAddress == null) {
            String xRealIp = request.getHeader("X-Real-IP");
            return StringUtils.isBlank(xRealIp) ? request.getRemoteHost() : xRealIp;
        }
        return ipAddress;
    }


    public static String getRequestURIWithParameters(HttpServletRequest request) {
        String requestURI = request.getRequestURI();
        String queryString = request.getQueryString();
        return StringUtils.isBlank(queryString) ? requestURI : String.format("%s?%s", requestURI, queryString);
    }


    public static Map<String, String> getHeaders(HttpServletRequest request) {
        Enumeration<?> headerNames = request.getHeaderNames();
        Map<String, String> headers = Maps.newHashMap();
        while (headerNames.hasMoreElements()) {
            String nextHeader = headerNames.nextElement().toString();
            String nextHeaderValue = request.getHeader(nextHeader);
            headers.put(nextHeader, nextHeaderValue);
        }
        return headers;
    }
}
