package com.ec.primus.commons.utils;

import com.ec.primus.commons.constant.DatePeriod;
import com.ec.primus.commons.exception.DateException;
import org.apache.commons.lang3.StringUtils;

import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;


public class DateUtils {
    public static final String DEFAULT_DATE_SIMPLE_PATTERN = "yyyy-MM-dd";
    public static final String DEFAULT_DATETIME_24HOUR_PATTERN = "yyyy-MM-dd HH:mm:ss";
    private static final ThreadLocal<SimpleDateFormat> simpleDateFormatCache = new ThreadLocal<>();

    private static final ThreadLocal<Calendar> calendarCache = new ThreadLocal<>();


    public static Date getDateFromString(String dateString) {
        return getDateFromString(dateString, "yyyy-MM-dd HH:mm:ss");
    }


    public static Date getDateFromString(String dateString, String pattern) {
        try {
            SimpleDateFormat df = buildDateFormat(pattern);
            return df.parse(dateString);
        } catch (ParseException e) {
            throw new DateException(String.format("Could not parse %s with pattern %s.", new Object[]{dateString, pattern}), e);
        }
    }


    public static String getDateFormat(Date date, String pattern) {
        SimpleDateFormat simpleDateFormat = buildDateFormat(pattern);
        return simpleDateFormat.format(date);
    }


    public static String getDateFormat(String date, String datePattern, String formatPattern) {
        Date parsedDate = getDateFromString(date, datePattern);
        SimpleDateFormat simpleDateFormat = buildDateFormat(formatPattern);
        return simpleDateFormat.format(parsedDate);
    }


    public static Date getDateOfSecondsBack(int secondsBack, Date date) {
        return dateBack(13, secondsBack, date);
    }


    public static Date getDateOfMinutesBack(int minutesBack, Date date) {
        return dateBack(12, minutesBack, date);
    }


    public static Date getDateOfHoursBack(int hoursBack, Date date) {
        return dateBack(11, hoursBack, date);
    }


    public static Date getDateOfDaysBack(int daysBack, Date date) {
        return dateBack(5, daysBack, date);
    }


    public static Date getDateOfWeeksBack(int weeksBack, Date date) {
        return dateBack(4, weeksBack, date);
    }


    public static Date getDateOfMonthsBack(int monthsBack, Date date) {
        return dateBack(2, monthsBack, date);
    }


    public static Date getDateOfYearsBack(int yearsBack, Date date) {
        return dateBack(1, yearsBack, date);
    }


    public static int getSecondOfMinute(Date date) {
        return getNumberOfGranularity(13, date);
    }


    public static int getMinuteOfHour(Date date) {
        return getNumberOfGranularity(12, date);
    }


    public static int getHourOfDay(Date date) {
        return getNumberOfGranularity(11, date);
    }


    public static int getDayOfWeek(Date date) {
        return getNumberOfGranularity(7, date);
    }


    public static int getDayOfMonth(Date date) {
        return getNumberOfGranularity(5, date);
    }


    public static int getDayOfYear(Date date) {
        return getNumberOfGranularity(6, date);
    }


    public static int getWeekOfMonth(Date date) {
        return getNumberOfGranularity(4, date);
    }


    public static int getWeekOfYear(Date date) {
        return getNumberOfGranularity(3, date);
    }


    public static int getMonthOfYear(Date date) {
        return getNumberOfGranularity(2, date) + 1;
    }


    public static int getYear(Date date) {
        return getNumberOfGranularity(1, date);
    }


    public static boolean isFirstDayOfTheMonth(Date date) {
        return getDayOfMonth(date) == 1;
    }


    public static boolean isLastDayOfTheMonth(Date date) {
        Date dateOfMonthsBack = getDateOfMonthsBack(-1, date);
        int dayOfMonth = getDayOfMonth(dateOfMonthsBack);
        Date dateOfDaysBack = getDateOfDaysBack(dayOfMonth, dateOfMonthsBack);
        return dateOfDaysBack.equals(date);
    }


    public static Date getCurrentDate() {
        return new Date();
    }


    public static Timestamp currentTimestamp() {
        return new Timestamp(System.currentTimeMillis());
    }


    public static Date buildDate(long times) {
        return new Date(times);
    }


    public static long subSeconds(Date date1, Date date2) {
        return subTime(date1, date2, DatePeriod.SECOND);
    }


    public static long subMinutes(Date date1, Date date2) {
        return subTime(date1, date2, DatePeriod.MINUTE);
    }


    public static long subHours(Date date1, Date date2) {
        return subTime(date1, date2, DatePeriod.HOUR);
    }


    public static long subDays(String dateString1, String dateString2) {
        Date date1 = getDateFromString(dateString1, "yyyy-MM-dd");
        Date date2 = getDateFromString(dateString2, "yyyy-MM-dd");
        return subDays(date1, date2);
    }


    public static long subDays(Date date1, Date date2) {
        return subTime(date1, date2, DatePeriod.DAY);
    }


    public static long subMonths(String dateString1, String dateString2) {
        Date date1 = getDateFromString(dateString1, "yyyy-MM-dd");
        Date date2 = getDateFromString(dateString2, "yyyy-MM-dd");
        return subMonths(date1, date2);
    }


    public static long subMonths(Date date1, Date date2) {
        Calendar calendar1 = buildCalendar(date1);
        int monthOfDate1 = calendar1.get(2);
        int yearOfDate1 = calendar1.get(1);
        Calendar calendar2 = buildCalendar(date2);
        int monthOfDate2 = calendar2.get(2);
        int yearOfDate2 = calendar2.get(1);
        int subMonth = Math.abs(monthOfDate1 - monthOfDate2);
        int subYear = Math.abs(yearOfDate1 - yearOfDate2);
        return subYear * 12 + subMonth;
    }


    public static long subYears(String dateString1, String dateString2) {
        Date date1 = getDateFromString(dateString1, "yyyy-MM-dd");
        Date date2 = getDateFromString(dateString2, "yyyy-MM-dd");
        return subMonths(date1, date2);
    }


    public static long subYears(Date date1, Date date2) {
        return Math.abs(getYear(date1) - getYear(date2));
    }


    public static Date formatToStartOfDay(Date date) {
        try {
            SimpleDateFormat dateFormat = buildDateFormat("yyyy-MM-dd");
            String formattedDate = dateFormat.format(date);
            return dateFormat.parse(formattedDate);
        } catch (ParseException pe) {
            throw new DateException("Unparseable date specified.", pe);
        }
    }


    public static Date formatToEndOfDay(Date date) {
        return getDateOfSecondsBack(1, getDateOfDaysBack(-1, formatToStartOfDay(date)));
    }

    public static String format2String(LocalDateTime time) {
        return format2String(Timestamp.valueOf(time), (String)null);
    }

    public static String format2String(Timestamp ts, String fmt) {
        if(ts == null) {
            return null;
        } else {
            if(fmt == null) {
                fmt = "yyyy-MM-dd HH:mm:ss";
            }

            try {
                SimpleDateFormat sdf = new SimpleDateFormat(fmt);
                return sdf.format(ts);
            } catch (Exception var3) {
                return null;
            }
        }
    }

    private static SimpleDateFormat buildDateFormat(String pattern) {
        SimpleDateFormat simpleDateFormat = (SimpleDateFormat) simpleDateFormatCache.get();
        if (simpleDateFormat == null) {
            simpleDateFormat = new SimpleDateFormat();
            simpleDateFormatCache.set(simpleDateFormat);
        }
        simpleDateFormat.applyPattern(pattern);
        return simpleDateFormat;
    }


    private static Calendar buildCalendar() {
        Calendar calendar = (Calendar) calendarCache.get();
        if (calendar == null) {
            calendar = GregorianCalendar.getInstance();
            calendarCache.set(calendar);
        }
        return calendar;
    }


    private static Calendar buildCalendar(Date date) {
        Calendar calendar = buildCalendar();
        calendar.setTime(date);
        return calendar;
    }

    private static long subTime(Date date1, Date date2, long granularity) {
        long time1 = date1.getTime();
        long time = date2.getTime();
        long subTime = Math.abs(time1 - time);
        return subTime / granularity;
    }

    private static int getNumberOfGranularity(int granularity, Date date) {
        Calendar calendar = buildCalendar(date);
        return calendar.get(granularity);
    }


    private static long getTimeBackInMillis(int granularity, int numberToBack, Date date) {
        Calendar calendar = buildCalendar(date);
        calendar.add(granularity, -numberToBack);
        return calendar.getTimeInMillis();
    }

    private static Date dateBack(int granularity, int numberToBack, Date date) {
        long timeBackInMillis = getTimeBackInMillis(granularity, numberToBack, date);
        return buildDate(timeBackInMillis);
    }

    /**
     * 格式化时间字符串为Timestamp
     *
     * @param dateStr 时间字符串
     * @return
     */
    public static Timestamp getTimestamp(String dateStr) {
        if (StringUtils.isNotBlank(dateStr) && dateStr.matches("\\d{4}-\\d{1,2}-\\d{1,2}")) {
            return getTimestamp(dateStr, "yyyy-MM-dd");
        }
        return getTimestamp(dateStr, "yyyy-MM-dd HH:mm:ss");
    }
    public static Timestamp getTimestamp(String dateStr, String dateFmt) {
        Timestamp timestamp = null;
        SimpleDateFormat f = new SimpleDateFormat(dateFmt);
        try {
            timestamp = new Timestamp(f.parse(dateStr).getTime());
        } catch (ParseException ignored) {
        }
        return timestamp;
    }

}