package com.ec.primus.commons.utils;

import com.ec.primus.commons.exception.CommonExceptions;
import com.google.common.collect.Lists;

import java.lang.annotation.Annotation;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.core.env.Environment;
import org.springframework.web.context.ContextLoader;


public class SpringContextUtils
        implements ApplicationContextAware {
    private static ApplicationContext APPLICATION_CONTEXT;

    public SpringContextUtils() {
    }

    public SpringContextUtils(ApplicationContext applicationContext) {
        if (APPLICATION_CONTEXT != null) {
            return;
        }
        AssertUtils.notNull(applicationContext, CommonExceptions.PARAM_ERROR, "ApplicationContext is null");
        APPLICATION_CONTEXT = applicationContext;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        APPLICATION_CONTEXT = applicationContext;
    }

    private static ApplicationContext getApplicationContext() {
        if (APPLICATION_CONTEXT == null) {
            APPLICATION_CONTEXT = ContextLoader.getCurrentWebApplicationContext();
        }
        return APPLICATION_CONTEXT;
    }

    public static <T> T getBean(Class<T> beanType) {
        return getApplicationContext() == null ? null : getApplicationContext().getBean(beanType);
    }

    public static Object getBean(String beanName) {
        return getApplicationContext() == null ? null : getApplicationContext().getBean(beanName);
    }

    public static List findBeansByAnnotation(Class<? extends Annotation> annotationClass) {
        ApplicationContext applicationContext = getApplicationContext();
        if (applicationContext == null) {
            return Collections.EMPTY_LIST;
        }
        Map beanMaps = applicationContext.getBeansWithAnnotation(annotationClass);
        if (MapUtils.isEmpty(beanMaps)) {
            return Collections.EMPTY_LIST;
        }
        return Lists.newArrayList(beanMaps.values());
    }

    public static <A extends Annotation> A findAnnotationFromBean(Class<A> annotationClass) {
        List beansWithAnnotation = findBeansByAnnotation(annotationClass);
        if (CollectionUtils.isNotEmpty(beansWithAnnotation)) {
            Object next = beansWithAnnotation.iterator().next();
            return AnnotationUtils.findAnnotation(next.getClass(), annotationClass);
        }
        return null;
    }


    public static Boolean isSpringValue(String key) {
        return (key != null) && ((key.startsWith("${")) || ((key.startsWith("#{")) && (key.endsWith("}"))));
    }

    public static String getProperty(String propertyKey) {
        return getProperty(propertyKey, null);
    }

    public static String getProperty(String propertyKey, String defaultValue) {
        if (StringUtils.isBlank(propertyKey)) {
            return null;
        }
        Environment environment = (Environment) getBean(Environment.class);
        if (environment == null) {
            return null;
        }
        String key = propertyKey;
        String defaultReturnValue = defaultValue;
        if (isSpringValue(propertyKey)) {
            key = propertyKey.substring(2, propertyKey.length() - 1);
            if (propertyKey.contains(":")) {
                String[] keyAndDefaultValue = key.split(":");
                key = keyAndDefaultValue[0];
                defaultReturnValue = keyAndDefaultValue[1];
            }
        }
        return environment.getProperty(key, defaultReturnValue);
    }
}