package com.ec.primus.commons.utils.bean;

import com.google.common.collect.Table;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.FatalBeanException;
import org.springframework.core.BridgeMethodResolver;
import org.springframework.core.convert.converter.Converter;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.WeakHashMap;

public class CopyUtils {
    private static WeakHashMap<Method, Method> bridgeMethodCache = new WeakHashMap();

    private CopyUtils() {
    }

    public static <T> T copyProperties(Object source, T target) {
        return copyProperties(source, target, false);
    }

    public static <T> T copyProperties(Object source, T target, String... ignoreProperties) {
        return copyProperties(source, target, false, ignoreProperties);
    }

    public static <T> T copyNotNullProperies(Object source, T target) {
        return copyProperties(source, target, true);
    }

    public static <T> T copyNotNullProperties(Object source, T target, String... ignoreProperties) {
        return copyProperties(source, target, true, ignoreProperties);
    }

    private static <T> T copyProperties(Object source, T target, boolean ignoreNullValue, String... ignoreProperties) {
        return copyProperties(source, target, ignoreNullValue, null, ignoreProperties);
    }

    static <T> T copyProperties(Object source, T target, boolean ignoreNullValue, Table<Class, Class, Converter> converterTable, String... ignoreProperties) {
        Assert.notNull(source, "Source must not be null");
        Assert.notNull(target, "Target must not be null");
        Class<?> actualEditable = target.getClass();
        PropertyDescriptor[] targetPds = BeanUtils.getPropertyDescriptors(actualEditable);
        List<String> ignoreList = ignoreProperties != null? Arrays.asList(ignoreProperties): Collections.emptyList();
        PropertyDescriptor[] var8 = targetPds;
        int var9 = targetPds.length;

        for(int var10 = 0; var10 < var9; ++var10) {
            PropertyDescriptor targetPd = var8[var10];
            if(!ignoreList.contains(targetPd.getName())) {
                Method writeMethod = resolveBridgeMethod(targetPd.getWriteMethod());
                if(writeMethod != null) {
                    PropertyDescriptor sourcePd = BeanUtils.getPropertyDescriptor(source.getClass(), targetPd.getName());
                    if(sourcePd != null) {
                        Method readMethod = resolveBridgeMethod(sourcePd.getReadMethod());
                        if(readMethod != null) {
                            Class sourcePropertyType = readMethod.getReturnType();
                            Class targetPropertyType = writeMethod.getParameterTypes()[0];
                            boolean useTable = converterTable != null && !converterTable.isEmpty();
                            Object value;
                            if(ClassUtils.isAssignable(targetPropertyType, sourcePropertyType)) {
                                try {
                                    value = readMethod.invoke(source);
                                    if(value != null || !ignoreNullValue) {
                                        writeMethod.invoke(target, value);
                                    }
                                } catch (Throwable var21) {
                                    throw new FatalBeanException("Could not copy property '" + targetPd.getName() + "' from source to target", var21);
                                }
                            } else if(useTable && converterTable.contains(sourcePropertyType, targetPropertyType)) {
                                try {
                                    value = readMethod.invoke(source);
                                    if(value != null || !ignoreNullValue) {
                                        Converter converter = (Converter)converterTable.get(sourcePropertyType, targetPropertyType);
                                        Object destValue = converter.convert(value);
                                        writeMethod.invoke(target, destValue);
                                    }
                                } catch (Throwable var22) {
                                    throw new FatalBeanException("Could not copy property '" + targetPd.getName() + "' from source to target", var22);
                                }
                            }
                        }
                    }
                }
            }
        }

        return target;
    }

    private static Method resolveBridgeMethod(Method sourceMethod) {
        return sourceMethod == null?null:(sourceMethod.isBridge()?(Method)bridgeMethodCache.computeIfAbsent(sourceMethod, (k) -> {
            return BridgeMethodResolver.findBridgedMethod(sourceMethod);
        }):sourceMethod);
    }
}