package com.ec.primus.excel.util;


import com.ec.primus.commons.enums.DescribableEnum;
import com.ec.primus.commons.utils.AssertUtils;
import com.ec.primus.excel.annotation.*;
import com.ec.primus.excel.bean.*;
import com.ec.primus.excel.enums.ExcelFieldCheckErrorType;
import com.ec.primus.excel.exception.ExcelExceptionType;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;

import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * excel工具类
 *
 * @author ning.zhang
 * @date 2021/7/14
 */
public class ExcelExportUtil {

    private ExcelExportUtil() {
        //nothing
    }

    private static class ExcelExportUtilHolder {
        static ExcelExportUtil instance = new ExcelExportUtil();
    }

    public static ExcelExportUtil newInstance() {
        return ExcelExportUtilHolder.instance;

    }

    /**
     * 数据集合导出excel表格
     *
     * @param dataList 待导出数据列表
     * @param dataBook excel工作簿
     * @param deleteColumns 需要删除的列
     * @throws IntrospectionException    IntrospectionException
     * @throws IllegalAccessException    IllegalAccessException
     * @throws InvocationTargetException InvocationTargetException
     */
    public void exportDataListToBook(List<? extends ExcelTemplate> dataList, Workbook dataBook, List<Integer> deleteColumns)
            throws IntrospectionException, IllegalAccessException, InvocationTargetException {
        //删除不需要导出的列标题单元格
        deleteColumn(dataBook,deleteColumns);
        if (CollectionUtils.isNotEmpty(dataList)) {
            buildWorkbook(dataList,dataBook,deleteColumns);
        }
    }
    /**
     * 数据集合导出excel表格
     *
     * @param dataList 待导出数据列表
     * @param dataBook excel工作簿
     * @param deleteColumns 需要删除的列
     * @throws IntrospectionException    IntrospectionException
     * @throws IllegalAccessException    IllegalAccessException
     * @throws InvocationTargetException InvocationTargetException
     */
    private void buildWorkbook(List<? extends ExcelTemplate> dataList, Workbook dataBook,List<Integer> deleteColumns)
            throws IntrospectionException, IllegalAccessException, InvocationTargetException {
        Class<? extends ExcelTemplate> clazz = dataList.get(0).getClass();
        ExcelColumnConversion excelColumnConversion = new ExcelColumnConversion();
        this.buildAnnotationFieldMap(excelColumnConversion, clazz);
        ExcelSheetName excelSheetName = clazz.getAnnotation(ExcelSheetName.class);
        Sheet sheet;
        Sheet templateSheet;
        if (excelSheetName != null && StringUtils.isNotBlank(excelSheetName.name())) {
            sheet = dataBook.getSheet(excelSheetName.name());
            templateSheet = ((SXSSFWorkbook) dataBook).getXSSFWorkbook().getSheet(excelSheetName.name());
        } else {
            //默认获取第一个sheet
            sheet = dataBook.getSheetAt(0);
            templateSheet = ((SXSSFWorkbook) dataBook).getXSSFWorkbook().getSheetAt(0);
        }
        AssertUtils.isTrue(sheet != null, ExcelExceptionType.EXCEL_SHEET_IS_EMPTY);
        AssertUtils.isTrue(!excelColumnConversion.getFieldNameDataMap().isEmpty(), ExcelExceptionType.NOT_HAVE_RECEIVE_DATA_FIELD);
        ExcelDataColumn excelDataColumn = clazz.getAnnotation(ExcelDataColumn.class);
        AssertUtils.isTrue(excelDataColumn != null && excelDataColumn.dataNumber() >= 1, ExcelExceptionType.DATA_LINE_ANNOTATION_IS_EMPTY);
        buildExcelCellMap(excelColumnConversion, templateSheet, excelDataColumn);
        AssertUtils.isTrue(!excelColumnConversion.getFieldNameCellMap().isEmpty(), ExcelExceptionType.NOT_HAVE_RECEIVE_DATA_FIELD);
        for (int rowNum = 0; rowNum < dataList.size(); rowNum++) {
            Row rowAnother = sheet.createRow(rowNum + excelDataColumn.dataNumber());
            writeDataToCell(excelColumnConversion, dataList, clazz, dataBook, rowNum, rowAnother,deleteColumns);
        }
        //触发函数生效
        sheet.setForceFormulaRecalculation(true);
    }

    /**
     * 构建excel字段单元格映射
     *
     * @param excelColumnConversion excel字段转换信息
     * @param templateSheet         excel工作簿
     * @param excelDataColumn       excel数据列注解
     */
    private void buildExcelCellMap(ExcelColumnConversion excelColumnConversion, Sheet templateSheet
            , ExcelDataColumn excelDataColumn) {
        int fieldNumber = excelDataColumn.fieldNumber();
        Row fieldRow = templateSheet.getRow(fieldNumber);
        Map<String, Cell> fieldNameCellMap = Maps.newHashMap();
        Map<Integer, Cell> fieldNumberCellMap = Maps.newHashMap();
        for (Cell cell : fieldRow) {
            Cell nameCell = fieldNameCellMap.get(cell.getStringCellValue());
            AssertUtils.isNull(nameCell, ExcelExceptionType.EXCEL_FIELD_NAME_EXIST);
            fieldNameCellMap.put(cell.getStringCellValue(), cell);
            fieldNumberCellMap.put(cell.getColumnIndex(), cell);
        }
        excelColumnConversion.setFieldNameCellMap(fieldNameCellMap);
        excelColumnConversion.setFieldNumberCellMap(fieldNumberCellMap);
    }

    /**
     * 获取数据类上有ExcelColumn注解的字段的数据映射
     *
     * @param excelColumnConversion excel字段转换信息
     * @param clazz                 数据类
     */
    private void buildAnnotationFieldMap(ExcelColumnConversion excelColumnConversion, Class<? extends ExcelTemplate> clazz) {
        Map<String, Pair<Field, ExcelColumn>> fieldNameMap = Maps.newHashMap();
        Map<Integer, Pair<Field, ExcelColumn>> fieldNumberMap = Maps.newHashMap();
        Field[] fields = clazz.getDeclaredFields();
        for (Field field : fields) {
            ExcelColumn excelColumn = field.getAnnotation(ExcelColumn.class);
            if (excelColumn != null) {
                Pair<Field, ExcelColumn> namePair = fieldNameMap.get(excelColumn.name());
                AssertUtils.isNull(namePair, ExcelExceptionType.EXCEL_DATA_FIELD_NAME_EXIST);
                fieldNameMap.put(excelColumn.name(), Pair.of(field, excelColumn));
                if (excelColumn.columnNumber() >= 0) {
                    Pair<Field, ExcelColumn> numberPair = fieldNumberMap.get(excelColumn.columnNumber());
                    AssertUtils.isNull(numberPair, ExcelExceptionType.EXCEL_DATA_FIELD_NUMBER_EXIST);
                    fieldNumberMap.put(excelColumn.columnNumber(), Pair.of(field, excelColumn));
                }
            }
        }
        excelColumnConversion.setFieldNameDataMap(fieldNameMap);
        excelColumnConversion.setFieldNumberDataMap(fieldNumberMap);
    }

    /**
     * 写入数据到excel单元格内
     *
     * @param excelColumnConversion  excel字段转换信息
     * @param dataList         导入数据列表
     * @param clazz            数据类
     * @param templateBook     excel工作簿
     * @param rowNum           当前写入数据的索引(list的下标)
     * @param rowAnother       当前写入
     * @param deleteColumns 需要删除的列
     * @throws IntrospectionException    IntrospectionException
     * @throws IllegalAccessException    IllegalAccessException
     * @throws InvocationTargetException InvocationTargetException
     */
    private void writeDataToCell(ExcelColumnConversion excelColumnConversion, List<? extends ExcelTemplate> dataList, Class<? extends ExcelTemplate> clazz
            , Workbook templateBook, int rowNum, Row rowAnother,List<Integer> deleteColumns)
            throws IntrospectionException, IllegalAccessException, InvocationTargetException {
        for (Map.Entry<String, Cell> entry : excelColumnConversion.getFieldNameCellMap().entrySet()) {
            Pair<Field, ExcelColumn> namePair = excelColumnConversion.getFieldNameDataMap().get(entry.getKey());
            Pair<Field, ExcelColumn> numberPair = excelColumnConversion.getFieldNumberDataMap().get(entry.getValue().getColumnIndex());
            //列号优先
            Pair<Field, ExcelColumn> fieldExcelColumnPair = numberPair != null ? numberPair : namePair;
            if (fieldExcelColumnPair == null || deleteColumns.contains(entry.getValue().getColumnIndex())) {
                continue;
            }
            Field annotationField = fieldExcelColumnPair.getLeft();
            PropertyDescriptor pd = new PropertyDescriptor(annotationField.getName(), clazz);
            Method method = pd.getReadMethod();
            Object object = method.invoke(dataList.get(rowNum));
            ExcelColumn excelColumn = annotationField.getAnnotation(ExcelColumn.class);
            Class aClass = excelColumn.dataClass();
            ExcelColumnDataFormat dataFormat = annotationField.getAnnotation(ExcelColumnDataFormat.class);
            Cell cell = rowAnother.createCell(entry.getValue().getColumnIndex());
            //设置自定义格式
            buildCellDataFormat(templateBook, object, aClass, dataFormat, cell);
        }
    }

    /**
     * 设置自定义格式
     *
     * @param templateBook excel工作簿
     * @param object       单元格值
     * @param aClass       数据类
     * @param dataFormat   数据格式
     * @param cell         单元格
     */
    private void buildCellDataFormat(Workbook templateBook, Object object, Class aClass, ExcelColumnDataFormat dataFormat, Cell cell) {
        if (dataFormat != null && StringUtils.isNotBlank(dataFormat.dataFormat())) {
            CellStyle style = templateBook.createCellStyle();
            DataFormat format = templateBook.createDataFormat();
            style.setDataFormat(format.getFormat(dataFormat.dataFormat()));
            cell.setCellStyle(style);
        }
        if (aClass == Integer.class) {
            cell.setCellValue((Integer) object);
        } else if (aClass == Long.class) {
            cell.setCellValue((Long) object);
        } else if (aClass == Double.class) {
            cell.setCellValue((Double) object);
        } else if (aClass == Date.class) {
            cell.setCellValue((Date) object);
        } else {
            if (object instanceof DescribableEnum) {
                cell.setCellValue(((DescribableEnum) object).getDesc());
            } else {
                cell.setCellValue(object == null ? "" : object.toString());
            }
        }
    }

    private void deleteColumn(Workbook templateBook,List<Integer> deleteColumns) {
        if (CollectionUtils.isNotEmpty(deleteColumns)) {
            //默认取第一个
            Sheet sheet = ((SXSSFWorkbook) templateBook).getXSSFWorkbook().getSheetAt(0);
            deleteColumns.forEach(columnToDelete -> {
                Row row = sheet.getRow(0);
                for (int cID = columnToDelete; cID < row.getLastCellNum(); cID++) {
                    Cell cOld = row.getCell(cID);
                    if (cOld != null) {
                        row.removeCell(cOld);
                    }
                    Cell cNext = row.getCell(cID + 1);
                    if (cNext != null) {
                        Cell cNew = row.createCell(cID, cNext.getCellTypeEnum());
                        cloneCell(cNew, cNext);
                        sheet.setColumnWidth(cID, sheet.getColumnWidth(cID + 1));
                    }
                }
            });
        }
    }

    private void cloneCell(Cell cNew, Cell cOld) {
        cNew.setCellComment(cOld.getCellComment());
        cNew.setCellStyle(cOld.getCellStyle());
        if (CellType.BOOLEAN == cNew.getCellTypeEnum()) {
            cNew.setCellValue(cOld.getBooleanCellValue());
        } else if (CellType.NUMERIC == cNew.getCellTypeEnum()) {
            cNew.setCellValue(cOld.getNumericCellValue());
        } else if (CellType.STRING == cNew.getCellTypeEnum()) {
            cNew.setCellValue(cOld.getStringCellValue());
        } else if (CellType.ERROR == cNew.getCellTypeEnum()) {
            cNew.setCellValue(cOld.getErrorCellValue());
        } else if (CellType.FORMULA == cNew.getCellTypeEnum()) {
            cNew.setCellValue(cOld.getCellFormula());
        }
    }

    /**
     * excel数据导入数据列表
     *
     * @param workbook excel工作簿
     * @param clazz    数据类
     * @return 数据列表
     * @throws IllegalAccessException    IllegalAccessException
     * @throws InstantiationException    InstantiationException
     * @throws IntrospectionException    IntrospectionException
     * @throws InvocationTargetException InvocationTargetException
     */
    public <T extends ExcelTemplate> List<T> importBookToDataList(Workbook workbook, Class<T> clazz)
            throws IllegalAccessException, InstantiationException, IntrospectionException, InvocationTargetException {
        List<T> listObject = Lists.newArrayList();
        ExcelDataColumn excelDataColumn = clazz.getAnnotation(ExcelDataColumn.class);
        AssertUtils.isTrue(excelDataColumn != null && excelDataColumn.dataNumber() >= 1, ExcelExceptionType.DATA_LINE_ANNOTATION_IS_EMPTY);
        ExcelSheetName excelSheetName = clazz.getAnnotation(ExcelSheetName.class);
        //定义工作表
        Sheet sheet;
        if (excelSheetName != null && StringUtils.isNotBlank(excelSheetName.name())) {
            sheet = workbook.getSheet(excelSheetName.name());
        } else {
            //默认获取第一个sheet
            sheet = workbook.getSheetAt(0);
        }
        ExcelColumnConversion excelColumnConversion = new ExcelColumnConversion();
        this.buildAnnotationFieldMap(excelColumnConversion, clazz);
        AssertUtils.isTrue(!excelColumnConversion.getFieldNameDataMap().isEmpty(), ExcelExceptionType.NOT_HAVE_RECEIVE_DATA_FIELD);
        buildExcelCellMap(excelColumnConversion,sheet,excelDataColumn);
        AssertUtils.isTrue(!excelColumnConversion.getFieldNameCellMap().isEmpty(), ExcelExceptionType.NOT_HAVE_RECEIVE_DATA_FIELD);
        //循环取每行的数据
        for (int rowIndex = 0; rowIndex < sheet.getPhysicalNumberOfRows(); rowIndex++) {
            Row dataRow = sheet.getRow(rowIndex + excelDataColumn.dataNumber());
            if (dataRow == null) {
                continue;
            }
            T obj = clazz.newInstance();
            obj.setLineNumber(dataRow.getRowNum() + 1);
            writeDataToObject(clazz, excelColumnConversion, dataRow, obj);
            listObject.add(obj);
        }
        return listObject;
    }

    /**
     * 写入数据到数据对象
     *
     * @param clazz            数据类
     * @param excelColumnConversion  excel字段转换信息
     * @param dataRow          数据行
     * @param obj              写入对象
     * @throws IntrospectionException    IntrospectionException
     * @throws IllegalAccessException    IllegalAccessException
     * @throws InvocationTargetException InvocationTargetException
     */
    private <T> void writeDataToObject(Class<T> clazz, ExcelColumnConversion excelColumnConversion, Row dataRow, T obj)
            throws IntrospectionException, IllegalAccessException, InvocationTargetException {
        for (Map.Entry<String, Pair<Field, ExcelColumn>> entry : excelColumnConversion.getFieldNameDataMap().entrySet()) {
            Cell nameCell = excelColumnConversion.getFieldNameCellMap().get(entry.getKey());
            Cell numberCell = excelColumnConversion.getFieldNumberCellMap().get(entry.getValue().getRight().columnNumber());
            Cell fieldNameCell = numberCell != null ? numberCell : nameCell;
            if (fieldNameCell == null) {
                continue;
            }
            Field field = entry.getValue().getLeft();
            ExcelColumn excelColumn = field.getAnnotation(ExcelColumn.class);
            PropertyDescriptor pd = new PropertyDescriptor(field.getName(), clazz);
            Cell cell = dataRow.getCell(fieldNameCell.getColumnIndex());
            if (cell != null) {
                Method method = pd.getWriteMethod();
                Class aClass = excelColumn.dataClass();
                if (cell.getCellTypeEnum() == CellType.STRING) {
                    cell.setCellValue(cell.getStringCellValue().trim());
                }
                if (aClass == Integer.class) {
                    int cellValue = cell.getCellTypeEnum() == CellType.NUMERIC ? ((Double) cell.getNumericCellValue()).intValue()
                            : Integer.parseInt(cell.getStringCellValue());
                    method.invoke(obj, cellValue);
                } else if (aClass == Long.class) {
                    long cellValue = cell.getCellTypeEnum() == CellType.NUMERIC ? ((Double) cell.getNumericCellValue()).longValue()
                            : Long.parseLong(cell.getStringCellValue());
                    method.invoke(obj, cellValue);
                } else if (aClass == Double.class) {
                    double cellValue = cell.getCellTypeEnum() == CellType.NUMERIC ? cell.getNumericCellValue()
                            : Double.parseDouble(cell.getStringCellValue());
                    method.invoke(obj, cellValue);
                } else if (aClass == Date.class) {
                    method.invoke(obj, cell.getDateCellValue());
                } else {
                    cell.setCellType(CellType.STRING);
                    method.invoke(obj, StringUtils.isNotBlank(cell.getStringCellValue()) ? cell.getStringCellValue() : cell.getStringCellValue().trim());
                }
            }
        }
    }

    /**
     * 数据校验
     * @param dataList 数据列表
     * @param <T> 数据Class泛型
     * @return 错误信息
     * @throws IntrospectionException IntrospectionException
     * @throws IllegalAccessException IllegalAccessException
     */
    public <T extends ExcelTemplate> List<ExcelFieldErrorInfo> checkTemplateData(List<T> dataList)
            throws IntrospectionException, IllegalAccessException {
        List<ExcelFieldErrorInfo> errorList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(dataList)) {
            Class<? extends ExcelTemplate> clazz = dataList.get(0).getClass();
            ExcelCheckConversion checkConversion = buildCheckFieldMap(clazz);
            if(!checkConversion.getFieldEmptyCheckMap().isEmpty() || !checkConversion.getFieldRepeatCheckMap().isEmpty()) {
                Map<String,T> uniqueDataMap = Maps.newHashMap();
                for (T data : dataList) {
                    checkEmptyData(errorList, checkConversion, data);
                    checkRepeatData(errorList, checkConversion, uniqueDataMap, data);
                }
            }
        }
        return errorList;
    }

    /**
     * 数据判空
     * @param errorList 错误数据
     * @param checkConversion Excel校验转换信息
     * @param data 数据
     * @param <T> 数据Class泛型
     * @throws IllegalAccessException IllegalAccessException
     */
    private <T extends ExcelTemplate> void checkEmptyData(List<ExcelFieldErrorInfo> errorList, ExcelCheckConversion checkConversion, T data) throws IllegalAccessException {
        Map<String, Field> fieldEmptyCheckMap = checkConversion.getFieldEmptyCheckMap();
        List<ExcelFieldInfo> fieldInfoList = Lists.newArrayList();
        for (Map.Entry<String, Field> entry : fieldEmptyCheckMap.entrySet()) {
            Field field = entry.getValue();
            field.setAccessible(true);
            Boolean emptyFlag;
            if (String.class == field.getType()) {
                emptyFlag = field.get(data) == null || StringUtils.isBlank(String.valueOf(field.get(data)));
            } else {
                emptyFlag = field.get(data) == null;
            }
            if (Boolean.TRUE.equals(emptyFlag)) {
                ExcelColumn excelColumn = field.getAnnotation(ExcelColumn.class);
                fieldInfoList.add(new ExcelFieldInfo()
                        .setFieldExcelName(excelColumn.name())
                        .setFieldName(field.getName())
                        .setFieldType(field.getType())
                        .setFieldValue(field.get(data)));

            }
        }
        if (CollectionUtils.isNotEmpty(fieldInfoList)) {
            errorList.add(new ExcelFieldErrorInfo()
                    .setErrorType(ExcelFieldCheckErrorType.EMPTY_OR_NULL)
                    .setLineNumber(data.getLineNumber())
                    .setFieldInfoList(fieldInfoList));
        }
    }

    /**
     * 数据判重
     * @param errorList 错误数据
     * @param checkConversion Excel校验转换信息
     * @param uniqueDataMap 数据映射(唯一键)
     * @param data 数据
     * @param <T> 数据Class泛型
     * @throws IllegalAccessException IllegalAccessException
     */
    private <T extends ExcelTemplate> void checkRepeatData(List<ExcelFieldErrorInfo> errorList
            , ExcelCheckConversion checkConversion, Map<String, T> uniqueDataMap, T data) throws IllegalAccessException {
        Map<String, Field> repeatCheckMap = checkConversion.getFieldRepeatCheckMap();
        if (!repeatCheckMap.isEmpty()) {
            StringBuilder uniqueBuilder = new StringBuilder();
            for (Map.Entry<String, Field> entry : repeatCheckMap.entrySet()) {
                Field field = entry.getValue();
                field.setAccessible(true);
                uniqueBuilder.append(String.valueOf(field.get(data)));
            }
            String uniqueKey = uniqueBuilder.toString();
            T t = uniqueDataMap.get(uniqueKey);
            if (t != null) {
                List<ExcelFieldInfo> fieldInfoList = Lists.newArrayList();
                for (Map.Entry<String, Field> entry : repeatCheckMap.entrySet()) {
                    Field field = entry.getValue();
                    field.setAccessible(true);
                    ExcelColumn excelColumn = field.getAnnotation(ExcelColumn.class);
                    fieldInfoList.add(new ExcelFieldInfo()
                            .setFieldExcelName(excelColumn.name())
                            .setFieldName(entry.getValue().getName())
                            .setFieldType(entry.getValue().getType())
                            .setFieldValue(field.get(data)));
                }
                errorList.add(new ExcelFieldErrorInfo()
                        .setErrorType(ExcelFieldCheckErrorType.DATA_REPEAT)
                        .setLineNumber(data.getLineNumber())
                        .setRepeatNumber(t.getLineNumber())
                        .setFieldInfoList(fieldInfoList));
            } else {
                uniqueDataMap.put(uniqueBuilder.toString(),data);
            }
        }
    }

    /**
     * 获取数据类上有ExcelFieldCheck注解的字段的数据映射
     *
     * @param clazz                 数据类
     * @return Excel校验转换信息
     */
    private ExcelCheckConversion buildCheckFieldMap(Class<? extends ExcelTemplate> clazz) {
        Map<String, Field> fieldEmptyCheckMap = Maps.newHashMap();
        Map<String, Field> fieldRepeatCheckMap = Maps.newHashMap();
        Field[] fields = clazz.getDeclaredFields();
        for (Field field : fields) {
            ExcelColumn excelColumn = field.getAnnotation(ExcelColumn.class);
            ExcelFieldCheck fieldCheck = field.getAnnotation(ExcelFieldCheck.class);
            if (excelColumn != null && fieldCheck != null) {
                if (Boolean.TRUE.equals(fieldCheck.emptyCheck())) {
                    fieldEmptyCheckMap.put(field.getName(),field);
                }
                if (Boolean.TRUE.equals(fieldCheck.uniqueJoin())) {
                    fieldRepeatCheckMap.put(field.getName(),field);
                }
            }
        }
        return new ExcelCheckConversion()
                .setFieldEmptyCheckMap(fieldEmptyCheckMap)
                .setFieldRepeatCheckMap(fieldRepeatCheckMap);
    }
}
