/*!
 * jLayout - JavaScript Layout Algorithms v0.2
 *
 * Licensed under the revised BSD License.
 * Copyright 2008, Bram Stein
 * All rights reserved.
 */
/*global jLayout */
;jLayout = {
	layout : function (spec, shared) {
		var that = {},
			my = shared || {};

		my.hgap = spec.hgap || 0;
		my.vgap = spec.vgap || 0;

		/**
		 * Lay out the container using a layout algorithm.
		 */
		that.layout = function (container) {
			return container;
		};

		/**
		 * Return the preferred size of the container.
		 */
		that.preferred = function (container) {
			return {width: my.hgap, height: my.vgap};
		};

		/**
		 * Return the minimum size the container is allowed to have.
		 */
		that.minimum = function (container) {
			return {width: my.hgap, height: my.vgap};
		};

		/**
		 * Return the maximum size the container is allowed to have.
		 */
		that.maximum = function (container) {
			return {width: Number.MAX_VALUE, height: Number.MAX_VALUE};
		};
		return that;
	},

	/**
	 * Grid layout
	 */
	grid : function (spec) {
		var my = {},
			that = this.layout(spec, my),
			items = spec.items || [],
			// initialize the number of columns to the number of items
			// we're laying out.
			columns = spec.columns || items.length,
			rows = spec.rows || 0;

		if (rows > 0) {
			columns = Math.floor((items.length + rows - 1) / rows);
		}
		else {
			rows = Math.floor((items.length + columns - 1) / columns);
		}
		
		// alex_mod ,fix grid's height & width(ΪԭľǼƽheight & width)
		var widths = spec.widths || [], heights = spec.heights || [];
		$.each([{ar:widths, len:columns}, {ar:heights, len:rows}], function(idx, it) {
			while(it.ar.length < it.len) {
				it.ar[it.ar.length] = -1
			}
		});

		that.layout = function (container) {
			var bounds = container.bounds();
			
			var i, j,
				insets = container.insets(),
				x = insets.left,
				y = insets.top,
				available_width = bounds.width - (insets.left + insets.right) - (columns - 1) * my.hgap,
				available_height = bounds.height - (insets.top + insets.bottom) - (rows - 1) * my.vgap;
				
			// alex_mod columns & rows¼ac_widths & ac_heights
			var ac_widths = Array.prototype.concat.call([], widths), 
				ac_heights = Array.prototype.concat.call([], heights);
			$.each([{ar:ac_widths, len:available_width}, {ar:ac_heights, len:available_height}], function(_i, it) {
				var free_c = 0;
				$.each(it.ar, function(idx, ilen) {
					if (ilen >= 0) {
						it.len -= ilen;
					} else {
						free_c++;
					}
				});
				if (free_c > 0) {
					var av_len = it.len / free_c;
					$.each(it.ar, function(idx, ilen) {
						if (ilen < 0) {
							it.ar[idx] = av_len;
						}
					})
				}
			});
			
			var wi = 0, hi = 0;
			for (var i = 0; i < items.length; i++) {
				var width = ac_widths[wi], height = ac_heights[hi];
				items[i].bounds({'x': x, 'y': y, 'width': width, 'height': height});
				
				if (wi < columns - 1) {
					x += width + my.hgap;
					wi++;
				} else {
					y += height + my.vgap;
					x = insets.left;
					wi = 0;
					hi++;
				}
				items[i].doLayout();
			}

			return container;
		};

		function typeLayout(type) {
			return function (container) {
				var i,
					width = 0,
					height = 0,
					type_size,
					insets = container.insets();
				
				var ca_widths = Array.prototype.concat.call([], widths), 
					ca_heights = Array.prototype.concat.call([], heights);
				
				var wi = 0, hi = 0;
				for (var i = 0; i < items.length; i++) {
					type_size = items[i][type + 'Size']();
					var width = ca_widths[wi], height = ca_heights[hi];
					if (widths[wi] < 0) {
						ca_widths[wi] = Math.max(ca_widths[wi], type_size.width)
					}
					if (heights[hi] < 0) {
						ca_heights[hi] = Math.max(ca_heights[hi], type_size.height)
					}
					
					if (wi < columns - 1) {
						wi++;
					} else {
						wi = 0;
						hi++;
					}
					items[i].doLayout();
				}
				
				// itemܵheight & width
				var ww = 0;
				$.each(ca_widths, function(idx, w) {if (w > 0) ww += w});
				var hh = 0;
				$.each(ca_heights, function(idx, h) {if (h > 0) hh += h});

				return {'width': insets.left + insets.right + ww + (columns - 1) * my.hgap,
						'height': insets.top + insets.bottom + hh + (rows - 1) * my.vgap};
			};
		}

		// this creates the min and preferred size methods, as they
		// only differ in the function they call.
		that.preferred = typeLayout('preferred');
		that.minimum = typeLayout('minimum');
		that.maximum = typeLayout('maximum');
		return that;
	},

	/**
	 * Border layout
	 */
	border : function (spec) {
		var my = {},
			that = this.layout(spec, my),
			east = spec.east,
			west = spec.west,
			north = spec.north,
			south = spec.south,
			center = spec.center;

		that.layout = function (container) {
			var size = container.bounds(),
				insets = container.insets(),
				top = insets.top,
				bottom = size.height - insets.bottom,
				left = insets.left,
				right = size.width - insets.right,
				tmp;

			if (north && north.isVisible()) {
				tmp = north.preferredSize();
				north.bounds({'x': left, 'y': top, 'width': right - left, 'height': tmp.height});
				north.doLayout();

				top += tmp.height + my.vgap;
			}
			if (south && south.isVisible()) {
				tmp = south.preferredSize();
				south.bounds({'x': left, 'y': bottom - tmp.height, 'width': right - left, 'height': tmp.height});
				south.doLayout();

				bottom -= tmp.height + my.vgap;
			}
			if (east && east.isVisible()) {
				tmp = east.preferredSize();
				east.bounds({'x': right - tmp.width, 'y': top, 'width': tmp.width, 'height': bottom - top});
				east.doLayout();

				right -= tmp.width + my.hgap;
			}
			if (west && west.isVisible()) {
				tmp = west.preferredSize();
				west.bounds({'x': left, 'y': top, 'width': tmp.width, 'height': bottom - top});
				west.doLayout();

				left += tmp.width + my.hgap;
			}
			if (center && center.isVisible()) {
				center.bounds({'x': left, 'y': top, 'width': right - left, 'height': bottom - top});
				center.doLayout();
			}
			return container;
		};

		function typeLayout(type) {
			return function (container) {
				var insets = container.insets(),
					width = 0,
					height = 0,
					type_size;

				if (east && east.isVisible()) {
					type_size = east[type + 'Size']();
					width += type_size.width + my.hgap;
					height = type_size.height;
				}
				if (west && west.isVisible()) {
					type_size = west[type + 'Size']();
					width += type_size.width + my.hgap;
					height = Math.max(type_size.height, height);
				}
				if (center && center.isVisible()) {
					type_size = center[type + 'Size']();
					width += type_size.width;
					height = Math.max(type_size.height, height);
				}
				if (north && north.isVisible()) {
					type_size = north[type + 'Size']();
					width = Math.max(type_size.width, width);
					height += type_size.height + my.vgap;
				}
				if (south && south.isVisible()) {
					type_size = south[type + 'Size']();
					width = Math.max(type_size.width, width);
					height += type_size.height + my.vgap;
				}

				return {'width': width + insets.left + insets.right,
						'height': height + insets.top + insets.bottom};
			};
		}
		that.preferred = typeLayout('preferred');
		that.minimum = typeLayout('minimum');
		that.maximum = typeLayout('maximum');
		return that;
	},
	flow : function(spec) {
		var my = {},
		that = this.layout(spec, my),
		items = spec.items || [],
		alignment = (spec.alignment && (spec.alignment === 'center' || spec.alignment === 'right' || spec.alignment === 'left') && spec.alignment) || 'left';		
		hgap = typeof spec.hgap === 'number' && !isNaN(spec.hgap) ? spec.hgap : 5;
		vgap = typeof spec.vgap === 'number' && !isNaN(spec.vgap) ? spec.vgap : 5;
		that.layout = function (container) {
			var parentSize = container.bounds(),
				insets = container.insets(),
				i = 0,
				len = items.length,
				itemSize,
				currentRow = [],
				rowSize = {
					width: 0,
					height: 0
				},
				offset = {
					x: insets.left,
					y: insets.top
				};

			parentSize.width -= insets.left + insets.right;
			parentSize.height -= insets.top + insets.bottom;

			for (; i < len; i += 1) {
				if (items[i].isVisible()) {
					itemSize = items[i].preferredSize();
					
					if ((rowSize.width + itemSize.width) > parentSize.width) {
						align(currentRow, offset, rowSize, parentSize);

						currentRow = [];
						offset.y += rowSize.height;
						offset.x = insets.left;
						rowSize.width = 0;
						rowSize.height = 0;
					}
					rowSize.height = Math.max(rowSize.height, itemSize.height + vgap);
					rowSize.width += itemSize.width + hgap;

					currentRow.push(items[i]);
				}
			}
			align(currentRow, offset, rowSize, parentSize);
			return container;
		};

		function align(row, offset, rowSize, parentSize) {
			var location = {
					x: offset.x,
					y: offset.y
				},
				i = 0,
				len = row.length;

			switch (alignment) {
				case 'center': {
					location.x += (hgap + parentSize.width - rowSize.width) / 2;
					break;
				}
				case 'right': {
					location.x += parentSize.width - rowSize.width + hgap;
					break;
				}
			}

			for (; i < len; i += 1) {
				location.y = offset.y;
				row[i].bounds(location);
				row[i].doLayout();
				location.x += row[i].bounds().width + hgap;
			}
		}

		function typeLayout(type) {
			return function (container) {
				var i = 0, 
					width = 0, 
					height = 0, 
					typeSize,
					firstComponent = false,
					insets = container.insets();

				for (; i < items.length; i += 1) {
					if (items[i].isVisible()) {
						typeSize = items[i][type + 'Size']();
						height = Math.max(height, typeSize.height);
						width += typeSize.width;
					}
				}

				return {
					'width': width + insets.left + insets.right + (items.length - 1) * hgap,
					'height': height + insets.top + insets.bottom
				};
			};
		}

		that.preferred = typeLayout('preferred');
		that.minimum = typeLayout('minimum');
		that.maximum = typeLayout('maximum');		

		return that;
	}
};
