package com.biz.eisp.base.converter;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import javax.persistence.AttributeConverter;
import java.io.Serializable;
import java.util.Arrays;
import java.util.List;

/**
 * 基础类型集合转换器
 * @author bruce.qin
 * @usage
 * @reviewer
 * @since 2017/4/18
 */
public abstract class AbstractListBaseTypeConverter<E extends Serializable> implements AttributeConverter<List<? extends E>, String> {

    private static final String separator = ",";

    private ToDBStringFunction toDBStringFunction;

    private ToSystemObjectFunction toSystemObjectFunction;

    public AbstractListBaseTypeConverter() {
        this.toDBStringFunction = new ToDBStringFunction();
        this.toSystemObjectFunction = new ToSystemObjectFunction();
    }

    @Override
    public String convertToDatabaseColumn(List<? extends E> attribute) {
        if (toDBStringFunction == null){
            this.toDBStringFunction = new ToDBStringFunction();
        }
        if (CollectionUtils.isNotEmpty(attribute)){
            String dbString = StringUtils.join(Lists.transform(attribute, toDBStringFunction), separator);
            return dbString;
        }
        return null;
    }

    @Override
    public List<? extends E> convertToEntityAttribute(String dbData) {
        if (toSystemObjectFunction == null){
            this.toSystemObjectFunction = new ToSystemObjectFunction();
        }
        if (StringUtils.isNotEmpty(dbData)){
            String[] strs = dbData.split(separator);
            List<String> strings = Arrays.asList(strs);
            List<E> result = Lists.newArrayList(Lists.transform(strings, toSystemObjectFunction));
            return result;
        }
        return Lists.newArrayList();
    }

    /**
     * 实现将数据库数据转换成业务数据
     * @param attributeString
     * @return
     */
    protected abstract E toSystemObject(String attributeString);

    /**
     * 实现将业务数据转换成数据库数据,禁止结果String中存在英文逗号
     * @param attribute
     * @return
     */
    protected abstract String toDBAttributeString(E attribute);

    private class ToDBStringFunction implements Function<E,String>{
        @Override
        public String apply(E e) {
            return toDBAttributeString(e);
        }
    }

    private class ToSystemObjectFunction implements Function<String,E>{
        @Override
        public E apply(String s) {
            return toSystemObject(s);
        }
    }
}
