package com.biz.eisp.base.core.dao;

import com.biz.eisp.base.common.exception.BusinessException;
import com.biz.eisp.base.common.util.JdbcProxy;
import com.biz.eisp.base.common.util.StringUtil;
import com.biz.eisp.base.utils.Constants;
import com.biz.eisp.base.utils.NumberUtils;
import com.biz.eisp.page.Page;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.*;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.metadata.ClassMetadata;
import org.springframework.orm.hibernate4.SessionFactoryUtils;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ReflectionUtils;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.sql.*;
import java.util.*;




/**
 * DAO基类
 * @author liukai
 *
 */
public class HibernateBaseDao {
	
	public static final Log log = LogFactory.getLog("HibernateBaseDao");
	
	private SessionFactory sessionFactory;
	
	public void setSessionFactory(SessionFactory sessionFactory) {
		this.sessionFactory = sessionFactory;
	}

	public SessionFactory getSessionFactory() {
		return sessionFactory;
	}

	public Session getSession() {
		return sessionFactory.getCurrentSession();
	}
	
	/**
	 * 新增或修改一个对象
	 * @param object
	 */
	@Transactional
	public void saveOrUpdate(Object object) {
		getSession().saveOrUpdate(object);
	}
	
	/**
	 * 新增或修改一个对象
	 * @param object
	 */
	@Transactional
	public String save(Object object) {
		return (String) getSession().save(object);
	}
	
	
	/**
	 * 新增或修改集合中的全部对象
	 * @param collection
	 */
	@Transactional
	public void saveAll(Collection collection) {
		for (Iterator it = collection.iterator(); it.hasNext();) {
			save(it.next());
		}
	}
	

	
	/**
	 * 删除一个对象
	 * @param object
	 */
	@Transactional
	public void delete(Object object) {
		getSession().delete(object);
	}
	/**
	 * 删除HQL查询到的全部对象
	 * @param hql
	 * @param values
	 */
	@Transactional
	public void deleteAll(String hql, Object... values) {
		deleteAll(find(hql, values));
	}
	/**
	 * 删除集合中的全部对象
	 * @param collection
	 */
	@Transactional
	public void deleteAll(Collection collection) {
		for (Iterator it = collection.iterator(); it.hasNext();) {
			delete(it.next());
		}
	}
	
	/**
	 * 根据ID删除一个对象
	 * @param <T>
	 * @param entityClass
	 * @param id
	 */
	@Transactional
	public <T> void deleteById(Class<T> entityClass, Serializable id) {
		delete(get(entityClass, id));
	}


	
	/**
	 * 根据ID获取一个对象，如果查不到返回null
	 * @param <T>
	 * @param entityClass
	 * @param id
	 * @return
	 */
	public <T> T get(Class<T> entityClass, Serializable id) {
		return (T) getSession().get(entityClass, id);
	}
	/**
	 * 
	 * @Title load  
	 * @Description   
	 * @param @param <T>
	 * @param @param entityClass
	 * @param @param id
	 * @param @return
	 * @return T 
	 * @throws  
	 * @author lvq  
	 * @date Feb 7, 2014 11:09:23 AM
	 */
	public <T> T load(Class<T> entityClass, Serializable id) {
		return (T) getSession().load(entityClass, id);
	}
	/**
	 * 根据ID获取一个对象，如果查不到返回null
	 * @param <T>
	 * @param entityClass
	 * @param id
	 * @param mode 锁模式
	 * @return
	 */
	public <T> T get(Class<T> entityClass, Serializable id, LockMode mode) {
		return (T) getSession().get(entityClass, id, mode);
	}
	
	/**
	 * 获取全部的对象
	 * @param <T>
	 * @param entityClass
	 * @return
	 */
	public <T> List<T> getAll(Class<T> entityClass) {
		return createCriteria(entityClass).list();
	}
	
	/**
	 * 获取全部的对象并指定排序方式
	 * @param <T>
	 * @param entityClass
	 * @param orderBy 排序表达式，如："empName desc"
	 * @return
	 */
	public <T> List<T> getAll(Class<T> entityClass, String orderBy) {
		return find("from " + entityClass.getSimpleName() + " order by " + orderBy);
	}
	
	/**
	 * HQL查询
	 * @param queryString HQL语句
	 * @param values HQL参数
	 * @return
	 */
	public List find(String queryString, Object... values) {
		return createQuery(queryString, values).list();
	}
	
	/**
	 * HQL查询
	 * @param queryString
	 * @param mode 锁模式
	 * @param alise
	 * @param values
	 * @return
	 */
	public List find(String queryString, LockMode mode, String alise, Object... values) {
		return createQuery(queryString, mode, alise, values).list();
	}
	
	/**
	 * HQL查询
	 * @param <T>
	 * @param entityClass 查询结果的类型
	 * @param queryString HQL语句
	 * @param values HQL参数
	 * @return
	 */
	public <T> List<T> findEntity(Class<T> entityClass, String queryString, Object... values) {
		return find(queryString, values);
	}
	
	/**
	 * HQL查询一个整数
	 * @param queryString HQL语句
	 * @param values HQL参数
	 * @return 
	 */
	public Integer findInt(String queryString, Object... values) {
		return (Integer) findUnique(queryString, values);
	}

	/**
	 * HQL查询一个长整数
	 * @param queryString HQL语句
	 * @param values HQL参数
	 * @return 
	 */
	public Long findLong(String queryString, Object... values) {
		return (Long) findUnique(queryString, values);
	}

	/**
	 * HQL唯一查询，仅返回第一个对象
	 * @param queryString HQL语句
	 * @param values HQL参数
	 * @return 
	 */
	public Object findUnique(String queryString, Object... values) {
		return createQuery(queryString, values).setMaxResults(1).uniqueResult();
	}

	/**
	 * HQL唯一查询，仅返回第一个对象
	 * @param entityClass 查询结果的类型
	 * @param queryString HQL语句
	 * @param values HQL参数
	 * @return 
	 */
	public <T> T findUniqueEntity(Class<T> entityClass, String queryString, Object... values) {
		return (T) findUnique(queryString, values);
	}

	public Object sqlQueryForObject(String sql, Object ... params) {
		Query query = this.getSession().createSQLQuery(sql);
		this.addParams(query, params);
		return query.uniqueResult();
	}
	
	private void addParams(Query query,Object... params){
		if(params!=null&&params.length>0){
			for(int i=0;i<params.length;i++){
				query.setParameter(i, params[i]);
			}
		}
	}
	
	/**
	 * HQL分面查询
	 * @param <T>
	 * @param entityClass 查询结果的类型
	 * @param page page对象
	 * @param queryString HQL语句
	 * @param values HQL参数
	 * @return
	 */
	public <T> List<T> findPage(Class<T> entityClass, Page page, String queryString, Object... values) {
		int pageStart = page.getInt(Page.PAGE);
		int pageSize = page.getInt(Page.ROWS);
		
//		page.put(Page.TOTAL_COUNT, String.valueOf(this.findLong("select count(*) " + queryString.substring(queryString.indexOf("from")))));
//		page.compute();
		
		Query query = createQuery(queryString, values);
		if (page.isPaging()) {
			ScrollableResults scrollableResults = query.scroll(ScrollMode.SCROLL_SENSITIVE);
			scrollableResults.last();
			page.put(Page.TOTAL, String.valueOf(scrollableResults.getRowNumber() + 1));
			page.compute();
			query.setFirstResult(pageStart);
			query.setMaxResults(pageSize);
		}
		
		return query.list();
	}
	
	/**
	 * 查询一个对象的属性值在数据库中是不是唯一
	 * @param entity 实体对象
	 * @param propertyNames 属性的名称，可多个 如："prop1,prop2"
	 * @return
	 */
	public boolean isUnique(Object entity, String propertyNames) {
		Class clazz = getTrueClass(entity);
		Criteria criteria = createCriteria(clazz).setProjection(Projections.rowCount());
		String[] nameList = StringUtil.split(propertyNames, ",");
		try {
			boolean isQuery = false;
			for (String name : nameList) {
				Object obj = PropertyUtils.getProperty(entity, name);
				if (obj != null) {
					criteria.add(Restrictions.eq(name, obj));
					isQuery = true;
				}
				else {
					isQuery = false;
				}
			}
			if (!isQuery) {
				return true;
			}
			String idName = getIdName(clazz);
			Serializable id = getId(clazz, entity);
			if (id != null) {
				criteria.add(Restrictions.not(Restrictions.eq(idName, id)));
			}
		} 
		catch (Exception e) {
			ReflectionUtils.handleReflectionException(e);
		}
		return (Integer) criteria.uniqueResult() == 0;
	}
	
	public <T> List<T> findByCriteria(Class<T> entityClass, Criterion... criterion) {
		return createCriteria(entityClass, criterion).list();
	}


	public Query createQuery(String queryString, Object... values) {
		Query query = getSession().createQuery(queryString);
		for (int i = 0; i < values.length; i++) {
			query.setParameter(i, values[i]);
		}
		return query;
	}
	
	public Query createQuery(String queryString, LockMode mode, String alise, Object... values) {
		Query query = getSession().createQuery(queryString);
		query.setLockMode(alise, mode);
		for (int i = 0; i < values.length; i++) {
			query.setParameter(i, values[i]);
		}
		return query;
	}
	
	public Criteria createCriteria(Class entityClass, Criterion... criterions) {
		Criteria criteria = getSession().createCriteria(entityClass);
		for (Criterion c : criterions) {
			criteria.add(c);
		}
		return criteria;
	}
	
	public void flush() {
		getSession().flush();
	}
	
	public void clear() {
		getSession().clear();
	}
	
	public void evict(Object object) {
		getSession().evict(object);
	}
	
	

	public Serializable getId(Class entityClass, Object entity) throws NoSuchMethodException, IllegalAccessException,
			InvocationTargetException {
		return (Serializable) PropertyUtils.getProperty(entity, getIdName(entityClass));
	}
	
	public String getIdName(Class entityClass) {
		ClassMetadata meta = getSessionFactory().getClassMetadata(entityClass);
		return meta.getIdentifierPropertyName();
	}
	
	public Class getTrueClass(Object object) {
		Class clazz = object.getClass();
		String className = clazz.getName();
		int index = className.indexOf("$");
		if (index != -1) {
			className = className.substring(0, index);
			try {
				clazz = Class.forName(className);
			}
			catch(ClassNotFoundException ex) {
				ReflectionUtils.handleReflectionException(ex);
			}
		}
		return clazz;
	}
	
	/**
	 * HQL查询，会通过Page对象动态组装HQL语句，Action黙认调用的方法
	 * @param <T>
	 * @param entityClass 查询结果的类型
	 * @param page page对象
	 * @return
	 */
	public <T> List<T> query(Class<T> entityClass, Page page) {
		return this.query(entityClass, page, page.get(Page.AUTH_PREFIX));
	}
	
	/**
	 * HQL查询，会通过Page对象动态组装HQL语句，Action黙认调用的方法
	 * @param <T>
	 * @param entityClass 查询结果的类型
	 * @param page page对象
	 * @param authPrefix 授权对象的路径表达表，如果授权对象（baseOrg或baseEmp）不在当前对象中，则需要指定授权对象的路径名称，
	 *   如：有如下结构currObj.obj1.obj2，授权对象在obj2中，则表达式应为"obj1_obj2"，授权对象在obj1中，则表达式应为"obj1"
	 * @return
	 */
	public <T> List<T> query(Class<T> entityClass, Page page, String authPrefix) {
		StringBuilder sb = new StringBuilder("from " + entityClass.getSimpleName() + " where 1=1");
		sb.append(makeQueryConditions(entityClass, page, "", authPrefix));
		sb.append(makeQuerySort(entityClass, page, ""));
		
	   	return findPage(entityClass, page, sb.toString());
	}
	
	public void initAuthExp(Class entityClass, Page page, String expression, String authPrefix) {
		String entityName = entityClass.getSimpleName();
		String[] exps = StringUtil.split(expression, ";");
		String keys = "";
		String values = "";
		for (String exp : exps) {

			if (entityName.indexOf("BaseOrg") == 0 && exp.indexOf("baseEmployee") != -1) {
				continue;
			}

			String[] kv = StringUtil.split(exp, "=");
			String key = kv[0];
			String value = page.get(kv[1]);
			
			if (StringUtil.isNotBlank(authPrefix)) {
				int p = key.indexOf("_");
				key = key.substring(0, p + 1) + authPrefix + key.substring(p);
			}

			if (StringUtil.isNotBlank(value)) {
				String[] props = StringUtil.split(key, "_");
				if (props[1].toUpperCase().equals(entityName.toUpperCase())) {
					keys += StringUtil.replace(key, "_" + props[1], "") + " ;";
					values += value + ";";
				}
				else {
					try {
						Class c = entityClass;
						for (int i = 1; i < props.length; i++) {
							c = c.getDeclaredField(props[i]).getType();
						}
						keys +=  key + " ;";
						values += value + ";";
					}
					catch (Exception e) {}
				}
			}
			
		}
		if (!"".equals(keys)) {
			page.put(keys, values);
		}
	}
	
	/**
	 * 生成HQL where语句
	 * @param entityClass
	 * @param page
	 * @param alias 主表别称
	 * @return
	 */
	public String makeQueryConditions(Class entityClass, Page page, String alias) {
		return makeQueryConditions(entityClass, page, alias, null);
	}
	
	/**
	 * 生成HQL where语句
	 * @param entityClass
	 * @param page
	 * @param alias 当前所查询对象的别称
	 * @param authPrefix 授权对象的路径表达表，如果授权对象（baseOrg或baseEmp）不在当前对象中，则需要指定授权对象的路径名称，
	 *   如：有如下结构currObj.obj1.obj2，授权对象在obj2中，则表达式应为"obj1_obj2"，授权对象在obj1中，则表达式应为"obj1"
	 * @return
	 */
	public String makeQueryConditions(Class entityClass, Page page, String alias, String authPrefix) {
		String expressions = page.get(Page.DATA_AUTH);
		if (StringUtil.isNotBlank(expressions)) {
			String[] expArray = StringUtil.split(expressions, ",");
			for (String expression : expArray) {
				initAuthExp(entityClass, page, expression, authPrefix);
			}
		}

		alias = StringUtil.isNotBlank(alias) ? alias + "." : "";
		StringBuilder conditions = new StringBuilder(200);
		Set<String> keys = page.keySet();
		for (String key : keys) {
			String[] val = StringUtil.split(page.get(key), ";");
			if (val == null) {
				continue;
			}
			String[] exp = StringUtil.split(key, ";");
			StringBuilder cond = new StringBuilder(50);
			for (int i = 0; i < exp.length && i < val.length; i++) {
				if (exp[i].indexOf("$") == 0 && StringUtil.isNotEmpty(val[i])) {
					String prefix = exp[i].substring(0, exp[i].indexOf("_"));
					if (Constants.QUERY_EXP.containsKey(prefix)) {
						String name = alias + exp[i].substring(prefix.length() + 1).replaceAll("_", ".");
						String queryExp = Constants.QUERY_EXP.get(prefix);
						if (Constants.PATTERN_DATE.matcher(val[i]).matches()) {
//							name = "to_char(" + name + ",'yyyy-mm-dd')";
							if (prefix.equals("$lt") || prefix.equals("$le")) {
								val[i] = val[i] + " 23:59:59";
							}
						}
						if ("$in".equals(prefix)) {
							String v = "'" + StringUtil.replace(val[i], ",", "','") + "'";
							if (cond.length() == 0) {
								cond.append(" and (").append(name).append(StringUtil.replace(queryExp, "?", v));
							}
							else {
								cond.append(" or ").append(name).append(StringUtil.replace(queryExp, "?", v));
							}
						}
						else {
							String[] values = StringUtil.split(val[i], ",");
							for (String v : values) {
								if (cond.length() == 0) {
									cond.append(" and (").append(name).append(StringUtil.replace(queryExp, "?", v));
								}
								else {
									cond.append(" or ").append(name).append(StringUtil.replace(queryExp, "?", v));
								}
							}
						}
					}
				}
			}
			if (cond.length() > 0) {
				conditions.append(cond.append(")"));
			}
		}
		return conditions.toString();
	}
	
	/**
	 * 生成HQL order语句
	 * @param entityClass
	 * @param page
	 * @param alias 当前所查询对象的别称
	 * @return
	 */
	public String makeQuerySort(Class entityClass, Page page, String alias) {
		if (StringUtil.isNotBlank(page.get(Page.SORT))) {
			alias = StringUtil.isNotBlank(alias) ? alias + "." : "";
	   		return " order by " + alias + page.get(Page.SORT) + " " + page.get(Page.DIR);
	   	}
		return "";
	}
	
	public void makeParentInfo(Class entityClass, Object object, String descName) {
		String parentName = entityClass.getSimpleName();
		parentName = parentName.substring(0,1).toLowerCase() + parentName.substring(1);
		String idName = getIdName(getTrueClass(object));
		
		try {
			Long levelCode = (Long) PropertyUtils.getProperty(object, "levelCode");
			
			while (true) {
				String desc = StringUtil.toString(PropertyUtils.getProperty(object, descName));
				String currDesc = StringUtil.toString(PropertyUtils.getProperty(object, "pn" + levelCode));
				if (!desc.equals(currDesc)) {
					break;
				}
				
				String id = StringUtil.toString(PropertyUtils.getProperty(object, idName));
				String currId = StringUtil.toString(PropertyUtils.getProperty(object, "pi" + levelCode));
				if (!id.equals(currId)) {
					break;
				}

				String parentId = "";
				Object parent = PropertyUtils.getProperty(object, parentName);
				if (parent != null) {
					parentId = StringUtil.toString(PropertyUtils.getProperty(parent, idName));
				}
				String currParnetId = "";
				if (levelCode > 1) {
					currParnetId = StringUtil.toString(PropertyUtils.getProperty(object, "pi" + (levelCode-1)));
				}
				if (!parentId.equals(currParnetId)) {
					break;
				}
				
				return;
			}
			
			for (int i = 1; i <= 8; i++) {
				PropertyUtils.setProperty(object, "pi" + i, null);
				PropertyUtils.setProperty(object, "pn" + i, null);
			}
			this.updateParentInfo(entityClass, object, descName);
		}
		catch(Exception e) {
			throw new BusinessException(e.getMessage());
		}
		
	}
	
	private void updateParentInfo(Class entityClass, Object object, String descName) {
		String parentName = entityClass.getSimpleName();
		parentName = parentName.substring(0,1).toLowerCase() + parentName.substring(1);
		Object parent = object;
		StringBuilder pnall = new StringBuilder();
		try {
			while (parent != null) {
				Object id = PropertyUtils.getProperty(parent, getIdName(getTrueClass(object)));
				Object desc = PropertyUtils.getProperty(parent, descName);
				String levelCode = PropertyUtils.getProperty(parent, "levelCode") + "";
				
				PropertyUtils.setProperty(object, "pi" + levelCode, id);
				PropertyUtils.setProperty(object, "pn" + levelCode, desc);
//				if (!"1".equals(levelCode) || pnall.length() == 0) {
					if (pnall.length() > 0) {
						pnall.insert(0, "-");
					}
					pnall.insert(0, desc);
//				}
				
				parent = PropertyUtils.getProperty(parent, parentName);
			}
			
			PropertyUtils.setProperty(object, "pnall", pnall.toString());
			
			
			Iterator it = ((Set) PropertyUtils.getProperty(object, parentName + "s")).iterator();
			while (it.hasNext()) {
				updateParentInfo(entityClass, it.next(), descName);
				
			}
			
		}
		catch(Exception e) {
			throw new BusinessException(e.getMessage());
		}
	}
	
	
	public void makeSubCode(Class entityClass, Object object) {
		makeSubCode(entityClass, object, "");
	}
	
	public void makeSubCode(Class entityClass, Object object, String whereExp) {
		String parentName = entityClass.getSimpleName();
		parentName = parentName.substring(0,1).toLowerCase() + parentName.substring(1);

		String parentSubCode = "";
		Long parentLevelCode = 0L;
		
		try {
			Object parent = PropertyUtils.getProperty(object, parentName);
			if (parent != null) {
				parentSubCode = (String) PropertyUtils.getProperty(parent, "subCode");
				parentLevelCode = (Long) PropertyUtils.getProperty(parent, "levelCode");
			}

			Object id = PropertyUtils.getProperty(object, getIdName(getTrueClass(object)));
			if (id == null) {
				PropertyUtils.setProperty(object, "subCode", getValidSubCode(getTrueClass(object), parentSubCode, whereExp));
				PropertyUtils.setProperty(object, "levelCode", parentLevelCode + 1);
			}
			else {
				String subCode = (String) PropertyUtils.getProperty(object, "subCode");
				Long levelCode = (Long) PropertyUtils.getProperty(object, "levelCode");
				if (!parentSubCode.equals("") && parentSubCode.indexOf(subCode) == 0) {
					throw new Exception("不能把自己或自已的下级改变为自己的上级");
				}
				if (subCode == null || levelCode == null || subCode.equals("") || levelCode == 0 || subCode.indexOf(parentSubCode) != 0 || levelCode != (parentLevelCode + 1)) {
					updateSubCode(entityClass, object, whereExp);
				}
			}
		}
		catch(Exception e) {
			throw new BusinessException(e.getMessage());
		}
		
	}
	
	public void makeFullPath(Class entityClass, Object object) {
		makeFullPath(entityClass, object, "");
	}
	
	public void makeFullPath(Class entityClass, Object object, String whereExp) {
		String parentName = entityClass.getSimpleName();
		parentName = parentName.substring(0,1).toLowerCase() + parentName.substring(1);

		String parentFullPath = "";
		Long parentPath = 0L;
		
		try {
			Object parent = PropertyUtils.getProperty(object, parentName);
			if (parent != null) {
				parentFullPath = (String) PropertyUtils.getProperty(parent, "fullPath");
//				parentLevelCode = (Long) PropertyUtils.getProperty(parent, "levelCode");
			}

			Object id = PropertyUtils.getProperty(object, getIdName(getTrueClass(object)));
			if (id == null) {
				PropertyUtils.setProperty(object, "fullPath", getValidSubCode(getTrueClass(object), parentFullPath, whereExp));
//				PropertyUtils.setProperty(object, "levelCode", parentLevelCode + 1);
			}
			else {
//				String subCode = (String) PropertyUtils.getProperty(object, "subCode");
//				Long levelCode = (Long) PropertyUtils.getProperty(object, "levelCode");
//				if (!parentSubCode.equals("") && parentSubCode.indexOf(subCode) == 0) {
//					throw new Exception("不能把自己或自已的下级改变为自己的上级");
//				}
//				if (subCode == null || levelCode == null || subCode.equals("") || levelCode == 0 || subCode.indexOf(parentSubCode) != 0 || levelCode != (parentLevelCode + 1)) {
//					updateSubCode(entityClass, object, whereExp);
//				}
			}
		}
		catch(Exception e) {
			throw new BusinessException(e.getMessage());
		}
	}
	
	public void updateSubCode(Class entityClass, Object object) {
		updateSubCode(entityClass, object, "");
	}
	
	public void updateSubCode(Class entityClass, Object object, String whereExp) {
		String parentName = entityClass.getSimpleName();
		parentName = parentName.substring(0,1).toLowerCase() + parentName.substring(1);
		String parentSubCode = "";
		Long parentLevelCode = 0L;
		
		try {
			Object parent = PropertyUtils.getProperty(object, parentName);
			if (parent != null) {
				parentSubCode = (String) PropertyUtils.getProperty(parent, "subCode");
				parentLevelCode = (Long) PropertyUtils.getProperty(parent, "levelCode");
			}
			PropertyUtils.setProperty(object, "subCode", getValidSubCode(getTrueClass(object), parentSubCode, whereExp));
			PropertyUtils.setProperty(object, "levelCode", parentLevelCode + 1);
			save(object);
			
			Iterator it = ((Set) PropertyUtils.getProperty(object, parentName + "s")).iterator();
			while (it.hasNext()) {
				updateSubCode(entityClass, it.next(), whereExp);
			}

		}
		catch(Exception e) {
			throw new BusinessException(e.getMessage());
		}
		
	}
	
	public String getValidSubCode(Class entityClass, String parentSubCode) {
		return getValidSubCode(entityClass, parentSubCode, "");
	}
	
	public String getValidSubCode(Class entityClass, String parentSubCode, String whereExp) {
		if (StringUtil.isNotBlank(whereExp)) {
			whereExp = "and " + whereExp;
		}
		else {
			whereExp = "";
		}
		String queryString = "select subCode from " + entityClass.getSimpleName() + " where subCode like '" + parentSubCode + "___' " + whereExp + " order by subCode";
		Iterator it = this.find(queryString).iterator();
		String code = null;
		for (int index = 0; index < 999; index++) {
			if  (it.hasNext()) {
				String c = ((String) it.next());
				c = c.substring(c.length() - 3);
				
				if (index != Integer.parseInt(c)) {
					code = NumberUtils.format(index, "000");
					break;
				}
			}
			else {
				code = NumberUtils.format(index, "000");
				break;
			}
		}
		return parentSubCode + code;
	}
	
	
	
	public String getCode(Class entityClass, String alias){
//		String className = entityClass.getSimpleName();
//		String entityCode = className.substring(0, 1).toLowerCase() + className.substring(1) + "Code";
//		if (StringUtil.isBlank(alias)) {
//			alias = "";
//		}
//		else {
//			alias = alias + "-";
//		}
//		String hql = "select " + entityCode + " from " + entityClass  + " where " + entityCode + " like " + alias + "% order by " + entityCode + " desc limit 1";
//		String code = (String)this.findUnique(hql);
//		if(code == null){
//			return 1;
//		}
//		code = code.substring(0, 4);
//		
	
		return null;
	}
	
	
	
	
	
	
	
	
	
	
	
	
	
	/**
	 * 生成SQL授权where语句by组织
	 * @param page
	 * @param alias 组织授权表的别称
	 * @return
	 */
	public String makeSqlAuthExpByOrg(Page page, String alias) {
		if (StringUtil.isBlank(alias)) {
			alias = "";
		}
		else {
			alias = alias + ".";
		}
		StringBuilder exp = new StringBuilder("");
		
		int authLevel = Integer.parseInt(page.get(Page.DATA_AUTH_LEVEL));
		if (authLevel == 2) {
		}
		else if (authLevel == 3) {
		}
		else if (authLevel == 4) {
			String[] codes = page.get(Page.ORG_SUBCODE_KEY).split("\\,");
			for (String code : codes) {
				if (exp.length() == 0) {
					exp.append(" and (");
				}
				else {
					exp.append(" or ");
				}
				exp.append(alias).append("sub_code like '").append(code).append("%'");
			}
			exp.append(")");
		}
		else if (authLevel == 5) {
			exp.append(" and ").append(alias).append("org_id in (").append(page.get(Page.ORG_ID_KEY)).append(")");
		}
		else if (authLevel == 99) {
			if (StringUtil.isNotBlank(page.get(Page.ORG_AUTH_KEY))) {
				exp.append(" and ").append(alias).append("org_id in (").append(page.get(Page.ORG_AUTH_KEY)).append(")");
			}
		}
		return exp.toString();
		
	}
	
	/**
	 * 生成SQL授权where语句by人员
	 * @param page
	 * @param alias 人员授权表的别称
	 * @return
	 */
	public String makeSqlAuthExpByEmp(Page page, String alias) {
		if (StringUtil.isBlank(alias)) {
			alias = "";
		}
		else {
			alias = alias + ".";
		}
		String exp = "";
		int authLevel = Integer.parseInt(page.get(Page.DATA_AUTH_LEVEL));
		if (authLevel == 2) {
			exp += " and " + alias + "sub_code like '" + page.get(Page.PARENT_EMP_SUBCODE_KEY) + "%'";
		}
		else if (authLevel == 3) {
			exp += " and " + alias + "level_code>=" + page.get(Page.EMP_LEVELCODE_KEY);
		}
		else if (authLevel == 4) {
			exp += " and " + alias + "sub_code like '" + page.get(Page.EMP_SUBCODE_KEY) + "%'";
		}
		else if (authLevel == 5) {
			exp += " and " + alias + "emp_id=" + page.get(Page.EMP_ID_KEY);
		}
		else if (authLevel == 99) {
			if (StringUtil.isNotBlank(page.get(Page.EMP_AUTH_KEY))) {
				exp += " and " + alias + "emp_id in (" + page.get(Page.EMP_AUTH_KEY) + ")";
			}
		}
		return exp;
	}
	
	/**
	 * 生成SQL授权where语句by产品
	 * @param page
	 * @param alias 产品授权表的别称
	 * @return
	 */
	public String makeSqlAuthExpByProd(Page page, String alias) {
		return null;
	}
	
	/**
	 * 生成SQL order语句
	 * @param page
	 * @param alias 当前所查询对象的别称
	 * @return
	 */
	public String makeSqlSortExp(Page page, String alias) {
		return this.makeQuerySort(null, page, alias);
	}
	
	/**
	 * 执行SQL查询，返加一个Object[]类型的List
	 * @param sql
	 * @return List<Object[]>
	 */
	public List<Object[]> executeSqlQuery(String sql) {
		return (List<Object[]>)getSqlList(null, sql);
	}
	
	public List<Map> executeSqlQueryMap(String sql){
		return (List<Map>)getSqlListMap(null, sql);
	}
	
	/**
	 * 执行SQL查询，返加一个指定类型entityClass的List
	 * @param sql
	 * @return List<T>
	 */
	public <T> List<T> executeSqlQuery(Class<T> entityClass, String sql) {
		return getSqlList(entityClass, sql);
	}
	
	/**
	 * 执行唯一记录的SQL查询，返加一个Object[]类型的对象
	 * @param sql
	 * @return Object[]
	 */
	public Object[] executeUniqueSqlQuery(String sql) {
		Object[] object = null;
    	Iterator<Object[]> it = executeSqlQuery(sql).iterator();
    	if (it.hasNext()) {
    		object = it.next();
    	}
        return object;
	}
	
	/**
	 * 执行唯一记录的SQL查询，返加一个Map类型的对象
	 * @param sql
	 * @return Map
	 */
	public Map executeUniqueSqlQueryMap(String sql) {
		Map map = null;
		Iterator <Map> it = this.executeSqlQueryMap(sql).iterator();
		if(it.hasNext()) {
			map = it.next();
		}
		return map;
	}
	
	/**
	 * 执行唯一记录的SQL查询，返加一个指定类型entityClass的对象
	 * @param entityClass
	 * @param sql
	 * @return T
	 */
	public <T> T executeUniqueSqlQuery(Class<T> entityClass, String sql) {
		T object = null;
    	Iterator<T> it = executeSqlQuery(entityClass, sql).iterator();
    	if (it.hasNext()) {
    		object = it.next();
    	}
        return object;
	}
	
	/**
	 * 执行分页的SQL查询，返加一个Object[]类型的List
	 * @param sql
	 * @param page
	 * @return List<Object[]>
	 */
	public List<Object[]> executeSqlQuery(String sql, Page page) {
		return (List<Object[]>)getSqlList(null, sql, page);
	}

	/**
	 * 执行分页的SQL查询，返加一个Map类型的List
	 * @param sql
	 * @param page
	 * @return List<Map>
	 */
	public List<Map> executeSqlQueryMap(String sql, Page page){
		return (List<Map>)getSqlListMap(null, sql, page);
	}
	
	/**
	 * 执行分页的SQL查询，返加一个指定类型entityClass的List
	 * @param sql
	 * @param page
	 * @return List<T>
	 */
	public <T> List<T> executeSqlQuery(Class<T> entityClass, String sql, Page page) {
		return getSqlList(entityClass, sql, page);
	}
	
	/**
	 * 执行一个SQL，update或insert
	 * @param sql
	 * @return update或insert的记录数
	 */
	public int executeSqlUpdate(String sql) {
		int n = 0;
		Statement stat = null;
		try {
			stat = SessionFactoryUtils.getDataSource(sessionFactory).getConnection().createStatement();
			n = stat.executeUpdate(sql);
		} 
		catch (SQLException e) {
			throw new BusinessException(e.getMessage() + ":" + sql);
		}
		finally {
			try {
				if (stat != null) {
					stat.close();
				}
			}
			catch(SQLException e) {}
		}
		return n;
	}
	
	public void executeCall(String procedure) {
		CallableStatement stat = null;
		try {
			stat = SessionFactoryUtils.getDataSource(sessionFactory).getConnection().prepareCall(procedure);
			stat.executeUpdate();
		} 
		catch (SQLException e) {
			throw new BusinessException(e.getMessage() + ":" + procedure);
		}
		finally {
			try {
				if (stat != null) {
					stat.close();
				}
			}
			catch(SQLException e) {}
		}
	}
	
	private <T> List getSqlList(Class<T> entityClass, String sql) {
		List list = null;
		Statement stat = null;
		ResultSet rs = null;
		try {
			stat = SessionFactoryUtils.getDataSource(sessionFactory).getConnection().createStatement();
			rs = stat.executeQuery(sql);
			if (entityClass == null) {
				list = JdbcProxy.getObjectArrayList(rs);
			}
			else {
				list = JdbcProxy.getEntityList(rs, entityClass);
			}
			
		}
		catch (SQLException e) {
			throw new BusinessException(e.getMessage() + ":" + sql);
		}
		finally {
			if (rs != null) {
				try {
					rs.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
			}
			if (stat != null) {
				try {
					stat.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
			}
		}
		return list;
	}
	
	private <T> List getSqlList(Class<T> entityClass, String sql, Page page) {
		List list = null;
		int pageStart = page.getInt(Page.PAGE);
		int pageSize = page.getInt(Page.ROWS);
		Statement stat = null;
		ResultSet rs = null;
		try {
			stat = SessionFactoryUtils.getDataSource(sessionFactory).getConnection().createStatement(ResultSet.TYPE_SCROLL_SENSITIVE, ResultSet.CONCUR_READ_ONLY);
			rs = stat.executeQuery(sql);
			rs.last();
			page.put(Page.TOTAL, String.valueOf(rs.getRow()));
			page.compute();
			if (pageStart == 0) {
				rs.beforeFirst();
			}
			else {
				rs.absolute(pageStart);
			}
			
			if (entityClass == null) {
				list = JdbcProxy.getObjectArrayList(rs, page.getLong(Page.ROWS));
			}
			else {
				list = JdbcProxy.getEntityList(rs, page.getLong(Page.ROWS), entityClass);
			}
		}
		catch (SQLException e) {
			throw new BusinessException(e.getMessage() + ":" + sql);
		}
		finally {
			if (rs != null) {
				try {
					rs.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
			}
			if (stat != null) {
				try {
					stat.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
			}
		}
		return list;
	}
	
	private <T> List getSqlListMap(Class<T> entityClass, String sql) {
		List list = null;
		Statement stat = null;
		ResultSet rs = null;
		try {
			stat = SessionFactoryUtils.getDataSource(sessionFactory).getConnection().createStatement();
			rs = stat.executeQuery(sql);
			if (entityClass == null) {
				list = JdbcProxy.getObjectMapList(rs);
			}
			else {
				list = JdbcProxy.getEntityList(rs, entityClass);
			}
		}
		catch (SQLException e) {
			throw new BusinessException(e.getMessage() + ":" + sql);
		}
		finally {
			if (rs != null) {
				try {
					rs.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
			}
			if (stat != null) {
				try {
					stat.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
			}
		}
		return list;
	}
	
	private <T> List getSqlListMap(Class<T> entityClass, String sql, Page page) {
		List list = null;
		int pageSize = page.getInt(Page.ROWS);
		int pageStart = (page.getInt(Page.PAGE) - 1) * pageSize;
		Statement stat = null;
		ResultSet rs = null;
		try {
			stat = SessionFactoryUtils.getDataSource(sessionFactory).getConnection().createStatement(ResultSet.TYPE_SCROLL_SENSITIVE, ResultSet.CONCUR_READ_ONLY);
			rs = stat.executeQuery(sql);
			rs.last();
			page.put(Page.TOTAL, String.valueOf(rs.getRow()));
			page.compute();
			if (pageStart == 0) {
				rs.beforeFirst();
			}
			else {
				rs.absolute(pageStart);
			}
			if (entityClass == null) {
				list = JdbcProxy.getObjectMapList(rs, page.getLong(Page.ROWS));
			}
			else {
				list = JdbcProxy.getEntityList(rs, page.getLong(Page.ROWS), entityClass);
			}
		}
		catch (SQLException e) {
			throw new BusinessException(e.getMessage() + ":" + sql);
		}
		finally {
			if (rs != null) {
				try {
					rs.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
			}
			if (stat != null) {
				try {
					stat.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
			}
		}
		return list;
	}

	/**
	 * 查询报表平台中的存储过程值
	 * @param proc
	 * @param params
	 * @param page
	 * @return
	 */
	protected Map getProcValue(String procName, Map procParamValue, Page page, boolean isShowSql) {
		int pageStart = page.getInt(Page.PAGE);
		CallableStatement cstat = null;
		ResultSet rs = null;
		List list = null;
		Map procValue = new HashMap();
		Map otherData = new HashMap();
		try{
			Connection connection = SessionFactoryUtils.getDataSource(sessionFactory).getConnection();
			ResultSet procedureColumns = connection.getMetaData().getProcedureColumns(connection.getCatalog(), connection.getCatalog(), procName, null);
			String callValue = getCallValue(procedureColumns, procName, procParamValue, isShowSql);
			cstat = SessionFactoryUtils.getDataSource(sessionFactory).getConnection().prepareCall(callValue);
			setProcParamValue(cstat, procedureColumns, procParamValue);
			rs = cstat.executeQuery();
			rs.last();
			page.put(Page.TOTAL, String.valueOf(rs.getRow()));
			page.compute();
			if (pageStart == 0) {
				rs.beforeFirst();
			}
			else {
				rs.absolute(pageStart);
			}
			list = JdbcProxy.getObjectMapList(rs, page.getLong(Page.PAGE));
			setOutParameterValue(cstat, procedureColumns, otherData);
			procValue.put("dataList", list);
			procValue.put("otherData", otherData);
		} catch(Exception e) {
			e.printStackTrace();
		} finally {
			if (rs != null) {
				try {
					rs.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
			}
			if (cstat != null) {
				try {
					cstat.close();
				} catch (SQLException e) {
					e.printStackTrace();
				}
			}
		}
		return procValue;
	}

	//获取执行存储过程语句
	protected String getCallValue(ResultSet procedureColumns, String procName, Map procParamValue, boolean isShowSql) {
		StringBuilder callValue = new StringBuilder("{call "+ procName+"(");
		StringBuilder callLogValue = new StringBuilder("{call "+ procName+"(");
		try {
			boolean flag = procedureColumns.next();
			while(flag) {
				String columnName = procedureColumns.getString("COLUMN_NAME");
				short columnType = procedureColumns.getShort("COLUMN_TYPE");
				callValue.append("?,");
				if(columnType == 1) {
					callLogValue.append("\"" + procParamValue.get(columnName) + "\",");
				} else {
					callLogValue.append("@" + columnName + ",");
				}
				flag = procedureColumns.next();
			}
		} catch (Exception e) {
			throw new BusinessException("构造存储过程call语句失败\n" + e.getMessage());
		}
		callValue.append("&)}");
		callLogValue.append("&)}");
		if(isShowSql) {
			log.info("\n\n调用语句-->" + callLogValue.toString().replace(",&)}", ")}\n"));
		}
		return callValue.toString().replace(",&)}", ")}");
	}
	
	/**
	 * 设置存储过程参数值
	 * @param cstat
	 * @param procedureColumns
	 * @param procParamValue
	 */
	private void setProcParamValue(CallableStatement cstat, ResultSet procedureColumns, Map procParamValue) {
		String columnName = "";
		try {
			procedureColumns.beforeFirst();
			boolean flag = procedureColumns.next();
			while(flag) {
				short columnType = procedureColumns.getShort("COLUMN_TYPE");
				int dataType = procedureColumns.getShort("DATA_TYPE");
				columnName = procedureColumns.getString("COLUMN_NAME");
				
				Object paramValue = procParamValue.get(columnName);
				if(paramValue == null) {
					throw new BusinessException("不存在存储过程参数:" + columnName);
				}
				
				if(columnType == 1) {
					cstat.setObject(columnName, paramValue);
				} else {
					cstat.registerOutParameter(columnName, dataType);
				}
				
				flag = procedureColumns.next();
			}
		} catch (Exception e) {
			throw new BusinessException("存储过程参数【"+columnName+"】赋值时错误\n" + e.getMessage());
		}
	}
	
	/**
	 * 设置存储过程输出值
	 * @param cstat
	 * @param otherData
	 */
	private void setOutParameterValue(CallableStatement cstat, ResultSet procedureColumns, Map otherData) {
		String columnName = "";
		try {
			procedureColumns.beforeFirst();
			boolean flag = procedureColumns.next();
			while(flag) {
				short columnType = procedureColumns.getShort("COLUMN_TYPE");
				int dataType = procedureColumns.getShort("DATA_TYPE");
				columnName = procedureColumns.getString("COLUMN_NAME");
				if(columnType == 4) {
					if(dataType == Types.DECIMAL || dataType == Types.OTHER) {
						otherData.put(columnName, cstat.getBigDecimal(columnName));
					} else if(dataType == Types.INTEGER) {
						otherData.put(columnName, cstat.getInt(columnName));
					} else if(dataType == Types.VARCHAR) {
						otherData.put(columnName, cstat.getString(columnName));
					}
				}
				flag = procedureColumns.next();
			}
		} catch (SQLException e) {
			throw new BusinessException("获取outParameter:" + columnName + "值错误\n" + e.getMessage());
		}
	}
}
