package com.biz.eisp.base.core.dao.impl;

import com.biz.eisp.base.PersisterHolder;
import com.biz.eisp.base.common.exception.BusinessException;
import com.biz.eisp.base.common.util.CollectionUtil;
import com.biz.eisp.base.common.util.StringUtil;
import com.biz.eisp.base.core.dao.BaseDao;
import com.biz.eisp.base.interfacedao.common.InterfaceDaoUtil;
import com.biz.eisp.page.Page;
import org.hibernate.*;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * Dao基类
 * Dao 操作基类 接口实现类
 * @author liukai
 * @param <T> 泛型实体Class
 * @param <PK> 主键ID
 */
@SuppressWarnings("all")
@Repository
public class BaseDaoImpl <T, PK extends Serializable> implements BaseDao{

	/**
	 * 注入一个sessionFactory属性,并注入到父类(HibernateDaoSupport)
	 */
	@Autowired
	@Qualifier("sessionFactory")
	private SessionFactory sessionFactory;
	
	@Autowired
	@Qualifier("jdbcTemplate")
	private JdbcTemplate jdbcTemplate;

	/**
	 * 获取Hibernate Session对象<br>
	 * <b>注意</b>：事务必须是开启的(Required)，否则获取不到
	 * @return Hibernate Session对象
	 */
	public Session getSession() {
		if(PersisterHolder.getHibernatePersister() != null) { //如果自有线程不为空，则取自有线程的Session
			return PersisterHolder.getHibernatePersister().getSession();
		}
		return sessionFactory.getCurrentSession();
	}

	@Override
	public <T> Serializable save(T entity) {
		return getSession().save(entity);
	}

	@Override
	public <T> void saveOrUpdate(T entity) {
		getSession().saveOrUpdate(entity);
	}

	@Override
	public <T> void delete(T entity) {
		getSession().delete(entity);
	}

	@Override
	public <T> void batchSave(List<T> entitys) {
		for (int i = 0; i < entitys.size(); i++) {
			getSession().save(entitys.get(i));
			if (i % 50 == 0) {
				getSession().flush();
				getSession().clear();
			}
		}
		getSession().flush();
		getSession().clear();
	}

	@Override
	public <T> T get(Class<T> clazz, Serializable id) {
		return (T) getSession().get(clazz, id);
	}

	@Override
	public <T> List<T> findByProperty(Class <T> clazz, String propertyName, Object value) {
		return createCriteria(clazz,
				Restrictions.eq(propertyName, value)).list();
	}
	
	@Override
	public <T> List<T> findByPropertyisOrder(Class <T> clazz, String propertyName, Object value, boolean isAsc,String isAscForpropertyName) {
		return findByCriteria(clazz, isAsc,isAscForpropertyName,Restrictions.eq(propertyName, value));
	}

	@Override
	public <T> T findUniqueByProperty(Class<T> clazz, String propertyName, Object value) {
		return (T) createCriteria(clazz,
				Restrictions.eq(propertyName, value)).uniqueResult();
	}

	@Override
	public <T> List<T> loadAll(Class<T> clazz) {
		Criteria criteria = createCriteria(clazz);
		return criteria.list();
	}

	@Override
	public <T> void deleteEntityById(Class <T> clazz, Serializable id) {
		delete(get(clazz, id));
	}

	@Override
	public <T> void deleteAllEntity(Collection<T> entities) {
		int i = 0;
		for (Object entity : entities) {
			getSession().delete(entity);
			if (i % 50 == 0) {
				getSession().flush();
				getSession().clear();
			}
		}
		getSession().flush();
		getSession().clear();
	}

	@Override
	public <T> void updateEntity(T entity) {
		getSession().update(entity);
	}

	@Override
	public Integer executeSql(String sql, Object... params) {
		Query q = getSession().createSQLQuery(sql);
		addParams(q, params);
		return q.executeUpdate();
	}
	
	@Override
	public <T>List<T> findByCriteria(Class<T> clazz,boolean isAsc,String isAscForpropertyName,Criterion... criterions){
		Criteria criteria=createCriteria(clazz, criterions);
		if (isAsc) {
			criteria.addOrder(Order.asc(isAscForpropertyName));
		} else {
			criteria.addOrder(Order.desc(isAscForpropertyName));
		}
		return criteria.list();
	}
	
	@Override
	public <T>List<T> findByCriteria(Class<T> clazz,Criterion... criterions){
		Criteria criteria=createCriteria(clazz, criterions);
		return criteria.list();
	}

    @Override
    public <T> List<T> findByCriteria(Class<T> clazz, Page page, Criterion... criterions) {
        Criteria criteria = getSession().createCriteria(clazz);
        if (criterions!=null&&criterions.length>0) {
            for (Criterion cn : criterions) {
                criteria.add(cn);
            }
        }
        int pageNo = page.getPageNo();
        int pageSize = page.getInt(page.ROWS);
        criteria.setFirstResult(pageNo);
        criteria.setMaxResults(pageSize);
        return criteria.list();
    }

    /**
	 * 创建Criteria对象
	 * @param clazz 实体Class
	 * @param criterions
	 * @return Criteria对象
	 */
	private <T> Criteria createCriteria(Class<T> clazz,
			Criterion... criterions) {
		Criteria criteria = getSession().createCriteria(clazz);
		for (Criterion c : criterions) {
			if(StringUtil.isNotEmpty(c)){
				criteria.add(c);
			}
		}
		return criteria;
	}

	/**
	 * 构造查询总条数sql语句
	 * @param sql
	 * @return
	 */
	public String makeCountSql(String sql) {
		if(StringUtils.isEmpty(sql)) {
			throw new BusinessException("sql 语句为空");
		}
		
		int selectIndex = sql.trim().toLowerCase().indexOf("from");
		
		sql = "SELECT COUNT(*) AS countVal " + sql.substring(selectIndex);
		
		return sql;
	}
	
	/**
	 * 追加参数
	 * @param query
	 * @param params
	 */
	private void addParams(Query query,Object... params){
		if(params!=null&&params.length>0){
			for(int i=0;i<params.length;i++){
				query.setParameter(i, params[i]);
			}
		}
	}
	
	@Override
	public List<Map<String, Object>> findForMapList(String sql, Object... objs) {
		return this.jdbcTemplate.queryForList(sql, objs);
	}
	
	@Override
	public Map<String, Object> findForMap(String sql, Object... objs){
		return this.jdbcTemplate.queryForMap(sql, objs);
	}
	
	@Override
	public Long getCountForJdbcParam(String sql, Object... params) {
		return jdbcTemplate.queryForObject(sql, params, Long.class);
	}
	
	@Override
	public <T> List<T> findByHql(String hql, Object... params) {
		Query q = getSession().createQuery(hql);
		addParams(q, params);
		q.setCacheable(true);
		return q.list();
	}
	
	@Override
	public <T> List<T> findByHql(String hql, Page page, Object... params) {

		//查询总条数
		String onHql = this.makeCountSql(hql);
		Query query=getSession().createQuery(onHql);
		addParams(query, params);
		long count = (Long)query.uniqueResult();
		page.put(Page.TOTAL, String.valueOf(count));
		
		//分页结果查询 
		 Query query1 = getSession().createQuery(hql);
		 addParams(query1, params);
		 int pageNo = page.getPageNo();
		 int pageSize = page.getInt(page.ROWS);
         query1.setFirstResult(pageNo);
         query1.setMaxResults(pageSize);
        
		 return query1.list();
	}
	
	@Override
	public <T> T getUniqueBySql(Class<T> clazz, String sql, Object... params) {
		RowMapper<?> rm = BeanPropertyRowMapper.newInstance(clazz);
		return (T) jdbcTemplate.queryForObject(sql, rm,params);
	}

	@Override
	public <T> List<T> findBySql(Class<T> clazz, String sql, Object... params) {
	
		RowMapper<?> rm = BeanPropertyRowMapper.newInstance(clazz);
		return  (List<T>) jdbcTemplate.query(sql, rm, params);
	}

	@SuppressWarnings("all")
	@Override
	public <T> List<T> findBySql(Class<T> clazz, String sql, Page page, Object... params) {
		
		//拼装数量结果查询sql
		String countSql = this.makeCountSql(sql);
		//查询
		Integer count =jdbcTemplate.queryForObject(countSql, Integer.class,params);
		//设置分页总条数
		page.put(Page.TOTAL, String.valueOf(count));
		
		//分页结果查询 
		 int pages = page.getPageNo();
		 int rows = page.getInt(page.ROWS);
		 
		 sql =InterfaceDaoUtil.createPageSql(InterfaceDaoUtil.DATABSE_TYPE_MYSQL, sql,  pages,rows);
		RowMapper<?> rm = BeanPropertyRowMapper.newInstance(clazz);
		return  (List<T>) jdbcTemplate.query(sql, rm,params);
	}

	@Override
	public <T> T getUniqueByHql(Class<T> clazz, String hql, Object... params) {
		Query query=getSession().createQuery(hql);
		addParams(query, params);
		return (T) query.uniqueResult();
	}

	@Override
	public void updateBySql(String sql,Object...param) {
		jdbcTemplate.update(sql, param);
	}

	@Override
	public void callableStatementByName(String proc, List<String> list) {
		StringBuffer bufProc = new StringBuffer();
    	bufProc.append("{call " + proc);
    	int length = list.size();
    	bufProc.append(" (");
    	for (int i = 0; i < length; i++) {
			if(i>0){
				bufProc.append(",");
			}
			bufProc.append("?");
		}
    	bufProc.append(")}");
    	
    	SQLQuery query = getSession().createSQLQuery(bufProc.toString());
    	for (int i = 0; i < list.size(); i++) {
    		query.setString(i, list.get(i));
		}
    	query.executeUpdate();
	}

	
}
