package com.biz.eisp.base.core.service;

import com.biz.eisp.base.common.jsonmodel.ComboBox;
import com.biz.eisp.base.common.jsonmodel.ComboTree;
import com.biz.eisp.base.common.jsonmodel.TreeGrid;
import com.biz.eisp.base.common.tag.bean.ComboTreeModel;
import com.biz.eisp.base.common.tag.bean.TreeGridModel;
import com.biz.eisp.base.utils.UploadFile;
import com.biz.eisp.page.Page;
import com.biz.eisp.vo.OperationType;
import org.hibernate.Session;
import org.hibernate.criterion.Criterion;
import org.springframework.jdbc.core.JdbcTemplate;

import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * <p>基础Service<p>
 * 其他所有Service都应该继承于它。
 * 它提供控制器通用功能：
 * <li>提供对实体的CURD操作</li>
 * <li>提供丰富的通用查询功能</li>
 * @author liukai
 *
 */
public interface BaseService {

	/**
	 * 获取Session对象.
	 * <p>
	 * @return
	 */
	public Session getSession();

	/**
	 * 获取JdbcTemplate对象
	 * @return
	 */
	public JdbcTemplate getJdbcTemplate();
	
	/**
	 * 保存实体
	 * @param entity Hibernate POJO对象
	 * @return T-泛型实体对象
	 */
	public <T> Serializable save(T entity);

	/**
	 * 保存修改实体
	 * @param entity Hibernate POJO对象
	 */
	public <T> void saveOrUpdate(T entity);

	/**
	 * 批量保存实体
	 * @param entities Hibernate POJO对象列表
	 */
	public <T> void batchSave(List<T> entities);

	/**
	 * 删除实体
	 * @param entity Hibernate POJO对象
	 */
	public <T> void delete(T entity);
	
	/**
	 * 修改实体
	 * @param sql 语句
	 * @param param 参数 
	 */
	public  void updateBySql(String sql, Object... param);
	
	/**
	 * 删除实体主键删除
	 * @param clazz 实体Class
	 * @param id 实体主键ID
	 */
	public <T> void deleteEntityById(Class<T> clazz, Serializable id);

	/**
	 * 删除实体集合
	 * @param entities 实体集合
	 */
	public <T> void deleteAllEntity(Collection<T> entities);

	/**
	 * 修改实体
	 * @param entity Hibernate POJO对象
	 */
	public <T> void updateEntity(T entity);
	
	/**
	 * 根据实体类型和主键获取实体
	 * @param <T>
	 * @param clazz 实体Class
	 * @param id 实体主键ID
	 * @return T-泛型实体对象
	 */
	public <T> T get(Class<T> clazz, Serializable id);

	/**
	 * 通过属性获取实体<br>
	 * 例如：Student实体，存在code属性，则可以使用findUniqueByProperty(Student.class, "code", "0001")
	 * @param clazz 实体Class
	 * @param propertyName 实体属性
	 * @param value 值
	 * @return T-泛型实体对象列表
	 */
	public <T> List<T> findByProperty(Class<T> clazz,
                                      String propertyName, Object value);
	
	/**
	 * 通过属性获取实体带排序<br>
	 * 例如：Student实体，存在code属性，则可以使用findUniqueByProperty(Student.class, "code", "0001", true)
	 * @param clazz 实体Class
	 * @param propertyName 实体属性
	 * @param value 值
	 * @param isAsc 是否升序   true 升序， false 降序
	 * @param isAscForpropertyName 排序字段
	 * @return T-泛型实体对象列表
	 */
	public <T> List<T> findByPropertyisOrder(Class<T> clazz,
                                             String propertyName, Object value, boolean isAsc, String isAscForpropertyName);
	
	/**
	 * 根据属性获取实体唯一记录<br>
	 * 例如：Student实体，存在code属性，则可以使用findUniqueByProperty(Student.class, "code", "0001")
	 * @param clazz 实体Class
	 * @param propertyName 实体属性
	 * @param value 值
	 * @return T-泛型实体对象
	 */
	public <T> T findUniqueByProperty(Class<T> clazz,
                                      String propertyName, Object value);
	
	/**
	 * 加载全部实体
	 * @param clazz 实体Class
	 * @return T-泛型实体对象列表
	 */
	public <T> List<T> loadAll(final Class<T> clazz);

	/**
	 * QBC查询
	 * @param clazz
	 * @param criterions
	 * @return 对象集合
	 */
	public <T>List<T> findByCriteria(Class<T> clazz, boolean isAsc, String isAscForpropertyName, Criterion... criterions);
	
	/**
	 * QBC查询
	 * @param clazz
	 * @param criterions
	 * @return 对象集合
	 */
	public <T>List<T> findByCriteria(Class<T> clazz, Criterion... criterions);

    /**
     * QBC分页查询
     *
     * @param clazz
     * @param criterions
     * @return 对象集合
     */
    public <T> List<T> findByCriteria(Class<T> clazz, Page page, Criterion... criterions);

    /**
	 * 执行sql语句
	 * @param sql
	 * @param params 动态查询参数， 此参数sql使用?做为占位符
	 * @return 返回执行条数
	 */
	public Integer executeSql(String sql, Object... params);
	
	/**
	 * jdbc查询
	 * @param sql
	 * @param objs
	 * @return
	 */
	public List<Map<String, Object>> findForMapList(String sql, Object... objs) ;
	
	/**
	 * jdbc查询
	 * @param sql
	 * @param objs
	 * @return map
	 */
	public Map<String, Object> findForMap(String sql, Object... objs) ;
	
	/**
	 * jdbc查询结果条数 
	 * @param sql
	 * @param objs
	 * @return
	 */
	public Long getCountForJdbcParam(String sql, Object... params) ;
	
	/**
	 * 通过hql 查询语句查找对象
	 * @param hql hql语句
	 * @param params 动态查询参数， 此参数hql使用?做为占位符
	 * @return T-泛型实体对象列表
	 */
	public <T> List<T> findByHql(String hql, Object... params);
	
	/**
	 * 通过hql 查询语句查找对象
	 * @param clazz 返回对象类
	 * @param hql hql语句
	 * @param params 动态查询参数， 此参数hql使用?做为占位符
	 * @return 对象
	 */
	public <T> T getUniqueByHql(Class<T> clazz, String hql, Object... params);
	
	/**
	 * 通过hql 查询语句查找对象
	 * @param hql hql语句
	 * @param page 页面对象
	 * @param params 动态查询参数， 此参数hql使用?做为占位符
	 * @return
	 */
	public <T> List<T> findByHql(String hql, Page page, Object... params);
	
	/**
	 * 通过Sql查询语句查询单个对象
	 * @param clazz 返回对象类
	 * @param sql 查询sql语句
	 * @param params 动态查询参数
	 * @return clazz对象
	 */
	public <T> T getUniqueBySql(Class<T> clazz, String sql, Object... params);
	
	/**
	 * 通过sql 查询语句查找对象
	 * @param clazz 返回值对象
	 * @param sql sql语句
	 * @param params 动态查询参数， 此参数hql使用?做为占位符
	 * @return T-泛型实体对象列表
	 */
	public <T> List<T> findBySql(Class<T> clazz, String sql, Object... params);
	
	/**
	 * 通过sql 查询语句查找对象
	 * @param clazz 返回值对象
	 * @param sql sql语句
	 * @param page 页面对象
	 * @param params 动态查询参数， 此参数hql使用?做为占位符
	 * @return T-泛型实体对象列表
	 */
	public <T> List<T> findBySql(Class<T> clazz, String sql, Page page, Object... params);
	
	/**	
	 * 构建树形数据表.
	 * @param all 集合数据
	 * @param treeGridModel 树model
	 * @return 树model-list
	 */
	public List<TreeGrid> treegrid(List all, TreeGridModel treeGridModel);
	
	/**
	 * 构建ComboTree
	 * @param obj  处理对象集合
	 * @param comboTreeModel  ComboTree模型
	 * @param in 选中项
	 * @param recursive 是否递归子节点
	 * @return
	 */
	public List<ComboTree> comboTree(List all, ComboTreeModel comboTreeModel, List in, boolean recursive);
	
	/**
	 * 通过数据字典类型查询下拉列表所需值
	 * @param dictTypeCode
	 * @return 下拉列表集合
	 */
	public List<ComboBox> comboBoxByDict(String dictTypeCode);
	
	/**
	 * 获取下拉树列表
	 * @param list
	 * @param coTreeModel
	 * @param in
	 * @param intreeModel
	 * @return
	 */
	public List<ComboTree> findComboTreeByAll(List<?> list, ComboTreeModel coTreeModel, List<?> in,
                                              ComboTreeModel intreeModel);
	
	/**
	 * 树型数据生成subCode
	 * @param codeLength 长度
	 * @param tableName 数据表名
	 * @param parentId 父级Id
	 * @return
	 */
	public String generateTreeSubCode(Integer codeLength, String tableName, String parentId);
	
	/**
	 * 更新叶子节点字段（is_leaf字段），过滤enable_status字段.
	 * <p>
	 * @param tableName 数据库表名
	 */
	public void updateIsLeafColumn(String tableName);
	
	/**
	 * 更新叶子节点字段（is_leaf字段），不过滤enable_status字段.
	 * <p>
	 * @param tableName 数据库表名
	 */
	public void updateAllIsLeafColumn(String tableName);
	
	public Object uploadFile(UploadFile uploadFile, String extendServiceName, String businessKey);
	
	public Object uploadFile(UploadFile uploadFile);
	
	public HttpServletResponse viewOrDownloadFile(UploadFile uploadFile);
	
	/**
	 * 手动追加日志和删除废数据 
	 * @param relationId 之前存在的日志业务id
	 * @param relationObj 之前日志的业务对象
	 * @param updateToBusinessId 修改后的业务id
	 * @param logContent 日志详情
	 */
	public void addLogAndRemoveInvalidInfo(String relationId, Object relationObj, String updateToBusinessId, String logContent);
	
	/**
	 * 手动追加日志
	 * @param operationType 操作类型
	 * @param businessDesc 业务描述
	 * @param businessId 业务id
	 * @param  logContent 日志详情
	 */
	public void addLog(OperationType operationType, String businessDesc, String businessId, String logContent);
	
	/**
	 * 调用存储过程.
	 * <p>
	 * @param proc 存储过程名称
	 * @param list 存储过程参数值列表
	 */
	public void callableStatementByName(String proc, List<String> list);
}
