package com.bizunited.empower.business.common.service.internal;

import com.bizunited.empower.business.common.service.NoticeSmsService;
import com.bizunited.platform.core.service.sms.SmsService;
import com.bizunited.platform.core.service.sms.SmsTypeEnums;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.conn.ssl.DefaultHostnameVerifier;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Base64;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * 短信验证码发送实现类
 * <pre>
 *   1、实现nebula的rbac框架中的发送短信接口
 * </pre>
 *
 * @author Keller
 */
public class HuaWeiSmsServiceImpl implements SmsService, NoticeSmsService {
  /**
   * 日志
   */
  private final static Logger log = LoggerFactory.getLogger(HuaWeiSmsServiceImpl.class);
  /**
   * 短信发送url
   */
  private static final String SMS_API_URL = "https://rtcsms.cn-north-1.myhuaweicloud.com:10743/sms/batchSendSms/v1";
  /**
   * 无需修改,用于格式化鉴权头域,给"X-WSSE"参数赋值
   */
  private static final String WSSE_HEADER_FORMAT = "UsernameToken Username=\"%s\",PasswordDigest=\"%s\",Nonce=\"%s\",Created=\"%s\"";
  /**
   * 无需修改,用于格式化鉴权头域,给"Authorization"参数赋值
   */
  private static final String AUTH_HEADER_VALUE = "WSSE realm=\"SDP\",profile=\"UsernameToken\",type=\"Appkey\"";

  /**
   * 短信发送key
   */
  @Value("${sms.huawei.app_key:4pU98dK88VcUh2G2k5E1a3UP4hQx}")
  private String appKey = "4pU98dK88VcUh2G2k5E1a3UP4hQx";
  /**
   * 短信发送secret
   */
  @Value("${sms.huawei.app_secret:U8V66nU18331NuysRXea6q0II75s}")
  private String appSecret = "U8V66nU18331NuysRXea6q0II75s";
  /**
   * 短信发送渠道，签名处查看
   */
  @Value("${sms.huawei.channel:8821091005706}")
  private String channel = "8821091005706";
  /**
   * 短信签名名称
   */
  @Value("${sms.huawei.signature:智企分销}")
  private String signature = "智企分销";
  /**
   * 短信发送模板id
   */
  @Value("${sms.huawei.default_template_id:3739ae6c1a0745bfa2074efbfbf68a01}")
  private String templateId = "3739ae6c1a0745bfa2074efbfbf68a01";


  @Override
  public void sendSms(String phone, String content, SmsTypeEnums smsType) {
    log.debug("开始发送验证码信息至{},验证码为{}", phone, content);

    //设置模板参数 手机号与验证码
    String params = StringUtils.join("[\"", content, "\"]");


    String body = buildRequestBody(channel, phone, this.templateId, params, null, signature);
    Validate.notBlank(body, "发送消息体为空，请检查!");
    //请求Headers中的X-WSSE参数值
    String wsseHeader = buildWsseHeader(appKey, appSecret);
    Validate.notBlank(wsseHeader, "发送消息体为空，请检查!");

    //发送验证码
    this.execute(wsseHeader, body);
  }

  @Override
  public void sendSms(String phone, String templateId, String content, SmsTypeEnums smsType) {
    log.debug("开始发送验证码信息至{},验证码为{}", phone, content);

    //设置模板参数 手机号与验证码
    String params = StringUtils.join("[\"", content, "\"]");


    String body = buildRequestBody(channel, phone, templateId, params, null, signature);
    Validate.notBlank(body, "发送消息体为空，请检查!");
    //请求Headers中的X-WSSE参数值
    String wsseHeader = buildWsseHeader(appKey, appSecret);
    Validate.notBlank(wsseHeader, "发送消息体为空，请检查!");

    //发送验证码
    this.execute(wsseHeader, body);
  }

  @Override
  public void sendSms(String channel, String signature, String templateId, String phone, String[] params) {
    log.debug("开始发送信息至'{}', 渠道号'{}', 签名'{}',模板编号:'{}' ", phone, channel, signature, templateId);
    Validate.isTrue(ArrayUtils.isNotEmpty(params), "发送消息参数为空，请检查！");

    String content = Arrays.stream(params).map(s -> StringUtils.join("\"", s, "\"")).collect(Collectors.joining(","));
    content = StringUtils.join("[", content, "]");

    String body = buildRequestBody(channel, phone, templateId, content, null, signature);
    Validate.notBlank(body, "发送消息体为空，请检查!");
    //请求Headers中的X-WSSE参数值
    String wsseHeader = buildWsseHeader(appKey, appSecret);
    Validate.notBlank(wsseHeader, "发送消息体为空，请检查!");

    //发送验证码
    this.execute(wsseHeader, body);
  }

  /**
   * 发送消息体并返回字符串格式的响应内容
   *
   * @param body
   * @return
   */
  private void execute(String wsseHeader, String body) {
    try (CloseableHttpClient client = HttpClients.custom()
            .setSSLContext(new SSLContextBuilder().loadTrustMaterial(null,
                    (x509CertChain, authType) -> true).build())
            .setSSLHostnameVerifier(new DefaultHostnameVerifier())
            .build()) {

      HttpResponse response = client.execute(RequestBuilder.create("POST")
              .setUri(SMS_API_URL)
              .addHeader(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded")
              .addHeader(HttpHeaders.AUTHORIZATION, AUTH_HEADER_VALUE)
              .addHeader("X-WSSE", wsseHeader)
              .setEntity(new StringEntity(body)).build());
      int statusCode = response.getStatusLine().getStatusCode();
      if (statusCode != HttpStatus.SC_OK) {
        String responseStr = EntityUtils.toString(response.getEntity());
        log.debug("response: {}", responseStr);
        throw new RuntimeException("发送短信验证码失败!");
      }
    } catch (NoSuchAlgorithmException | KeyManagementException | KeyStoreException | IOException e) {
      throw new RuntimeException("创建短信发送请求失败", e);
    }
  }

  /**
   * 构造请求Body体
   *
   * @param sender
   * @param templateId
   * @param templateParas
   * @param signature     | 签名名称,使用国内短信通用模板时填写
   * @return
   */
  private String buildRequestBody(String sender, String receiver, String templateId, String templateParas, String statusCallBack, String signature) {
    Validate.isTrue(StringUtils.isNotBlank(sender) && StringUtils.isNotBlank(templateId) && StringUtils.isNotBlank(receiver), "短信发送发送人/接收人/模板编号为空，请检查！");
    return initDiffSms(sender, receiver, templateId, templateParas, statusCallBack, signature);
  }

  /**
   * 构造X-WSSE参数值
   *
   * @param appKey
   * @param appSecret
   * @return
   */
  private String buildWsseHeader(String appKey, String appSecret) {
    Validate.isTrue(StringUtils.isNotBlank(appKey) && StringUtils.isNotBlank(appSecret), "创建加密头信息失败: 数据appKey or appSecret 为空！");
    SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
    String time = sdf.format(new Date());
    String nonce = UUID.randomUUID().toString().replace("-", "");
    byte[] passwordDigest = DigestUtils.sha256(nonce + time + appSecret);
    String hexDigest = Hex.encodeHexString(passwordDigest);
    String passwordDigestBase64Str = Base64.getEncoder().encodeToString(hexDigest.getBytes());
    return String.format(WSSE_HEADER_FORMAT, appKey, passwordDigestBase64Str, nonce, time);
  }

  /**
   * 构造smsContent参数值
   *
   * @param sender
   * @param receiver
   * @param templateId
   * @param templateParas
   * @param signature     | 签名名称,使用国内短信通用模板时填写
   * @return
   */
  private String initDiffSms(String sender, String receiver, String templateId, String templateParas,
                             String statusCallBack, String signature) {
    Validate.isTrue(StringUtils.isNotBlank(receiver) && StringUtils.isNotEmpty(templateId), "初始化参数: 接收人或模板id为空，请检查！");
    List<NameValuePair> keyValues = Lists.newArrayList();

    keyValues.add(new BasicNameValuePair("from", sender));
    keyValues.add(new BasicNameValuePair("to", receiver));
    keyValues.add(new BasicNameValuePair("templateId", templateId));
    if (StringUtils.isNotBlank(templateParas)) {
      keyValues.add(new BasicNameValuePair("templateParas", templateParas));
    }
    if (StringUtils.isNotBlank(statusCallBack)) {
      keyValues.add(new BasicNameValuePair("statusCallback", statusCallBack));
    }
    if (StringUtils.isNotBlank(signature)) {
      keyValues.add(new BasicNameValuePair("signature", signature));
    }
    return URLEncodedUtils.format(keyValues, StandardCharsets.UTF_8);
  }

}
