package com.bizunited.empower.business.decoration.service.internal;

import com.bizunited.empower.business.allowable.service.CustomerBuyableProductService;
import com.bizunited.empower.business.decoration.service.DecorationProductVoService;
import com.bizunited.empower.business.product.common.aspect.LabelPreferentialItems;
import com.bizunited.empower.business.product.dto.DiscountResultDto;
import com.bizunited.empower.business.product.service.ProductVoService;
import com.bizunited.empower.business.product.service.handler.DiscountHandler;
import com.bizunited.empower.business.product.vo.ProductUnitAndPriceVo;
import com.bizunited.empower.business.product.vo.ProductUnitSpecificationAndPriceVo;
import com.bizunited.empower.business.product.vo.ProductVo;
import com.bizunited.platform.script.context.InvokeParams;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 店铺装修--商品分组展示商品的Service实现
 */
@Service
public class DecorationProductVoServiceImpl implements DecorationProductVoService {

  /** 最大显示数量 */
  private static final Integer DISPLAY_MAX_NUM = 20;

  @Autowired
  private CustomerBuyableProductService customerBuyableProductService;
  @Autowired
  @Qualifier("VirtualHeadDiscountHandler")
  DiscountHandler discountHandler;
  @Autowired
  private ProductVoService productVoService;

  /**
   * 根据装修商品组条件，查询数据，只查询已上架数据
   * @param productCategoryFlatCode 快速商品分类编码（可以为空，为空时查询全部商品）
   * @param displayOrder 展示排序（目前只支持销售价的升序何降序，3表示升序，4表示降序）
   * @param displayNum 展示数量（不超过20个）
   * @param customerCode 客户编码
   */
  @Override
  @LabelPreferentialItems("#customerCode")
  public List<ProductVo> findByDecorationCondition(String productCategoryFlatCode, Integer displayOrder, Integer displayNum, String customerCode) {
    String tenantCode = TenantUtils.getTenantCode();
    if(StringUtils.isBlank(tenantCode)){
      return Lists.newArrayList();
    }

    InvokeParams invokeParams = new InvokeParams();
    invokeParams.putInvokeParam("tenantCode",tenantCode);
    invokeParams.putInvokeParam("shelfStatus","1");
    if(StringUtils.isNotBlank(productCategoryFlatCode)){
      invokeParams.putInvokeParam("categoryFlatCode",productCategoryFlatCode);
    }

    //当客户编码为空时，直接查询符合条件的商品返回（可能无商品分类）
    if(StringUtils.isBlank(customerCode)){
      Page<ProductVo> productVoPage = productVoService.findByConditions(PageRequest.of(0,displayNum != null ? displayNum : DISPLAY_MAX_NUM),invokeParams);
      if(productVoPage == null || CollectionUtils.isEmpty(productVoPage.getContent())){
        return Lists.newArrayList();
      }
      //填充销售价，为排序做准备
      this.fillSellingPrice(productVoPage);
      return this.processSort(productVoPage,displayOrder);
    }

    //查询客户所能购买的商品范围
    Set<String> productSpecificationCodes = customerBuyableProductService.findSpecificationsByCustomerCode(customerCode);
    if(CollectionUtils.isEmpty(productSpecificationCodes)){
      return Lists.newArrayList();
    }

    //执行到这里，一定是有客户编码的
    invokeParams.putInvokeParam("customerCode",customerCode);
    //查询客户具体能购买或展示的商品
    invokeParams.putInvokeParam("productCodeList",StringUtils.join(productSpecificationCodes,","));
    Page<ProductVo> productVoPage = productVoService.findByConditions(PageRequest.of(0,displayNum != null ? displayNum : DISPLAY_MAX_NUM),invokeParams);
    if(productVoPage == null || CollectionUtils.isEmpty(productVoPage.getContent())){
      return Lists.newArrayList();
    }
    //转换价格
    this.transSellingPrice(productVoPage,customerCode);
    //处理价格排序（按照基本单位的销售价格比较）
    return this.processSort(productVoPage,displayOrder);
  }

  /**
   * 设置销售价(基本单位)，为后续的排序做准备
   */
  private void fillSellingPrice(Page<ProductVo> productVoPage){
    List<ProductVo> content = productVoPage.getContent();
    for(ProductVo productVo : content){
      for(ProductUnitAndPriceVo up : productVo.getProductUnitAndPrices()){
        if(up.getBasicUnit()){
          productVo.setBaseUnitSellPrice(up.getSellingPrice());
          break;
        }
      }
    }
  }

  /**
   * 转换价格
   */
  private void transSellingPrice(Page<ProductVo> productVoPage, String customerCode){
    for(ProductVo p : productVoPage.getContent()){
      DiscountResultDto element = new DiscountResultDto();
      element.setCustomerCode(customerCode);
      element.setProductCode(p.getProductCode());
      List<DiscountResultDto> processResult = this.discountHandler.process(element, null);
      for(DiscountResultDto dto : processResult){
        BigDecimal discountPrice = dto.getAfterDiscountPrice();
        String unitCode = dto.getUnitCode();
        Set<ProductUnitSpecificationAndPriceVo> productUnitAndPrices = p.getProductUnitSpecificationAndPrices();
        if(!CollectionUtils.isEmpty(productUnitAndPrices)){
          for(ProductUnitSpecificationAndPriceVo up : productUnitAndPrices){
            if(up.getProductUnit() != null && StringUtils.equals(up.getProductUnit().getUnitCode(),unitCode)){
              up.setSellingPrice(discountPrice);
              if(up.getBasicUnit()){
                p.setBaseUnitSellPrice(discountPrice);
                break;
              }
            }
          }
        }
      }
    }
  }


  /**
   * 处理排序
   * 展示排序（目前只支持销售价的升序何降序，3表示升序，4表示降序）
   */
  private List<ProductVo> processSort(Page<ProductVo> source, Integer displayOrder){
    if(displayOrder == null || displayOrder > 4 || displayOrder < 3){
      displayOrder = 3;
    }
    final Integer sortType = displayOrder;
    return source.getContent().stream().sorted(Comparator.comparing(ProductVo::getBaseUnitSellPrice,(e1,e2) -> {
      if(sortType == 3){
        return e1.compareTo(e2);
      }else{
        return e2.compareTo(e1);
      }
    })).collect(Collectors.toList());
  }
}
