package com.bizunited.empower.business.decoration.service.internal;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.bizunited.empower.business.decoration.entity.DecorationTerminalMiniSchema;
import com.bizunited.empower.business.decoration.entity.PictureAdvertisement;
import com.bizunited.empower.business.decoration.entity.ProductGroup;
import com.bizunited.empower.business.decoration.enums.DecorationTypeEnum;
import com.bizunited.empower.business.decoration.repository.DecorationTerminalMiniRepository;
import com.bizunited.empower.business.decoration.service.DecorationTerminalMiniService;
import com.bizunited.empower.business.decoration.service.PictureAdvertisementService;
import com.bizunited.empower.business.decoration.service.ProductGroupService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.platform.venus.sdk.service.file.FileHandleService;
import com.bizunited.platform.venus.sdk.vo.OrdinaryFileVo;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.bizunited.empower.business.decoration.constant.DecorationConstants.SCHEMA_ELEMENT_MAX_SIZE;
import static com.bizunited.empower.business.decoration.constant.DecorationConstants.SHCEMA_CODE_PREFIX;
import static com.bizunited.empower.business.decoration.constant.DecorationConstants.SHCEMA_FILE_PATH_PREFIX;
import static com.bizunited.empower.business.decoration.constant.DecorationRedisKey.SCHEMA_CODE_AUTO_INC_KEY;
import static com.bizunited.empower.business.decoration.enums.DecorationTypeEnum.PIC_ADVERT;
import static com.bizunited.empower.business.decoration.enums.DecorationTypeEnum.SHOP_WINDOW;


/**
 * 小程序端装修结构体信息Service层
 */
@Service
public class DecorationTerminalMiniServiceImpl implements DecorationTerminalMiniService {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(DecorationTerminalMiniServiceImpl.class);

  @Autowired
  private DecorationTerminalMiniRepository decorationTerminalMiniRepository;
  @Autowired
  private FileHandleService fileHandleService;
  @Autowired
  private PictureAdvertisementService pictureAdvertisementService;
  @Autowired
  private ProductGroupService productGroupService;
  @Autowired
  private RedisMutexService redisMutexService;

  /**
   * 允许的且必须的结构体元素类型集合
   */
  private static final Set<String> SCHEMA_ELEMENT_TYPES = DecorationTypeEnum.codes();

  /**
   * 保存或更新小程序端装修结构体jsonSchema
   *
   * @param jsonArr 结构体
   */
  @Override
  @Transactional
  public DecorationTerminalMiniSchema save(JSONArray jsonArr) {
    Validate.notEmpty(jsonArr, "传入的装修结构体不能为空");
    String tenatCode = TenantUtils.getTenantCode();
    Validate.notBlank(tenatCode, "租户编码不能为空");
    DecorationTerminalMiniSchema schema = this.findByTenantCode(tenatCode);
    Date now = new Date();
    String account = SecurityUtils.getUserAccount();
    String fileName;
    if (schema == null) {
      schema = new DecorationTerminalMiniSchema();
      schema.setCreateAccount(account);
      schema.setCreateTime(now);
      schema.setTenantCode(tenatCode);
      schema.setCode(this.generateCode(tenatCode));
      schema.setRelativePath(SHCEMA_FILE_PATH_PREFIX);
      fileName = StringUtils.join(schema.getCode(), "_", tenatCode, ".txt");
      schema.setFileName(fileName);
    }
    schema.setModifyAccount(account);
    schema.setModifyTime(now);

    //校验数据
    this.validateStructure(jsonArr);

    //保存结构体文件
    String jsonArrStr = JSONArray.toJSONString(jsonArr);
    fileName = schema.getFileName();

    OrdinaryFileVo ordinaryFileVo = fileHandleService.fileUpload(SHCEMA_FILE_PATH_PREFIX, account, null, fileName, jsonArrStr.getBytes(StandardCharsets.UTF_8));
    schema.setRelativePath(ordinaryFileVo.getRelativeLocal());
    schema.setFileName(ordinaryFileVo.getFileName());
    //保存数据库
    return decorationTerminalMiniRepository.saveAndFlush(schema);
  }

  /**
   * 生成结构数据编码
   */
  private String generateCode(String tenantCode) {
    String redisKey = String.format(SCHEMA_CODE_AUTO_INC_KEY, tenantCode);
    String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
    return StringUtils.join(SHCEMA_CODE_PREFIX, index);
  }

  /**
   * 查询指定租户的装修结构
   *
   * @param tenantCode 租户编码
   */
  @Override
  public DecorationTerminalMiniSchema findByTenantCode(String tenantCode) {
    if (StringUtils.isBlank(tenantCode)) {
      return null;
    }
    return decorationTerminalMiniRepository.findByTenantCode(tenantCode);
  }


  /**
   * pc端 查询指定租户的装修结构(jsonArray格式)
   *
   * @param tenantCode 租户编码
   */
  @Override
  public JSONArray findSchemaByTenantCode(String tenantCode) {
    if (StringUtils.isBlank(tenantCode)) {
      return new JSONArray();
    }
    DecorationTerminalMiniSchema schema = this.findByTenantCode(tenantCode);
    if (schema == null || StringUtils.isAnyBlank(schema.getRelativePath(), schema.getFileName())) {
      return new JSONArray();
    }
    byte[] bytes = this.getBytes(schema);
    if (bytes == null || bytes.length == 0) {
      return new JSONArray();
    }
    //文件数据转换为schemaJson结构
    JSONArray shcema = JSONArray.parseArray(new String(bytes, StandardCharsets.UTF_8));
    //填充图片广告和橱窗商品数据
    for (int index = 0; index < shcema.size(); index++) {
      JSONObject element = shcema.getJSONObject(index);
      String code = element.getString("type");
      Validate.notBlank(code, "装修结构数据的type信息不能为空");
      if (!StringUtils.equals(PIC_ADVERT.getCode(), code) && !StringUtils.equals(SHOP_WINDOW.getCode(), code)) {
        continue;
      }
      this.processPicAdvertisement(element);
      this.processShopWindow(element);
    }

    return shcema;
  }

  /**
   * 小程序端 查询指定租户的装修结构(jsonArray格式)
   */
  @Override
  public JSONArray findSchema() {
    return this.findSchemaByTenantCode(TenantUtils.getTenantCode());
  }


  /**
   * 校验结构数据
   *
   * @param jsonArr 结构数据
   */
  private void validateStructure(JSONArray jsonArr) {
    Validate.notEmpty(jsonArr, "装修结构数据不能为空");
    Validate.isTrue(jsonArr.size() == SCHEMA_ELEMENT_MAX_SIZE, "装修结构数据元素不足或超过限定数量%d", SCHEMA_ELEMENT_MAX_SIZE);
    Set<String> elementCodes = Sets.newHashSet();
    for (int index = 0; index < SCHEMA_ELEMENT_MAX_SIZE; index++) {
      JSONObject element = jsonArr.getJSONObject(index);
      String code = element.getString("type");
      Validate.notBlank(code, "装修结构数据的type信息不能为空");
      Validate.notEmpty(element.getJSONObject("props"), "【%s】装修结构数据的props信息不能为空", code);
      Validate.notBlank(element.getString("visible"), "【%s】装修结构数据的visible信息不能为空", code);
      Validate.isTrue(SCHEMA_ELEMENT_TYPES.contains(code), "发现未知的装修结构元素类型，请联系管理员");
      JSONObject props = element.getJSONObject("props");
      JSONArray jsonArrData = props.getJSONArray("data");
      Validate.notNull(jsonArrData, "装修【%s】结构data属性不能为空", code);
      if (StringUtils.equals(PIC_ADVERT.getCode(), code) || StringUtils.equals(SHOP_WINDOW.getCode(), code)) {
        jsonArrData.clear();
      }
      elementCodes.add(code);
    }
    Validate.isTrue(elementCodes.size() == SCHEMA_ELEMENT_MAX_SIZE, "装修结构数据元素重复，请检查");
  }


  /**
   * 组装图片广告数据
   *
   * @param element 图片广告结构
   */
  private void processPicAdvertisement(JSONObject element) {
    if (!StringUtils.equals(element.getString("type"), PIC_ADVERT.getCode())) {
      return;
    }
    JSONObject props = element.getJSONObject("props");
    Validate.notEmpty(props, "图片广告装修结构数据的props信息不能为空");
    JSONArray jsonArrData = props.getJSONArray("data");
    Validate.notNull(jsonArrData, "图片广告装修结构数据的data信息不能为空");
    jsonArrData.clear();
    List<PictureAdvertisement> pas = pictureAdvertisementService.findByTenantCode(TenantUtils.getTenantCode());
    if (!CollectionUtils.isEmpty(pas)) {
      //抽取有效且已上架的数据(上架且已生效或永久有效)
      List<PictureAdvertisement> result = pas.stream().filter(e -> (e.getTstatus() == 1 || e.getTstatus() == 4) && e.getShelfStatus() == 1).collect(Collectors.toList());
      for (PictureAdvertisement e : result) {
        String jsonStr = JSONObject.toJSONString(e);
        JSONObject paJson = JSONObject.parseObject(jsonStr);
        jsonArrData.add(paJson);
      }
    }
  }


  /**
   * 组装橱窗商品数据
   *
   * @param element 橱窗商品结构
   */
  private void processShopWindow(JSONObject element) {
    if (!StringUtils.equals(element.getString("type"), SHOP_WINDOW.getCode())) {
      return;
    }
    JSONObject props = element.getJSONObject("props");
    Validate.notEmpty(props, "橱窗商品装修结构数据的props信息不能为空");
    JSONArray jsonArrData = props.getJSONArray("data");
    Validate.notNull(jsonArrData, "橱窗商品装修结构数据的data信息不能为空");
    jsonArrData.clear();
    List<ProductGroup> pgs = productGroupService.findByTenantCode(TenantUtils.getTenantCode());
    if (!CollectionUtils.isEmpty(pgs)) {
      //抽取有效数据(已上架)
      List<ProductGroup> result = pgs.stream().filter(e -> e.getTstatus() == 1 && e.getShelfStatus() == 1).collect(Collectors.toList());
      for (ProductGroup e : result) {
        String jsonStr = JSONObject.toJSONString(e);
        JSONObject paJson = JSONObject.parseObject(jsonStr);
        jsonArrData.add(paJson);
      }
    }
  }


  private byte[] getBytes(DecorationTerminalMiniSchema schema) {
    String redisLockKey = StringUtils.join(schema.getRelativePath(), schema.getFileName());
    boolean tryLock = false;
    try {
      tryLock = redisMutexService.tryLock(redisLockKey, TimeUnit.SECONDS, 5, 2);
      if (!tryLock) {
        throw new IllegalArgumentException("加载装修结构信息失败，请重试");
      }
      return fileHandleService.findContentByFilePathAndFileRename(schema.getRelativePath(), schema.getFileName());
    } catch (Exception e) {
      String errorMsg = e.getMessage();
      LOGGER.error(errorMsg, e);
      throw new IllegalArgumentException("错误信息：" + errorMsg);
    } finally {
      if (tryLock) {
        redisMutexService.unlock(redisLockKey);
      }
    }
  }

}
