package com.bizunited.empower.business.decoration.service.internal;

import com.bizunited.empower.business.decoration.dto.PictureAdvertisementDto;
import com.bizunited.empower.business.decoration.entity.PictureAdvertisement;
import com.bizunited.empower.business.decoration.repository.PictureAdvertisementRepository;
import com.bizunited.empower.business.decoration.repository.internal.PictureAdvertisementRepositoryCustom;
import com.bizunited.empower.business.decoration.service.PictureAdvertisementService;
import com.bizunited.empower.business.decoration.vo.PictureAdvertisementVo;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.math.NumberUtils;
import org.redisson.api.RMap;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import static com.bizunited.empower.business.decoration.constant.DecorationConstants.PIC_ADV_CODE_PREFIX;
import static com.bizunited.empower.business.decoration.constant.DecorationRedisKey.DECORATION_HIT_NUM_MAP_KEY;
import static com.bizunited.empower.business.decoration.constant.DecorationRedisKey.PIC_ADV_CODE_AUTO_INC_KEY;

/**
 * DecorationPictureAdvertisement业务模型的服务层接口实现
 * @author hc
 */
@Service("DecorationPictureAdvertisementServiceImpl")
public class PictureAdvertisementServiceImpl implements PictureAdvertisementService {
  @Autowired
  private PictureAdvertisementRepository pictureAdvertisementRepository;
  @Autowired
  @Qualifier("_DecorationPictureAdvertisementRepositoryImpl")
  private PictureAdvertisementRepositoryCustom pictureAdvertisementRepositoryCustom;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private RedissonClient redissonClient;


  @Transactional
  @Override
  public PictureAdvertisement create(PictureAdvertisement pictureAdvertisement) {
    return this.createForm(pictureAdvertisement);
  }


  @Transactional
  @Override
  public PictureAdvertisement createForm(PictureAdvertisement pictureAdvertisement) {
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    Date now = new Date();
    pictureAdvertisement.setCreateAccount(SecurityUtils.getUserAccount());
    pictureAdvertisement.setCreateTime(now);
    pictureAdvertisement.setModifyAccount(SecurityUtils.getUserAccount());
    pictureAdvertisement.setModifyTime(now);
    pictureAdvertisement.setTenantCode(TenantUtils.getTenantCode());
    pictureAdvertisement.setCode(this.generateCode(TenantUtils.getTenantCode()));

    //1.数据边界校验
    this.createValidation(pictureAdvertisement);

    //2.数据业务逻辑校验
    this.businessValidate(pictureAdvertisement);

    // 返回最终处理的结果，里面带有详细的关联信息
    return pictureAdvertisementRepository.saveAndFlush(pictureAdvertisement);
  }


  /**
   * 在创建一个新的DecorationPictureAdvertisement模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(PictureAdvertisement pictureAdvertisement) {
    Validate.notNull(pictureAdvertisement, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(pictureAdvertisement.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    pictureAdvertisement.setId(null);
    Validate.notBlank(pictureAdvertisement.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(pictureAdvertisement.getCode(), "添加信息时，广告编号不能为空！");
    Validate.notBlank(pictureAdvertisement.getFileName(), "添加信息时，图片名称不能为空！");
    Validate.notBlank(pictureAdvertisement.getRelativePath(), "添加信息时，图片路径不能为空！");
    Validate.notBlank(pictureAdvertisement.getTitle(), "添加信息时，广告标题名称不能为空！");

    Validate.notNull(pictureAdvertisement.getSortIndex(), "添加信息时，排序值不能为空！");
    Validate.notNull(pictureAdvertisement.getShelfStatus(), "添加信息时，上下架状态不能为空！");
    Validate.notNull(pictureAdvertisement.getRelationType(), "添加信息时，关联类型不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(pictureAdvertisement.getTenantCode() == null || pictureAdvertisement.getTenantCode().length() < 255 , "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getCode() == null || pictureAdvertisement.getCode().length() < 64 , "广告编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(pictureAdvertisement.getTitle() == null || pictureAdvertisement.getTitle().length() < 255 , "广告标题名称,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getLinkCode() == null || pictureAdvertisement.getLinkCode().length() < 255 , "关联的事物编码,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getContent() == null || pictureAdvertisement.getContent().length() < 255 , "广告内容,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getRelativePath() == null || pictureAdvertisement.getRelativePath().length() < 255 , "图片路径,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getFileName() == null || pictureAdvertisement.getFileName().length() < 255 , "图片名称,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getExtend1() == null || pictureAdvertisement.getExtend1().length() < 255 , "扩展字段1,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getExtend2() == null || pictureAdvertisement.getExtend2().length() < 255 , "扩展字段2,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getExtend3() == null || pictureAdvertisement.getExtend3().length() < 255 , "扩展字段3,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getExtend4() == null || pictureAdvertisement.getExtend4().length() < 255 , "扩展字段4,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getExtend5() == null || pictureAdvertisement.getExtend5().length() < 255 , "扩展字段5,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getExtend6() == null || pictureAdvertisement.getExtend6().length() < 255 , "扩展字段6,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getExtend7() == null || pictureAdvertisement.getExtend7().length() < 255 , "扩展字段7,在进行添加时填入值超过了限定长度(255)，请检查!");
    PictureAdvertisement currentPictureAdvertisement = this.findByCode(pictureAdvertisement.getCode());
    Validate.isTrue(currentPictureAdvertisement == null, "广告编号已存在,请检查");
    currentPictureAdvertisement = this.findByTitle(pictureAdvertisement.getTitle());
    Validate.isTrue(currentPictureAdvertisement == null, "广告标题名称已存在,请检查");
  }

  /**
   * 生成图片广告数据编码
   */
  private String generateCode(String tenantCode) {
    String redisKey = String.format(PIC_ADV_CODE_AUTO_INC_KEY, tenantCode);
    String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
    return StringUtils.join(PIC_ADV_CODE_PREFIX, index);
  }


  private void businessValidate(PictureAdvertisement advertisement){
    //验证生效和失效时间，比如
    //  生效时间为空，则广告状态为立即生效
    //  失效时间为空，则广告状态为永久生效（注：如果为永久有效，需要定时任务判断生效日期的情况）
    //  生效时间 < 当前时间 < 失效时间等情况的判定
    Integer status = this.decideStatus(advertisement.getEffectiveDate(),advertisement.getInvalidDate(),new Date(),advertisement,advertisement.getShelfStatus());
    advertisement.setTstatus(status);
    Validate.notNull(advertisement.getTstatus(), "广告状态不能为空！");
    //验证关联方式
    //  如果关联是商品，则linkCode必须有值
    //  如果关联是无链接，则linkCode为空
    if(advertisement.getRelationType() == 1){
      Validate.notBlank(advertisement.getLinkCode(),"关联事物的编码不能为空");
    }else if(advertisement.getRelationType() == 2){
      advertisement.setLinkCode(null);
    }
  }

  /**
   * 比较失效和生效日期，并设置状态
   */
  private int compareDateTime(PictureAdvertisement advertisement){
    Validate.notNull(advertisement.getEffectiveDate(),"生效日期不能为空");
    Validate.notNull(advertisement.getInvalidDate(),"失效日期不能为空");
    long effectiveTime = advertisement.getEffectiveDate().getTime();
    long invalidTime = advertisement.getInvalidDate().getTime();
    Validate.isTrue(NumberUtils.compare(effectiveTime,invalidTime) < 0,"生效时间必须小于失效时间");
    long nowTime = new Date().getTime();
    int condition1 = NumberUtils.compare(effectiveTime,nowTime);
    int condition2 = NumberUtils.compare(nowTime,invalidTime);

    //如果生效时间大于当前时间，广告状态为待生效0
    //如果生效时间小于当前时间，且失效时间大于当前时间，广告状态为已生效1
    //如果失效时间大于等于当前时间，广告状态为已失效3
    return condition1 > 0 ? 0 : (condition2 < 0 ? 1 : 3);
  }

  @Transactional
  @Override
  public PictureAdvertisement update(PictureAdvertisement pictureAdvertisement) {
    return this.updateForm(pictureAdvertisement);
  }


  @Transactional
  @Override
  public PictureAdvertisement updateForm(PictureAdvertisement pictureAdvertisement) {
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     * 
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(pictureAdvertisement);
    // ===================基本信息
    String currentId = pictureAdvertisement.getId();
    Optional<PictureAdvertisement> op_currentDecorationPictureAdvertisement = this.pictureAdvertisementRepository.findById(currentId);
    PictureAdvertisement currentPictureAdvertisement = op_currentDecorationPictureAdvertisement.orElse(null);
    Validate.notNull(currentPictureAdvertisement,"未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentPictureAdvertisement.setModifyAccount(SecurityUtils.getUserAccount());
    currentPictureAdvertisement.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentPictureAdvertisement.setTenantCode(pictureAdvertisement.getTenantCode());
    currentPictureAdvertisement.setCode(pictureAdvertisement.getCode());
    currentPictureAdvertisement.setTitle(pictureAdvertisement.getTitle());
    currentPictureAdvertisement.setShelfStatus(pictureAdvertisement.getShelfStatus());
    currentPictureAdvertisement.setContent(pictureAdvertisement.getContent());
    currentPictureAdvertisement.setEffectiveDate(pictureAdvertisement.getEffectiveDate());
    currentPictureAdvertisement.setInvalidDate(pictureAdvertisement.getInvalidDate());
    currentPictureAdvertisement.setRelationType(pictureAdvertisement.getRelationType());
    currentPictureAdvertisement.setLinkCode(pictureAdvertisement.getLinkCode());
    currentPictureAdvertisement.setRelativePath(pictureAdvertisement.getRelativePath());
    currentPictureAdvertisement.setSortIndex(pictureAdvertisement.getSortIndex());
    currentPictureAdvertisement.setFileName(pictureAdvertisement.getFileName());
    currentPictureAdvertisement.setExtend1(pictureAdvertisement.getExtend1());
    currentPictureAdvertisement.setExtend2(pictureAdvertisement.getExtend2());
    currentPictureAdvertisement.setExtend3(pictureAdvertisement.getExtend3());
    currentPictureAdvertisement.setExtend4(pictureAdvertisement.getExtend4());
    currentPictureAdvertisement.setExtend5(pictureAdvertisement.getExtend5());
    currentPictureAdvertisement.setExtend6(pictureAdvertisement.getExtend6());
    currentPictureAdvertisement.setExtend7(pictureAdvertisement.getExtend7());
    currentPictureAdvertisement.setExtend8(pictureAdvertisement.getExtend8());
    currentPictureAdvertisement.setExtend9(pictureAdvertisement.getExtend9());
    currentPictureAdvertisement.setExtend10(pictureAdvertisement.getExtend10());
    currentPictureAdvertisement.setExtend11(pictureAdvertisement.getExtend11());

    //业务逻辑验证
    this.businessValidate(currentPictureAdvertisement);
    return pictureAdvertisementRepository.saveAndFlush(currentPictureAdvertisement);
  }


  /**
   * 在更新一个已有的DecorationPictureAdvertisement模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(PictureAdvertisement pictureAdvertisement) {
    Validate.isTrue(!StringUtils.isBlank(pictureAdvertisement.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(pictureAdvertisement.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(pictureAdvertisement.getCode(), "修改信息时，广告编号不能为空！");
    Validate.notBlank(pictureAdvertisement.getTitle(), "修改信息时，广告标题名称不能为空！");
    Validate.notNull(pictureAdvertisement.getShelfStatus(), "修改信息时，上下架状态不能为空！");
    Validate.notNull(pictureAdvertisement.getRelationType(), "修改信息时，关联类型不能为空！");
    Validate.notBlank(pictureAdvertisement.getFileName(), "修改信息时，图片名称不能为空！");
    Validate.notBlank(pictureAdvertisement.getRelativePath(), "修改信息时，图片路径不能为空！");
    Validate.notNull(pictureAdvertisement.getSortIndex(), "修改信息时，排序值不能为空！");
    
    // 重复性判断，基本属性，需要满足unique = true
    PictureAdvertisement currentForCode = this.findByCode(pictureAdvertisement.getCode());
    Validate.isTrue(currentForCode == null || StringUtils.equals(currentForCode.getId() , pictureAdvertisement.getId()) , "广告编号已存在,请检查");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(pictureAdvertisement.getTenantCode() == null || pictureAdvertisement.getTenantCode().length() < 255 , "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getCode() == null || pictureAdvertisement.getCode().length() < 64 , "广告编号,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(pictureAdvertisement.getTitle() == null || pictureAdvertisement.getTitle().length() < 255 , "广告标题名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getContent() == null || pictureAdvertisement.getContent().length() < 255 , "广告内容,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getLinkCode() == null || pictureAdvertisement.getLinkCode().length() < 255 , "关联的事物编码,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getRelativePath() == null || pictureAdvertisement.getRelativePath().length() < 255 , "图片路径,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getFileName() == null || pictureAdvertisement.getFileName().length() < 255 , "图片名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getExtend1() == null || pictureAdvertisement.getExtend1().length() < 255 , "扩展字段1,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getExtend2() == null || pictureAdvertisement.getExtend2().length() < 255 , "扩展字段2,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getExtend3() == null || pictureAdvertisement.getExtend3().length() < 255 , "扩展字段3,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getExtend4() == null || pictureAdvertisement.getExtend4().length() < 255 , "扩展字段4,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getExtend5() == null || pictureAdvertisement.getExtend5().length() < 255 , "扩展字段5,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getExtend6() == null || pictureAdvertisement.getExtend6().length() < 255 , "扩展字段6,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(pictureAdvertisement.getExtend7() == null || pictureAdvertisement.getExtend7().length() < 255 , "扩展字段7,在进行修改时填入值超过了限定长度(255)，请检查!");
    currentForCode = this.findByTitle(pictureAdvertisement.getTitle());
    Validate.isTrue(currentForCode == null || StringUtils.equals(currentForCode.getId() , pictureAdvertisement.getId()) , "广告标题名称已存在,请检查");
  }


  @Override
  public PictureAdvertisement findDetailsById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.pictureAdvertisementRepository.findDetailsById(id);
  }


  @Override
  public PictureAdvertisement findById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<PictureAdvertisement> op = pictureAdvertisementRepository.findById(id);
    return op.orElse(null); 
  }


  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    PictureAdvertisement current = this.findById(id);
    if(current != null) { 
      pictureAdvertisementRepository.delete(current);
    }
  }


  @Override
  public PictureAdvertisement findByCode(String code) {
    if(StringUtils.isBlank(code) || StringUtils.isBlank(TenantUtils.getTenantCode())) {
      return null;
    }
    return this.pictureAdvertisementRepository.findByCodeAndTenantCode(code,TenantUtils.getTenantCode());
  }

  /**
   * 批量更新广告状态(用于定时任务)
   * @param advertisements 广告set集合
   */
  @Override
  @Transactional
  public void updateStatusBatch(Set<PictureAdvertisement> advertisements) {
    Validate.notEmpty(advertisements,"传入的广告信息不能为空");
    pictureAdvertisementRepository.saveAll(advertisements);
  }

  /**
   * 更新点击量
   * @param code 广告编码
   */
  @Override
  public void updateHitNumByCode(String code) {
    Validate.notBlank(code,"传入的广告编码不能为空");
    //todo 这里需要单独的创建锁来进行保存自增值，后面会对该方法进行调整
    long autoIncrement = redisMutexService.getAndIncrement(code,3);
    RMap<String,Long> selectNumMap = redissonClient.getMap(DECORATION_HIT_NUM_MAP_KEY);
    selectNumMap.put(code,autoIncrement);
  }

  /**
   * 分页查询
   * @param dto 查询条件数据
   * @param pageable 分页属性
   */
  @Override
  public Page<PictureAdvertisementVo> queryPage(PictureAdvertisementDto dto, Pageable pageable) {
    if(dto == null){
      dto = new PictureAdvertisementDto();
      dto.setTenantCode(TenantUtils.getTenantCode());
    }

    if(StringUtils.isBlank(dto.getTenantCode())){
      dto.setTenantCode(TenantUtils.getTenantCode());
    }
    Page<PictureAdvertisementVo> pages = pictureAdvertisementRepositoryCustom.queryPage(dto,pageable);
    if(pages == null || CollectionUtils.isEmpty(pages.getContent())){
      return new PageImpl<>(Lists.newArrayList(),pageable,0L);
    }
    List<PictureAdvertisementVo> content = pages.getContent();
    //处理排序
    //顺序依次为已生效的、待生效的、已下架的、已失效的、权重排序、最新时间
    content = this.processSort(content);

    //从redis获取点击量信息，并且填充到分页信息中，此信息无需太过准确，所以不必加锁
    RMap<String,Long> selectNumMap = redissonClient.getMap(DECORATION_HIT_NUM_MAP_KEY);
    content.forEach(e -> {
      Long value = selectNumMap.get(e.getCode());
      e.setHitNum(value == null ? 0L : value);
    });
    return new PageImpl<>(content,pageable,pages.getTotalElements());
  }

  /**
   * 根据图片广告标题进行查询
   * @param title 图片广告标题名称
   */
  @Override
  public PictureAdvertisement findByTitle(String title) {
    if(StringUtils.isBlank(title)){
      return null;
    }
    String tanatCode = TenantUtils.getTenantCode();
    if(StringUtils.isBlank(tanatCode)){
      return null;
    }
    return pictureAdvertisementRepository.findByTitleAndTenantCode(title,tanatCode);
  }

  /**
   * 根据租户编码进行查询
   * @param tenantCode 租户编码
   */
  @Override
  public List<PictureAdvertisement> findByTenantCode(String tenantCode) {
    if(StringUtils.isBlank(tenantCode)){
      return Lists.newArrayList();
    }
    return pictureAdvertisementRepository.findByTenantCode(tenantCode);
  }

  /**
   * 改变上下架状态
   * @param code 图片广告编码
   * @param selfStatus 上下架状态
   */
  @Override
  @Transactional
  public void updateByCodeAndSelfStatus(String code, Integer selfStatus) {
    Validate.notBlank(code,"图片广告编码不能为空");
    Validate.notNull(selfStatus,"上下架状态必须传入");
    Validate.notBlank(TenantUtils.getTenantCode(),"租户编码不能为空");
    Validate.isTrue(selfStatus ==1 || selfStatus == 2,"未知的上下架状态，请检查");
    PictureAdvertisement pictureAdvertisement = this.findByCode(code);
    Validate.notNull(pictureAdvertisement,"根据【%s】编码未能获取到相应图片广告");
    Integer tstatus = this.decideStatus(pictureAdvertisement.getEffectiveDate(),pictureAdvertisement.getInvalidDate(),new Date(),pictureAdvertisement,selfStatus);
    pictureAdvertisement.setTstatus(tstatus);
    pictureAdvertisement.setShelfStatus(selfStatus);
    pictureAdvertisement.setModifyTime(new Date());
    pictureAdvertisement.setModifyAccount(SecurityUtils.getUserAccount());
    pictureAdvertisementRepository.saveAndFlush(pictureAdvertisement);
  }


  /**
   * 处理排序
   * 注意：产品要求顺序依次为已生效的、待生效的、已下架的、已失效的、权重排序、创建时间来处理
   */
  private List<PictureAdvertisementVo> processSort(List<PictureAdvertisementVo> content){
    List<PictureAdvertisementVo> result = Lists.newArrayList();
    //生效中
    List<PictureAdvertisementVo> hasEffective = content.stream().filter(e -> e.getTstatus() == 1 || e.getTstatus() == 4)
            .sorted(Comparator.comparing(PictureAdvertisementVo::getCreateTime).reversed())
            .sorted(Comparator.comparing(PictureAdvertisementVo::getSortIndex).reversed()).collect(Collectors.toList());

    //待生效
    List<PictureAdvertisementVo> waitEffective = content.stream().filter(e -> e.getTstatus() == 0)
            .sorted(Comparator.comparing(PictureAdvertisementVo::getCreateTime).reversed())
            .sorted(Comparator.comparing(PictureAdvertisementVo::getSortIndex).reversed()).collect(Collectors.toList());

    //已失效
    List<PictureAdvertisementVo> hasInvalid = content.stream().filter(e -> e.getTstatus() == 3)
            .sorted(Comparator.comparing(PictureAdvertisementVo::getCreateTime).reversed())
            .sorted(Comparator.comparing(PictureAdvertisementVo::getSortIndex).reversed()).collect(Collectors.toList());

    //已下架
    List<PictureAdvertisementVo> downSelf = content.stream().filter(e -> e.getTstatus() == 2)
            .sorted(Comparator.comparing(PictureAdvertisementVo::getCreateTime).reversed())
            .sorted(Comparator.comparing(PictureAdvertisementVo::getSortIndex).reversed()).collect(Collectors.toList());

    result.addAll(hasEffective);
    result.addAll(waitEffective);
    result.addAll(hasInvalid);
    result.addAll(downSelf);
    return result;
  }



  /**
   * 根据失效时间与生效时间、当前时间，判断当前图片广告的状态信息
   * 注意：还需要结合当前广告图片的上下架状态
   * @param ectiveDate 生效时间
   * @param invalidDate 失效时间
   * @param now 当前时间
   * @param picAdvert 当前图片广告
   * @param sourceSelfStatus 指定更新的上下架值
   */
  private Integer decideStatus(Date ectiveDate, Date invalidDate, Date now, PictureAdvertisement picAdvert,Integer sourceSelfStatus){
    Validate.isTrue(sourceSelfStatus == 1 || sourceSelfStatus == 2,"未知的上下架状态，请检查");
    //如果图片广告是下架，那么当前状态status为已下架
    if(sourceSelfStatus == 2){
      return sourceSelfStatus;
    }

    //如果图片广告是上架，需要根据当前时间、失效时间、生效时间去判断
    Integer tstatus = null;
    if(ectiveDate == null){
      if(invalidDate == null){
        tstatus = 1;
      }else{
        long invalidTime = invalidDate.getTime();
        long nowTime = now.getTime();
        tstatus = invalidTime < nowTime ? 3 : 1;
      }
    }else{
      if(invalidDate == null){
        tstatus = 4;
      }else{
        tstatus = this.compareDateTime(picAdvert);
      }
    }
    return tstatus;
  }
}
