package com.bizunited.empower.business.decoration.service.internal;

import com.bizunited.empower.business.decoration.dto.ProductGroupDto;
import com.bizunited.empower.business.decoration.entity.ProductGroup;
import com.bizunited.empower.business.decoration.repository.ProductGroupRepository;
import com.bizunited.empower.business.decoration.repository.internal.ProductGroupRepositoryCustom;
import com.bizunited.empower.business.decoration.service.ProductGroupService;
import com.bizunited.empower.business.decoration.vo.ProductGroupVo;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.empower.business.common.util.SecurityUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.bizunited.empower.business.decoration.constant.DecorationConstants.DISPLAY_MAX_NUM;
import static com.bizunited.empower.business.decoration.constant.DecorationConstants.PRODUCT_GROUP_CODE_PREFIX;
import static com.bizunited.empower.business.decoration.constant.DecorationRedisKey.PRODUCT_GROUP_CODE_AUTO_INC_KEY;

/**
 * DecorationProductGroup业务模型的服务层接口实现
 * @author hc
 */
@Service("DecorationProductGroupServiceImpl")
public class ProductGroupServiceImpl implements ProductGroupService {

  @Autowired
  private ProductGroupRepository productGroupRepository;
  @Autowired
  @Qualifier("_ProductGroupRepositoryImpl")
  private ProductGroupRepositoryCustom productGroupRepositoryCustom;
  @Autowired
  private RedisMutexService redisMutexService;



  @Transactional
  @Override
  public ProductGroup create(ProductGroup productGroup) {
    return this.createForm(productGroup);
  }


  @Transactional
  @Override
  public ProductGroup createForm(ProductGroup productGroup) {
   /* 
    * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
    * 1、如果当前模型对象不是主模型
    * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
    * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
    * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
    * 2、如果当前模型对象是主业务模型
    *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
    *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
    *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
    * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
    *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
    *   2.3.2、以及验证每个分组的OneToMany明细信息
    * */
    Date now = new Date();
    productGroup.setCreateAccount(SecurityUtils.getUserAccount());
    productGroup.setCreateTime(now);
    productGroup.setModifyAccount(SecurityUtils.getUserAccount());
    productGroup.setModifyTime(now);
    productGroup.setTenantCode(TenantUtils.getTenantCode());
    productGroup.setCode(this.generateCode(TenantUtils.getTenantCode()));
    //注意：因为上下架状态与商品组状态是一样的，所以取值按照上下架来设定
    productGroup.setTstatus(productGroup.getShelfStatus());

    this.createValidation(productGroup);
    
    // ===============================
    //  和业务有关的验证填写在这个区域    
    // ===============================
    
    this.productGroupRepository.saveAndFlush(productGroup);
    // 返回最终处理的结果，里面带有详细的关联信息
    return productGroup;
  }


  /**
   * 在创建一个新的DecorationProductGroup模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(ProductGroup productGroup) {
    Validate.notNull(productGroup, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(productGroup.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    productGroup.setId(null);
    Validate.notBlank(productGroup.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(productGroup.getCode(), "添加信息时，商品组编号不能为空！");
    Validate.notBlank(productGroup.getTitle(), "添加信息时，商品组标题名称不能为空！");
    Validate.notNull(productGroup.getSortIndex(), "添加信息时，排序权重不能为空！");
    Validate.notNull(productGroup.getShelfStatus(), "添加信息时，上下架状态不能为空！");
    Validate.notNull(productGroup.getTstatus(), "添加信息时，商品组状态不能为空！");
    Validate.notNull(productGroup.getDisplayNum(), "添加信息时，展示数量不能为空！");
    Validate.notNull(productGroup.getDisplayType(), "添加信息时，展示类型不能为空！");
    Validate.notNull(productGroup.getDisplayOrder(), "添加信息时，展示顺序不能为空！");
    Validate.isTrue(productGroup.getDisplayNum() <= DISPLAY_MAX_NUM,"展示数量不能超过%d个",DISPLAY_MAX_NUM);
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(productGroup.getTenantCode() == null || productGroup.getTenantCode().length() < 255 , "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getCode() == null || productGroup.getCode().length() < 64 , "商品组编号,在进行添加时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productGroup.getTitle() == null || productGroup.getTitle().length() < 255 , "商品组标题名称,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getExtend1() == null || productGroup.getExtend1().length() < 255 , "扩展字段1,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getExtend2() == null || productGroup.getExtend2().length() < 255 , "扩展字段2,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getExtend3() == null || productGroup.getExtend3().length() < 255 , "扩展字段3,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getExtend4() == null || productGroup.getExtend4().length() < 255 , "扩展字段4,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getExtend5() == null || productGroup.getExtend5().length() < 255 , "扩展字段5,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getExtend6() == null || productGroup.getExtend6().length() < 255 , "扩展字段6,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getExtend7() == null || productGroup.getExtend7().length() < 255 , "扩展字段7,在进行添加时填入值超过了限定长度(255)，请检查!");
    ProductGroup currentProductGroup = this.findByCode(productGroup.getCode());
    Validate.isTrue(currentProductGroup == null, "商品组编号已存在,请检查");
    currentProductGroup = this.findByTitle(productGroup.getTitle());
    Validate.isTrue(currentProductGroup == null, "商品组标题名称已存在,请检查");
    if (productGroup.getDisplayType()==2){
      Validate.notBlank(productGroup.getProductCategoryCode(), "按分类展示商品时，商品分类编号不能为空！");
      Validate.notBlank(productGroup.getProductCategoryName(), "按分类展示商品时，商品分类名称不能为空！");
      Validate.notBlank(productGroup.getProductCategoryFlatCode(), "按分类展示商品时，商品快速分类编码不能为空！");
    }
  }

  /**
   * 生成图片广告数据编码
   */
  private String generateCode(String tenantCode) {
    String redisKey = String.format(PRODUCT_GROUP_CODE_AUTO_INC_KEY, tenantCode);
    String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
    return StringUtils.join(PRODUCT_GROUP_CODE_PREFIX, index);
  }


  @Transactional
  @Override
  public ProductGroup update(ProductGroup productGroup) {
    return this.updateForm(productGroup);
  }


  @Transactional
  @Override
  public ProductGroup updateForm(ProductGroup productGroup) {
    /* 
     * 针对1.1.3版本的需求，这个对静态模型的修改操作做出调整，新的过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     * TODO 1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理（求删除、新增绑定的代码已生成）
     * 
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     *  2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *    2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *    2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    
    this.updateValidation(productGroup);
    // ===================基本信息
    String currentId = productGroup.getId();
    Optional<ProductGroup> op_currentDecorationProductGroup = this.productGroupRepository.findById(currentId);
    ProductGroup currentProductGroup = op_currentDecorationProductGroup.orElse(null);
    Validate.notNull(currentProductGroup,"未发现指定的原始模型对象信");
    // 开始赋值——更新时间与更新人
    Date now = new Date();
    currentProductGroup.setModifyAccount(SecurityUtils.getUserAccount());
    currentProductGroup.setModifyTime(now);
    // 开始重新赋值——一般属性
    currentProductGroup.setTenantCode(productGroup.getTenantCode());
    currentProductGroup.setCode(productGroup.getCode());
    currentProductGroup.setTitle(productGroup.getTitle());
    currentProductGroup.setSortIndex(productGroup.getSortIndex());
    currentProductGroup.setShelfStatus(productGroup.getShelfStatus());
    currentProductGroup.setTstatus(currentProductGroup.getShelfStatus());
    currentProductGroup.setDisplayNum(productGroup.getDisplayNum());
    currentProductGroup.setDisplayType(productGroup.getDisplayType());
    currentProductGroup.setDisplayOrder(productGroup.getDisplayOrder());
    currentProductGroup.setProductCategoryCode(productGroup.getProductCategoryCode());
    currentProductGroup.setProductCategoryFlatCode(productGroup.getProductCategoryFlatCode());
    currentProductGroup.setProductCategoryName(productGroup.getProductCategoryName());
    currentProductGroup.setExtend1(productGroup.getExtend1());
    currentProductGroup.setExtend2(productGroup.getExtend2());
    currentProductGroup.setExtend3(productGroup.getExtend3());
    currentProductGroup.setExtend4(productGroup.getExtend4());
    currentProductGroup.setExtend5(productGroup.getExtend5());
    currentProductGroup.setExtend6(productGroup.getExtend6());
    currentProductGroup.setExtend7(productGroup.getExtend7());
    currentProductGroup.setExtend8(productGroup.getExtend8());
    currentProductGroup.setExtend9(productGroup.getExtend9());
    currentProductGroup.setExtend10(productGroup.getExtend10());
    currentProductGroup.setExtend11(productGroup.getExtend11());
    
    this.productGroupRepository.saveAndFlush(currentProductGroup);
    return currentProductGroup;
  }


  /**
   * 在更新一个已有的DecorationProductGroup模型对象之前，该私有方法检查对象各属性的正确性，其id属性必须有值
   */
  private void updateValidation(ProductGroup productGroup) {
    Validate.isTrue(!StringUtils.isBlank(productGroup.getId()), "修改信息时，当期信息的数据编号（主键）必须有值！");
    
    // 基础信息判断，基本属性，需要满足not null
    Validate.notBlank(productGroup.getTenantCode(), "修改信息时，租户编号不能为空！");
    Validate.notBlank(productGroup.getCode(), "修改信息时，商品组编号不能为空！");
    Validate.notBlank(productGroup.getTitle(), "修改信息时，商品组标题名称不能为空！");
    Validate.notNull(productGroup.getSortIndex(), "修改信息时，排序权重不能为空！");
    Validate.notNull(productGroup.getTstatus(), "修改信息时，商品组状态不能为空！");
    Validate.notNull(productGroup.getShelfStatus(), "修改信息时，上下架状态不能为空！");
    Validate.notNull(productGroup.getDisplayNum(), "修改信息时，展示数量不能为空！");
    Validate.notNull(productGroup.getDisplayType(), "修改信息时，展示类型不能为空！");
    Validate.notNull(productGroup.getDisplayOrder(), "修改信息时，展示顺序不能为空！");
    Validate.isTrue(productGroup.getDisplayNum() <= DISPLAY_MAX_NUM,"展示数量不能超过%d个",DISPLAY_MAX_NUM);
    
    // 重复性判断，基本属性，需要满足unique = true
    ProductGroup currentForCode = this.findByCode(productGroup.getCode());
    Validate.isTrue(currentForCode == null || StringUtils.equals(currentForCode.getId() , productGroup.getId()) , "商品组编号已存在,请检查");
    currentForCode = this.findByTitle(productGroup.getTitle());
    Validate.isTrue(currentForCode == null || StringUtils.equals(currentForCode.getId() , productGroup.getId()) , "商品组标题名称已存在,请检查");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK，且canupdate = true
    Validate.isTrue(productGroup.getTenantCode() == null || productGroup.getTenantCode().length() < 255 , "租户编号,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getCode() == null || productGroup.getCode().length() < 64 , "商品组编号,在进行修改时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(productGroup.getTitle() == null || productGroup.getTitle().length() < 255 , "商品组标题名称,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getExtend1() == null || productGroup.getExtend1().length() < 255 , "扩展字段1,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getExtend2() == null || productGroup.getExtend2().length() < 255 , "扩展字段2,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getExtend3() == null || productGroup.getExtend3().length() < 255 , "扩展字段3,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getExtend4() == null || productGroup.getExtend4().length() < 255 , "扩展字段4,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getExtend5() == null || productGroup.getExtend5().length() < 255 , "扩展字段5,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getExtend6() == null || productGroup.getExtend6().length() < 255 , "扩展字段6,在进行修改时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(productGroup.getExtend7() == null || productGroup.getExtend7().length() < 255 , "扩展字段7,在进行修改时填入值超过了限定长度(255)，请检查!");
    if (productGroup.getDisplayType()==2){
      Validate.notBlank(productGroup.getProductCategoryCode(), "按分类展示商品时，商品分类编号不能为空！");
      Validate.notBlank(productGroup.getProductCategoryName(), "按分类展示商品时，商品分类名称不能为空！");
      Validate.notBlank(productGroup.getProductCategoryFlatCode(), "按分类展示商品时，商品快速分类编号不能为空！");
    }
  }


  @Override
  public ProductGroup findDetailsById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    return this.productGroupRepository.findDetailsById(id);
  }


  @Override
  public ProductGroup findById(String id) {
    if(StringUtils.isBlank(id)) { 
      return null;
    }
    
    Optional<ProductGroup> op = productGroupRepository.findById(id);
    return op.orElse(null); 
  }


  @Override
  @Transactional
  public void deleteById(String id) {
    // 只有存在才进行删除
    Validate.notBlank(id , "进行删除时，必须给定主键信息!!");
    ProductGroup current = this.findById(id);
    if(current != null) { 
      this.productGroupRepository.delete(current);
    }
  }


  @Override
  public ProductGroup findByCode(String code) {
    if(StringUtils.isBlank(code) || StringUtils.isBlank(TenantUtils.getTenantCode())) {
      return null;
    }
    return this.productGroupRepository.findByCodeAndTenantCode(code,TenantUtils.getTenantCode());
  }

  @Override
  public ProductGroup findByTitle(String title) {
    if(StringUtils.isBlank(title) || StringUtils.isBlank(TenantUtils.getTenantCode())) {
      return null;
    }
    return productGroupRepository.findByTitleAndTenantCode(title,TenantUtils.getTenantCode());
  }

  /**
   * 根据条件，分页查询
   * @param dto dto条件数据
   * @param pageable 分页条件
   */
  @Override
  public Page<ProductGroupVo> queryPage(ProductGroupDto dto, Pageable pageable) {
    if(dto == null){
      dto = new ProductGroupDto();
      dto.setTenantCode(TenantUtils.getTenantCode());
    }
    if(StringUtils.isBlank(dto.getTenantCode())){
      dto.setTenantCode(TenantUtils.getTenantCode());
    }

    Page<ProductGroupVo> pages = productGroupRepositoryCustom.queryPage(dto,pageable);
    if(pages == null || CollectionUtils.isEmpty(pages.getContent())){
      return Page.empty(pageable);
    }
    List<ProductGroupVo> content = pages.getContent();
    //处理排序
    //顺序依次为已生效的、待生效的、已下架的、已失效的、权重排序、最新时间
    content = this.processSort(content);
    return new PageImpl<>(content,pageable,pages.getTotalElements());
  }

  /**
   * 根据租户编码，查询商品组数据
   * @param tenantCode 租户编码
   */
  @Override
  public List<ProductGroup> findByTenantCode(String tenantCode) {
    if(StringUtils.isBlank(tenantCode)){
      return Lists.newArrayList();
    }
    return productGroupRepository.findByTenantCode(tenantCode);
  }

  @Override
  @Transactional
  public void updateSelfStatusByCode(String code, Integer selfStatus) {
    String tenantCode = TenantUtils.getTenantCode();
    Validate.notBlank(tenantCode,"租户编码不存在，请检查");
    if(StringUtils.isNotBlank(code) && selfStatus != null){
      //因为上下架状态selfStatus与商品组状态值相等，所以当selfStatus等于1时， 商品组状态也为1.  当selfStatus等于2时， 商品组状态也为2.
      ProductGroup productGroup = this.findByCode(code);
      productGroup.setTstatus(selfStatus);
      productGroup.setShelfStatus(selfStatus);
      productGroup.setModifyAccount(SecurityUtils.getUserAccount());
      productGroup.setModifyTime(new Date());
      productGroupRepository.saveAndFlush(productGroup);
    }
  }

  /**
   * 根据租户编码，查询有效的商品组数据
   * @param tenantCode 租户编号
   */
  @Override
  public List<ProductGroup> findEffectiveByTenantCode(String tenantCode) {
    if(StringUtils.isBlank(tenantCode) && StringUtils.isBlank(TenantUtils.getTenantCode())){
      return Lists.newArrayList();
    }

    if(StringUtils.isBlank(tenantCode)){
      tenantCode = TenantUtils.getTenantCode();
    }

    return productGroupRepository.findEffectiveByTenantCode(tenantCode);
  }


  /**
   * 处理排序
   * 注意：产品要求顺序依次为已生效的、已下架的、权重排序、创建时间来处理
   */
  private List<ProductGroupVo> processSort(List<ProductGroupVo> content){
    List<ProductGroupVo> result = Lists.newArrayList();
    //已生效
    List<ProductGroupVo> hasEffective = content.stream().filter(e -> e.getTstatus() == 1)
            .sorted(Comparator.comparing(ProductGroupVo::getCreateTime).reversed())
            .sorted(Comparator.comparing(ProductGroupVo::getSortIndex).reversed()).collect(Collectors.toList());

    //已下架
    List<ProductGroupVo> downSelf = content.stream().filter(e -> e.getTstatus() == 2)
            .sorted(Comparator.comparing(ProductGroupVo::getCreateTime).reversed())
            .sorted(Comparator.comparing(ProductGroupVo::getSortIndex).reversed()).collect(Collectors.toList());

    result.addAll(hasEffective);
    result.addAll(downSelf);
    return result;
  }
}
