package com.bizunited.empower.business.decoration.service.internal;

import com.bizunited.empower.business.decoration.constant.DecorationRedisKey;
import com.bizunited.empower.business.decoration.entity.UserInvitation;
import com.bizunited.empower.business.decoration.repository.UserInvitationRepository;
import com.bizunited.empower.business.decoration.service.UserInvitationService;
import com.bizunited.platform.common.service.redis.RedisMutexService;
import com.bizunited.platform.common.util.tenant.TenantUtils;
import com.bizunited.platform.user2.entity.UserEntity;
import com.bizunited.platform.user2.service.user.UserService;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.transaction.Transactional;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * UserInvitation业务模型的服务层接口实现
 *
 * @author saturn
 */
@Service("UserInvitationServiceImpl")
public class UserInvitationServiceImpl implements UserInvitationService {
  @Autowired
  private UserInvitationRepository userInvitationRepository;
  @Autowired
  private UserService userService;
  @Autowired
  private RedisMutexService redisMutexService;

  private static final Logger log = LoggerFactory.getLogger(UserInvitationServiceImpl.class);
  /** 用户验证码长度 */
  private static final Integer USER_INVITATION_LENGTH = 6;

  @Transactional
  @Override
  public UserInvitation create(UserInvitation userInvitation) {
    UserInvitation current = this.createForm(userInvitation);
    //==================================================== 
    //    这里可以处理第三方系统调用（或特殊处理过程）
    //====================================================
    return current;
  }

  @Transactional
  @Override
  public UserInvitation createForm(UserInvitation userInvitation) {
    /*
     * 针对1.1.3版本的需求，这个对静态模型的保存操作做出调整，新的包裹过程为：
     * 1、如果当前模型对象不是主模型
     * 1.1、那么创建前只会验证基本信息，直接的ManyToOne关联（单选）和ManyToMany关联（多选）
     * 1.2、验证完成后，也只会保存当前对象的基本信息，直接的单选
     *  1.3、ManyToMany的关联（多选），暂时需要开发人员自行处理
     * 2、如果当前模型对象是主业务模型
     *  2.1、创建前会验证当前模型的基本属性，单选和多选属性
     *  2.2、然后还会验证当前模型关联的各个OneToMany明细信息，调用明细对象的服务，明每一条既有明细进行验证
     *  （2.2的步骤还需要注意，如果当前被验证的关联对象是回溯对象，则不需要验证了）
     * 2.3、还会验证当前模型关联的各个OneToOne分组，调用分组对象的服务，对分组中的信息进行验证
     *   2.3.1、包括验证每一个分组项的基本信息、直接的单选、多选信息
     *   2.3.2、以及验证每个分组的OneToMany明细信息
     * */
    userInvitation.setTenantCode(TenantUtils.getTenantCode());
    this.createValidation(userInvitation);

    // ===============================
    //  和业务有关的验证填写在这个区域
    // ===============================

    String userAccount = userInvitation.getUserAccount();
    Validate.notBlank(userAccount, "生成邀请码时，对应的员工账户不应为空!");

    UserEntity user = userService.findByTenantCodeAndAccount(TenantUtils.getTenantCode(), userAccount);
    Validate.notNull(user, "生成邀请码时，对应的员工不存在!");

    UserInvitation dbUserInvitation = this.findByTenantCodeAndAccount(TenantUtils.getTenantCode(), userAccount);
    Validate.isTrue(Objects.isNull(dbUserInvitation)
            || StringUtils.isBlank(dbUserInvitation.getInvitationCode()), "该员工已经生成邀请码无需再生成");

    int i = 3;
    UserInvitation saveUserInvitation;
    do {
      saveUserInvitation = resolveInvitationCodeAndSave(userInvitation);
    } while (Objects.isNull(saveUserInvitation) && i-- > 0);
    Validate.notNull(saveUserInvitation, "生成邀请码失败，有可能是生成的邀请码重复!");

    // 返回最终处理的结果，里面带有详细的关联信息
    return saveUserInvitation;
  }

  /**
   * 在创建一个新的UserInvitation模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   */
  private void createValidation(UserInvitation userInvitation) {
    Validate.notNull(userInvitation, "进行当前操作时，信息对象必须传入!!");
    // 判定那些不能为null的输入值：条件为 caninsert = true，且nullable = false
    Validate.isTrue(StringUtils.isBlank(userInvitation.getId()), "添加信息时，当期信息的数据编号（主键）不能有值！");
    userInvitation.setId(null);
    Validate.notBlank(userInvitation.getTenantCode(), "添加信息时，租户编号不能为空！");
    Validate.notBlank(userInvitation.getUserAccount(), "添加信息时，业务员账户不能为空！");
    // 验证长度，被验证的这些字段符合特征: 字段类型为String，且不为PK （注意连续空字符串的情况） 
    Validate.isTrue(userInvitation.getTenantCode() == null || userInvitation.getTenantCode().length() < 255, "租户编号,在进行添加时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(userInvitation.getUserAccount() == null || userInvitation.getUserAccount().length() < 64, "业务员账户,在进行添加时填入值超过了限定长度(64)，请检查!");
  }


  /**
   * 查询指定经销商，用户账户的邀请码信息
   * @param tenantCode 指定经销商
   * @param userAccount 指定用户账户
   * @return
   */
  private UserInvitation findByTenantCodeAndAccount(String tenantCode, String userAccount) {
    if (StringUtils.isAnyBlank(tenantCode, userAccount)) {
      return null;
    }

    return userInvitationRepository.findByTenantCodeAndAccounts(tenantCode, userAccount);
  }

  /**
   * 解决员工邀请码的生成及信息保存
   * @param userInvitation 带有员工账号的实体
   * @return 生成以后的员工邀请码实体
   */
  private UserInvitation resolveInvitationCodeAndSave(UserInvitation userInvitation) {
    String invitationCode = generateInvitationCode(USER_INVITATION_LENGTH, 5);
    if (StringUtils.isBlank(invitationCode)) {
      return null;
    }

    String cacheLock = String.format(DecorationRedisKey.CUSTOMER_INVITATION_CODE_CACHE_LOCK_CODE,
            TenantUtils.getTenantCode(), invitationCode);
    try {
      redisMutexService.lock(cacheLock);
      UserInvitation userInvitationCode = getCustomerByInvitationCode(invitationCode);
      if (Objects.isNull(userInvitationCode)) {
        userInvitation.setInvitationCode(invitationCode);
        userInvitationRepository.saveAndFlush(userInvitation);
        return userInvitation;
      }
      log.warn("生成的邀请码: {} 已存在！！", invitationCode);
    } catch (Exception e) {
      log.error("生成邀请码失败：", e);
    } finally {
      redisMutexService.unlock(cacheLock);
    }

    return null;
  }

  /**
   * 生成员工邀请码，如果生成的邀请码存在则循环指定次数继续生成直到生成的邀请码不存在或循环结束
   * @param codeLength 邀请码的长度
   * @param cas 如果不存在循环的次数
   * @return
   */
  private String generateInvitationCode(Integer codeLength, int cas) {
    if (cas < 0) {
      cas = 5;
    }

    do {
      String invitationCode = RandomStringUtils.randomNumeric(codeLength);
      UserInvitation userInvitation = getCustomerByInvitationCode(invitationCode);
      if (Objects.isNull(userInvitation)) {
        return invitationCode;
      }
    } while (cas-- > 0);

    return null;
  }

  /**
   * 查询指定邀请码的员工邀请码信息
   * @param invitationCode 指定邀请码
   * @return 指定邀请码的员工邀请码信息
   */
  private UserInvitation getCustomerByInvitationCode(String invitationCode) {
    Validate.notBlank(invitationCode, "获取指定员工邀请码信息时，邀请码不能为空");

    return userInvitationRepository.findByInvitationCode(invitationCode);
  }

  @Override
  public List<UserInvitation> findByUserAccount(Set<String> userAccount) {
    if (CollectionUtils.isEmpty(userAccount)) {
      return null;
    }
    return this.userInvitationRepository.findByUserAccount(TenantUtils.getTenantCode(), userAccount);
  }

  @Override
  @Transactional
  public void deleteByUserAccount(String userAccount) {
    // 只有存在才进行删除
    Validate.notBlank(userAccount, "删除业务为员邀请码时，必须指定业务员账户信息!!");

    UserInvitation current =
            userInvitationRepository.findByTenantCodeAndAccounts(TenantUtils.getTenantCode(), userAccount);
    if (current != null) {
      this.userInvitationRepository.delete(current);
    }
  }

  @Override
  public void validateInvitationCode(String tenantCode, String userAccount, String invitationCode) {
    Validate.notBlank(tenantCode , "验证邀请码时，租户码必填");
    Validate.notBlank(userAccount , "验证邀请码时，业务员账号必填");
    Validate.notBlank(invitationCode , "验证邀请码是时，邀请码必填");

    UserInvitation userInvitation =
            userInvitationRepository.findAllByUserAccountAndInvitationCode(tenantCode, userAccount, invitationCode);
   Validate.notNull(userInvitation, "指定租户的业务员邀请码：%s 不存在", invitationCode);
  }

  @Override
  public UserInvitation findByInvitationCode(String invitationCode) {
    if (StringUtils.isBlank(invitationCode)){
      return null;
    }

    return userInvitationRepository.findByInvitationCode(invitationCode);
  }
}
