package com.bizunited.empower.business.payment.configuration;

import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.springframework.amqp.core.Binding;
import org.springframework.amqp.core.BindingBuilder;
import org.springframework.amqp.core.FanoutExchange;
import org.springframework.amqp.core.Queue;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.core.RabbitAdmin;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Map;

/**
 * 以当前哨兵所服务的资源桶名作为queue的名字，然后绑定到指定的exhange上
 * @author yinwenjie
 */
@Component
@Configuration
@ConditionalOnProperty(value = "spring.rabbitmq.host")
public class RabbitmqConfig {
  @Value("${operate.bucketCode}")
  private String bucketCode;
  @Resource
  private ConnectionFactory connectionFactory;
  /**
   * 固定的分发交换机的名字（当电子账户发生变化时的事件通知交换机）
   */
  private static final String EXCHANGE_ELECTRONIC_ACCOUNT_NAME = "electronic.account.event.exchanges";
  /**
   * 固定的分发交换机的名字（当订单变化相关的事件通知交换机）
   */
  public static final String EXCHANGE_PAYMENT_NAME = "payment.event.exchanges";
  /**
   * 固定的分发交换机的名字（当转账相关的事件通知交换机）
   */
  public static final String EXCHANGE_TRANSFER_NAME = "transfer.event.exchanges";
  /**
   * 固定的分发交换机的名字（当退款相关的事件通知交换机）
   */
  public static final String EXCHANGE_REFUND_NAME = "refund.event.exchanges";
  /**
   * 死信交换机
   */
  private static final String DEAD_LETTER_EXCHANGE_NAME = "dead-letter-exchange";

  @Bean
  public Queue getQueue() {
    /*
     * 由于远端RabbitMQ可能并没有完成队列、交换机和绑定关系的初始化，
     * 所以需要在这里进行可能的初始化动作，
     * 
     * 1、初始化和租户设定变化相关的交换机、队列
     * 队列名：{bucketCode} + “_electronic_account_queue”
     * */
    
    // 1、=========
    // 如果没有指定的queue，则会进行queue的创建
    RabbitAdmin admin = new RabbitAdmin(connectionFactory);
    Map<String, Object> arguments = Maps.newHashMap();
    arguments.put("x-dead-letter-exchange", DEAD_LETTER_EXCHANGE_NAME);
    // 电子账户队列和交换机
    String queueName = StringUtils.join(this.bucketCode , "_electronic_account_queue");
    Queue queue = this.bindQueueExchange(admin, queueName, EXCHANGE_ELECTRONIC_ACCOUNT_NAME, arguments);

    // 转账队列和交换机
    String transferQueueName = StringUtils.join(this.bucketCode , "_transfer.queue");
    this.bindQueueExchange(admin, transferQueueName, EXCHANGE_TRANSFER_NAME, arguments);

    String paymentQueueName = StringUtils.join(this.bucketCode , "_payment.queue");
    this.bindQueueExchange(admin, paymentQueueName, EXCHANGE_PAYMENT_NAME, arguments);

    String refundQueueName = StringUtils.join(this.bucketCode , "_refund.queue");
    this.bindQueueExchange(admin, refundQueueName, EXCHANGE_REFUND_NAME, arguments);

    return queue;
  }

  /**
   * 绑定队列和交换机
   * @param queueName
   * @param exchangeName
   * @param arguments
   */
  private Queue bindQueueExchange(RabbitAdmin admin, String queueName, String exchangeName, Map<String, Object> arguments) {
    Queue queue = new Queue(queueName, true, false, false, arguments);
    admin.declareQueue(queue);
    // 如果没有指定的Exchange，则会进行Exchange的创建
    FanoutExchange fanoutExchange = new FanoutExchange(exchangeName);
    admin.declareExchange(fanoutExchange);
    // 如果没有进行绑定，则进行一次绑定
    Binding binding = BindingBuilder.bind(queue).to(fanoutExchange);
    admin.declareBinding(binding);
    return queue;
  }


}