package com.bizunited.empower.business.payment.controller;

import com.bizunited.empower.business.payment.dto.CustomerCreditDto;
import com.bizunited.empower.business.payment.entity.CustomerCredit;
import com.bizunited.empower.business.payment.service.CustomerCreditService;
import com.bizunited.empower.business.payment.vo.CustomerCreditVo;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * CustomerCredit业务模型的MVC Controller层实现，基于HTTP Restful风格
 *
 * @author saturn
 */
@RestController
@Api(tags = "客户信用控制器")
@RequestMapping("/v1/customerCredit")
public class CustomerCreditController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(CustomerCreditController.class);

  @Autowired
  private CustomerCreditService customerCreditService;

  /**
   * 初始化多个客户的信用
   *
   * @param customerCreditDtos
   * @return
   */
  @PostMapping("")
  @ApiOperation("初始化客户的信用")
  public ResponseModel create(@RequestBody @ApiParam("初始化多个客户信用数据") List<CustomerCreditDto> customerCreditDtos) {
    try {
      List<CustomerCredit> customerCredits = customerCreditService.init(customerCreditDtos);
      return this.buildHttpResultW(customerCredits);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 初始化客户的信用
   *
   * @param customerCode
   * @param initialAmount
   * @param remark
   * @return
   */
  @PostMapping("init")
  @ApiOperation("初始化客户的信用")
  public ResponseModel init(@RequestParam @ApiParam("客户编码") String customerCode,
                            @RequestParam @ApiParam("初始额度") BigDecimal initialAmount,
                            @RequestParam(required = false) @ApiParam("备注") String remark) {
    try {
      CustomerCredit customerCredit = customerCreditService.init(customerCode, initialAmount, remark);
      return this.buildHttpResultW(customerCredit);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 更新客户信用总额度
   *
   * @param amount
   * @param remark
   * @return
   */
  @PutMapping("updateAmount")
  @ApiOperation("更新客户信用总额度")
  public ResponseModel updateAmount(@RequestParam @ApiParam("客户编码") String customerCode,
                                    @RequestParam @ApiParam("调整额度") BigDecimal amount,
                                    @RequestParam(required = false) @ApiParam("备注") String remark) {
    try {
      CustomerCredit customerCredit = customerCreditService.updateAmount(customerCode, amount, remark);
      return this.buildHttpResultW(customerCredit);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 更新客户信用状态
   *
   * @param customerCode
   * @param tstatus
   * @return
   */
  @PutMapping("updateStatus")
  @ApiOperation("更新客户信用状态")
  public ResponseModel updateStatus(@RequestParam @ApiParam("客户编码") String customerCode,
                                    @RequestParam @ApiParam("调整额度") Integer tstatus) {
    try {
      CustomerCredit customerCredit = customerCreditService.updateStatis(customerCode, tstatus);
      return this.buildHttpResultW(customerCredit);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 该接口功能负责生成基于“授信额度”的应收账款以及相应的收款凭证，在此基础上付款中心中该经销商的信用额度将被立即扣减。
   *
   * @param customerCode
   * @param amount
   * @param businessNo
   * @param remark
   * @return
   */
  @PostMapping("receipt")
  @ApiOperation("该接口功能负责生成基于“授信额度”的应收账款以及相应的收款凭证，在此基础上付款中心中该经销商的信用额度将被立即扣减。")
  public ResponseModel receipt(@RequestParam @ApiParam("客户编码") String customerCode,
                               @RequestParam @ApiParam("支付金额") BigDecimal amount,
                               @RequestParam @ApiParam("业务单号") String businessNo,
                               @RequestParam(required = false) @ApiParam("备注") String remark) {
    try {
      CustomerCredit customerCredit = customerCreditService.receipt(customerCode, amount, businessNo, remark);
      return this.buildHttpResultW(customerCredit);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 该接口功能负责验证在扣减信用额度前对信用额度支付的额度校验，如果信用额度不足则返回false，能够正常扣减信用则返回true
   *
   * @param customerCode
   * @param amount
   * @return
   */
  @GetMapping("validReceipt")
  @ApiOperation("该接口功能负责验证在扣减信用额度前对信用额度支付的额度校验，如果信用额度不足则返回false，能够正常扣减信用则返回true")
  public ResponseModel validReceipt(@RequestParam @ApiParam("客户编码") String customerCode,
                                    @RequestParam @ApiParam("支付金额") BigDecimal amount) {
    try {
      boolean result = customerCreditService.validReceipt(customerCode, amount);
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 恢复客户信用的额度
   *
   * @param customerCode
   * @param amount
   * @param businessNo
   * @param remark
   * @return
   */
  @PostMapping("recover")
  @ApiOperation("恢复客户信用的额度")
  public ResponseModel recover(@RequestParam @ApiParam("客户编码") String customerCode,
                               @RequestParam @ApiParam("还款金额") BigDecimal amount,
                               @RequestParam @ApiParam("业务单号") String businessNo,
                               @RequestParam(required = false) @ApiParam("备注") String remark) {
    try {
      CustomerCredit customerCredit = customerCreditService.recover(customerCode, amount, businessNo, remark);
      return this.buildHttpResultW(customerCredit);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据客户编码获取信息信息
   *
   * @param customerCode
   * @return
   */
  @GetMapping("findByCustomerCode")
  @ApiOperation("根据客户编码获取信用信息")
  public ResponseModel findByCustomerCode(@RequestParam @ApiParam("客户编码") String customerCode) {
    try {
      CustomerCredit customerCredit = customerCreditService.findByCustomerCode(customerCode);
      return this.buildHttpResultW(customerCredit, "customerCreditBills");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 多条件分页查询
   *
   * @param pageable
   * @return
   */
  @GetMapping("findByConditions")
  @ApiOperation(value = "多条件分页查询", notes = "分页参数为page和size，page从0开始，size默认10")
  public ResponseModel findByConditionsForCustomer(@PageableDefault(10) Pageable pageable,
                                                   HttpServletRequest request) {
    try {
      Map<String, Object> params = new HashMap<>();
      Enumeration<String> names = request.getParameterNames();
      if (names != null) {
        while (names.hasMoreElements()) {
          String name = names.nextElement();
          String[] values = request.getParameterValues(name);
          if (values != null && values.length == 1) {
            params.put(name, values[0]);
          } else if (values != null && values.length > 1) {
            params.put(name, values);
          }
        }
      }
      InvokeParams conditions = new InvokeParams();
      conditions.putInvokeParams(params);
      Page<CustomerCreditVo> page = this.customerCreditService.findByConditions(pageable, conditions);
      return this.buildHttpResultW(page);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

} 
