package com.bizunited.empower.business.payment.controller;

import com.bizunited.empower.business.payment.entity.CustomerWallet;
import com.bizunited.empower.business.payment.service.CustomerWalletService;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;

/**
 * CustomerWallet业务模型的MVC Controller层实现，基于HTTP Restful风格
 * @author saturn
 */
@RestController
@RequestMapping("/v1/customerWallet")
public class CustomerWalletController extends BaseController { 
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(CustomerWalletController.class);
  
  @Autowired
  private CustomerWalletService customerWalletService;

  /**
   * 初始化电子钱包
   * @param customerCode
   * @return
   */
  @PostMapping("init")
  @ApiOperation("初始化电子钱包")
  public ResponseModel init(@RequestParam @ApiParam("客户编码") String customerCode) {
    try {
      CustomerWallet wallet = customerWalletService.init(customerCode);
      return this.buildHttpResultW(wallet);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 该接口功能负责生成基于“客户余额”的应收账款以及相应的收款凭证，在此基础上付款中心中该经销商的余额将被立即扣减。
   *
   * @param customerCode
   * @param amount
   * @param businessNo
   * @param remark
   * @return
   */
  @PostMapping("receipt")
  @ApiOperation(value = "扣款", notes = "该接口功能负责生成基于“客户余额”的应收账款以及相应的收款凭证，在此基础上付款中心中该经销商的余额将被立即扣减。")
  public ResponseModel receipt(@RequestParam @ApiParam("客户编码") String customerCode,
                               @RequestParam @ApiParam("扣款金额") BigDecimal amount,
                               @RequestParam @ApiParam("业务类型，1：提现、2：充值、3：转账结余、4：退货退款、4：应付单") Integer businessType,
                               @RequestParam @ApiParam("业务单号") String businessNo,
                               @RequestParam(required = false) @ApiParam("备注") String remark) {
    try {
      CustomerWallet wallet = customerWalletService.receipt(customerCode, amount, businessType, businessNo, remark);
      return this.buildHttpResultW(wallet);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 该接口功能负责生成基于“客户余额”的应收账款以及相应的收款凭证，在此基础上付款中心中该经销商的余额将被立即扣减。
   *
   * @param customerCode
   * @param amount
   * @param businessNo
   * @param validCode
   * @param remark
   * @return
   */
  @PostMapping("receiptByValidCode")
  @ApiOperation(value = "扣款，验证手机验证码", notes = "该接口功能负责生成基于“客户余额”的应收账款以及相应的收款凭证，在此基础上付款中心中该经销商的余额将被立即扣减。")
  public ResponseModel receiptByValidCode(@RequestParam @ApiParam("客户编码") String customerCode,
                               @RequestParam @ApiParam("扣款金额") BigDecimal amount,
                               @RequestParam @ApiParam("业务类型，1：提现、2：充值、3：转账结余、4：退货退款、4：应付单") Integer businessType,
                               @RequestParam @ApiParam("业务单号") String businessNo,
                               @RequestParam @ApiParam("短信验证码") String validCode,
                               @RequestParam(required = false) @ApiParam("备注") String remark) {
    try {
      CustomerWallet wallet = customerWalletService.receipt(customerCode, amount, businessType, businessNo, validCode, remark);
      return this.buildHttpResultW(wallet);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 恢复金额
   *
   * @param customerCode
   * @param amount
   * @param businessNo
   * @param remark
   * @return
   */
  @PostMapping("recover")
  @ApiOperation("恢复金额")
  public ResponseModel recover(@RequestParam @ApiParam("客户编码") String customerCode,
                               @RequestParam @ApiParam("恢复金额") BigDecimal amount,
                               @RequestParam @ApiParam("业务类型，1：提现、2：充值、3：转账结余、4：退货退款、4：应付单") Integer businessType,
                               @RequestParam @ApiParam("业务单号") String businessNo,
                               @RequestParam(required = false) @ApiParam("备注") String remark) {
    try {
      CustomerWallet wallet = customerWalletService.recover(customerCode, amount, businessType, businessNo, remark);
      return this.buildHttpResultW(wallet);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 提现
   * @param customerCode
   * @param amount
   * @param remark
   * @return
   */
  @PostMapping("withdraw")
  @ApiOperation("提现")
  public ResponseModel withdraw(@RequestParam @ApiParam("客户编码") String customerCode,
                                @RequestParam(required = false) @ApiParam("提现金额，不传提现全部余额") BigDecimal amount,
                                @RequestParam(required = false) @ApiParam("备注") String remark) {
    try {
      CustomerWallet wallet = customerWalletService.withdraw(customerCode, amount, remark);
      return this.buildHttpResultW(wallet);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 通过客户编码查询
   * */
  @GetMapping("/findByCustomerCode")
  @ApiOperation(value = "通过客户编码查询")
  public ResponseModel findByCustomerCode(@RequestParam @ApiParam("客户编码") String customerCode){
    try {
      CustomerWallet wallet = this.customerWalletService.findByCustomerCode(customerCode);
      return this.buildHttpResultW(wallet);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }


  /**
   * 通过主键进行数据的查询
   * */
  @GetMapping("/{id}")
  @ApiOperation(value = "通过主键进行数据的查询")
  public ResponseModel findById(@PathVariable("id") @ApiParam(name="id" , value="主键")String id){
    try {
      CustomerWallet current = this.customerWalletService.findById(id);
      return this.buildHttpResultW(current);
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }



} 
