package com.bizunited.empower.business.payment.controller;

import com.bizunited.empower.business.payment.common.enums.FundsChannelType;
import com.bizunited.empower.business.payment.entity.DefrayInfo;
import com.bizunited.empower.business.payment.service.DefrayInfoService;
import com.bizunited.empower.business.payment.vo.DefrayInfoVo;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * DefrayInfo业务模型的MVC Controller层实现，基于HTTP Restful风格
 *
 * @author saturn
 */
@RestController
@RequestMapping("/v1/defrayInfo")
public class DefrayInfoController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(DefrayInfoController.class);

  @Autowired
  private DefrayInfoService defrayInfoService;

  /**
   * 通过单据号新增付款单
   * fundsChannelType 支付类型 {@link FundsChannelType}
   */
  @ApiOperation(value = "通过单据号新增付款单 现金支付:4，信用支付:6")
  @PostMapping(value = "createByAssociatedCode")
  public ResponseModel createByAssociatedCode(@ApiParam("单据号") @RequestParam("associatedCode") String associatedCode, @ApiParam("金额") @RequestParam("amount") BigDecimal amount, @ApiParam("支付类型") @RequestParam("fundsChannelType") Integer fundsChannelType) {
    try {
      DefrayInfo current = this.defrayInfoService.createByAssociatedCode(associatedCode, amount, FundsChannelType.valueOfType(fundsChannelType));
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 通过单据号新增付款单
   * fundsChannelType 支付类型 {@link FundsChannelType}
   */
  @ApiOperation(value = "通过单据号新增付款单 现金支付:4，信用支付:6")
  @PostMapping(value = "")
  public ResponseModel create(@RequestBody DefrayInfo defrayInfo) {
    try {
      DefrayInfo current = this.defrayInfoService.create(defrayInfo);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 通过单据号新增收款单(无需确认收款)
   * fundsChannelType 支付类型 {@link com.bizunited.empower.business.payment.common.enums.FundsChannelType}
   */
  @ApiOperation(value = "通过单据号新增收款单(无需确认收款) 现金支付:4，信用支付:6")
  @PostMapping(value = "createWithoutConfirm")
  public ResponseModel createWithoutConfirm(@RequestBody DefrayInfo defrayInfo) {
    try {
      DefrayInfo current = this.defrayInfoService.createWithoutConfirm(defrayInfo);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 确认付款信息
   */
  @ApiOperation(value = "确认付款信息")
  @PostMapping(value = "confirm")
  public ResponseModel confirm(@ApiParam("付款单号") @RequestParam("defrayCode") String defrayCode) {
    try {
      DefrayInfo current = this.defrayInfoService.confirm(defrayCode);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 取消付款信息
   */
  @ApiOperation(value = "取消付款信息")
  @PostMapping(value = "cancel")
  public ResponseModel cancel(@ApiParam("付款单号") @RequestParam("defrayCode") String defrayCode) {
    try {
      DefrayInfo current = this.defrayInfoService.cancelByDefrayCode(defrayCode);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 相关的查询过程，http接口。通过主键进行数据的查询
   */
  @ApiOperation(value = "相关的查询过程，http接口。通过主键进行数据的查询")
  @GetMapping(value = "/{id}")
  public ResponseModel findById(@PathVariable("id") @ApiParam(name = "id", value = "主键") String id) {
    try {
      DefrayInfo current = this.defrayInfoService.findById(id);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照DefrayInfo实体中的（paymentInfo）关联的 应收账款编号进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。
   *
   * @param paymentInfo 关联的 应收账款编号
   */
  @ApiOperation(value = "按照DefrayInfo实体中的（paymentInfo）关联的 应收账款编号进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。")
  @RequestMapping(value = "/findDetailsByPaymentInfo", method = {RequestMethod.GET})
  public ResponseModel findDetailsByReceivableInfo(@RequestParam("paymentInfo") @ApiParam("关联的 应付账款编号") String paymentInfo) {
    try {
      Set<DefrayInfo> result = this.defrayInfoService.findDetailsByPaymentInfo(paymentInfo);
      return this.buildHttpResultW(result, "paymentInfo");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照ReceiptInfo实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。
   *
   * @param id 主键
   */
  @ApiOperation(value = "按照ReceiptInfo实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。")
  @RequestMapping(value = "/findDetailsById", method = {RequestMethod.GET})
  public ResponseModel findDetailsById(@RequestParam("id") @ApiParam("主键") String id) {
    try {
      DefrayInfo result = this.defrayInfoService.findDetailsById(id);
      return this.buildHttpResultW(result, "paymentInfo", "defrayAttachments");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照DefrayInfo实体中的（defrayCode）收款单流水编号进行查询
   *
   * @param defrayCode 收款单流水编号
   */
  @ApiOperation(value = "按照ReceiptInfo实体中的（defrayCode）收款单流水编号进行查询")
  @RequestMapping(value = "/findByDefrayCode", method = {RequestMethod.GET})
  public ResponseModel findByReceiptCode(@RequestParam("defrayCode") @ApiParam("收款单流水编号") String defrayCode) {
    try {
      DefrayInfo result = this.defrayInfoService.findByDefrayCode(defrayCode);
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照DefrayInfo实体中的（defrayCode）收款单流水编号进行查询（详情）
   *
   * @param defrayCode 收款单流水编号
   */
  @ApiOperation(value = "按照ReceiptInfo实体中的（defrayCode）收款单流水编号进行查询（详情）")
  @RequestMapping(value = "/findDetailsByDefrayCode", method = {RequestMethod.GET})
  public ResponseModel findDetailsByReceiptCode(@RequestParam("defrayCode") @ApiParam("收款单流水编号") String defrayCode) {
    try {
      DefrayInfo result = this.defrayInfoService.findDetailsByDefrayCode(defrayCode);
      return this.buildHttpResultW(result, "paymentInfo", "defrayAttachments");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 多条件分页查询
   * <br/>
   *
   * @param pageable
   * @return
   */
  @GetMapping("findByConditions")
  @ApiOperation(value = "多条件分页查询", notes = "分页参数为page和size，page从0开始，size默认10;receiptStatus:收款单状态，收款时间：startTime开始时间|endTime结束时间，")
  public ResponseModel findByConditions(@PageableDefault(10) Pageable pageable,
                                        HttpServletRequest request) {
    try {
      Map<String, Object> params = new HashMap<>();
      Enumeration<String> names = request.getParameterNames();
      if (names != null) {
        while (names.hasMoreElements()) {
          String name = names.nextElement();
          String[] values = request.getParameterValues(name);
          if (values != null && values.length == 1) {
            params.put(name, values[0]);
          } else if (values != null && values.length > 1) {
            params.put(name, values);
          }
        }
      }
      InvokeParams conditions = new InvokeParams();
      conditions.putInvokeParams(params);
      Page<DefrayInfoVo> page = this.defrayInfoService.findByConditions(pageable, conditions);
      return this.buildHttpResultW(page, "fundsAccount", "paymentInfo");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据筛选条件查询已付金额统计
   *
   * @param request
   * @return
   */
  @GetMapping("findTotalByConditions")
  @ApiOperation(value = "根据筛选条件查询已付金额统计")
  public ResponseModel findTotalByConditions(@PageableDefault(10) Pageable pageable, HttpServletRequest request) {
    try {
      Map<String, Object> params = new HashMap<>();
      Enumeration<String> names = request.getParameterNames();
      if (names != null) {
        while (names.hasMoreElements()) {
          String name = names.nextElement();
          String[] values = request.getParameterValues(name);
          if (values != null && values.length == 1) {
            params.put(name, values[0]);
          } else if (values != null && values.length > 1) {
            params.put(name, values);
          }
        }
      }
      InvokeParams conditions = new InvokeParams();
      conditions.putInvokeParams(params);
      BigDecimal payedAmountTotal = this.defrayInfoService.findTotalByConditions(pageable, conditions);
      return this.buildHttpResultW(payedAmountTotal);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
} 
