package com.bizunited.empower.business.payment.controller;

import com.bizunited.empower.business.payment.entity.ElectronicAccountBank;
import com.bizunited.empower.business.payment.service.ElectronicAccountBankService;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

/**
 * 电子账户银行业务模型的MVC Controller层实现，基于HTTP Restful风格
 * @Author: Paul Chan
 * @Date: 2021/4/12 上午10:50
 */
@RestController
@RequestMapping("/v1/electronicAccount/banks")
public class ElectronicAccountBankController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ElectronicAccountBankController.class);

  @Autowired
  private ElectronicAccountBankService electronicAccountBankService;

  /**
   * 新增银行卡
   * @param bank
   * @return
   */
  @PostMapping("")
  @ApiOperation("新增银行卡")
  public ResponseModel create(@RequestBody @ApiParam("银行卡信息") ElectronicAccountBank bank) {
    try {
      ElectronicAccountBank dbBank = electronicAccountBankService.create(bank);
      return this.buildHttpResultW(dbBank);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 修改银行卡
   * @param bank
   * @return
   */
  @PatchMapping("")
  @ApiOperation("修改银行卡")
  public ResponseModel update(@RequestBody @ApiParam("银行卡信息") ElectronicAccountBank bank) {
    try {
      ElectronicAccountBank dbBank = electronicAccountBankService.update(bank);
      return this.buildHttpResultW(dbBank);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 发送银行卡短信验证码
   * @param id
   * @return
   */
  @PostMapping("sendValidCode")
  @ApiOperation("发送银行卡短信验证码")
  public ResponseModel sendValidCodeById(@RequestParam @ApiParam("主键ID") String id) {
    try {
      electronicAccountBankService.sendValidCodeById(id);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 绑定银行卡
   * @param id
   * @param validCode 验证码或金额，如果是企业，则应该是一个金额，如果是个人，则是短信验证码
   * @return
   */
  @PostMapping("bindById")
  @ApiOperation("绑定银行卡")
  public ResponseModel bindById(@RequestParam @ApiParam("主键ID") String id,
                                @RequestParam @ApiParam("验证码或金额，如果是企业，则应该是一个金额，如果是个人，则是短信验证码") String validCode) {
    try {
      electronicAccountBankService.bindById(id, validCode);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据账户ID绑定银行卡
   * @param electronicAccountId
   * @param validCode 验证码或金额，如果是企业，则应该是一个金额，如果是个人，则是短信验证码
   * @return
   */
  @PostMapping("bindByElectronicAccountId")
  @ApiOperation("根据账户ID绑定银行卡")
  public ResponseModel bindByElectronicAccountId(@RequestParam @ApiParam("账户ID") String electronicAccountId,
                                @RequestParam @ApiParam("验证码或金额，如果是企业，则应该是一个金额，如果是个人，则是短信验证码") String validCode) {
    try {
      electronicAccountBankService.bindByElectronicAccountId(electronicAccountId, validCode);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 解绑银行卡
   * @param id
   * @param validCode
   * @return
   */
  @PutMapping("unbindById")
  @ApiOperation("解绑银行卡")
  public ResponseModel unbindById(@RequestParam @ApiParam("主键ID") String id,
                                  @RequestParam @ApiParam("验证码") String validCode) {
    try {
      electronicAccountBankService.unbindById(id, validCode);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 查询当前可用的银行卡信息
   * @return
   */
  @GetMapping("findEnableByTenant")
  @ApiOperation("查询当前可用的银行卡信息")
  public ResponseModel findEnableByTenant() {
    try {
      ElectronicAccountBank bank = electronicAccountBankService.findEnableByTenant();
      return this.buildHttpResultW(bank);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

}
