package com.bizunited.empower.business.payment.controller;

import com.bizunited.empower.business.payment.common.enums.ElAccountBillState;
import com.bizunited.empower.business.payment.entity.ElectronicAccountBill;
import com.bizunited.empower.business.payment.service.ElectronicAccountBillService;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.google.common.collect.Maps;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.Date;
import java.util.Map;

/**
 * 电子账户流水控制器
 * @Author: Paul Chan
 * @Date: 2021/4/20 上午11:49
 */
@RestController
@RequestMapping("/v1/electronicAccount/bills")
public class ElectronicAccountBillController extends BaseController {

  private static final Logger LOGGER = LoggerFactory.getLogger(ElectronicAccountBillController.class);

  @Autowired
  private ElectronicAccountBillService electronicAccountBillService;

  /**
   * 根据流水单号更新单号状态
   * @param billNo
   * @param state
   * @return
   */
  @PutMapping("updateStateByBillNo")
  @ApiOperation("根据流水单号更新单号状态")
  public ResponseModel updateStateByBillNo(@RequestParam @ApiParam("流水单号 ") String billNo,
                                           @RequestParam @ApiParam("流水状态，1：成功，2：失败") Integer state) {
    try {
      electronicAccountBillService.updateStateByBillNo(billNo, ElAccountBillState.valueOfState(state));
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }


  /**
   * 多条件分页查询
   * @return
   */
  @GetMapping("findByConditions")
  @ApiOperation(value = "多条件分页查询", notes = "分页查询，分页参数为page和size，page从0开始，size默认50")
  public ResponseModel findByConditions(@PageableDefault(50) Pageable pageable,
                                        @RequestParam(required = false) @ApiParam("操作类型，1：收入 2：支出") Integer type,
                                        @RequestParam(required = false) @ApiParam("流水状态，0：处理中 1：成功，2：失败") Integer state,
                                        @RequestParam(required = false) @ApiParam("业务类型，1：提现、2：应收、3：退货退款") Integer businessType,
                                        @RequestParam(required = false) @ApiParam("流水单号") String billNo,
                                        @DateTimeFormat(pattern = "yyyy-MM-dd") @RequestParam(required = false) @ApiParam("开始日期(yyyy-MM-dd)") Date startDate,
                                        @DateTimeFormat(pattern = "yyyy-MM-dd") @RequestParam(required = false) @ApiParam("结束日期(yyyy-MM-dd)") Date endDate) {
    try {
      Map<String, Object> conditions = Maps.newHashMap();
      conditions.put("type", type);
      conditions.put("state", state);
      conditions.put("businessType", businessType);
      conditions.put("billNo", billNo);
      conditions.put("startDate", startDate);
      conditions.put("endDate", endDate);
      Page<ElectronicAccountBill> page = electronicAccountBillService.findByConditions(conditions, pageable);
      return this.buildHttpResultW(page, "electronicAccount");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

}
