package com.bizunited.empower.business.payment.controller;

import com.bizunited.empower.business.payment.entity.ElectronicAccount;
import com.bizunited.empower.business.payment.service.ElectronicAccountService;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.math.BigDecimal;

/**
 * 电子账户业务模型的MVC Controller层实现，基于HTTP Restful风格
 * @author Paul Chan
 */
@RestController
@RequestMapping("/v1/electronicAccounts")
public class ElectronicAccountController extends BaseController { 
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ElectronicAccountController.class);
  
  @Autowired
  private ElectronicAccountService electronicAccountService;

  /**
   *
   * */
  @PostMapping(value="")
  @ApiOperation(value = "提交开户资料")
  public ResponseModel create(@RequestBody @ApiParam("开户资料") ElectronicAccount electronicAccount) {
    try {
      ElectronicAccount current = this.electronicAccountService.create(electronicAccount);
      return this.buildHttpResultW(current, "accountInfo", "certificate", "banks");
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 修改提交开户资料
   * */
  @PatchMapping(value="")
  @ApiOperation(value = "修改提交开户资料")
  public ResponseModel update(@RequestBody @ApiParam("开户资料") ElectronicAccount electronicAccount) {
    try {
      ElectronicAccount current = this.electronicAccountService.update(electronicAccount);
      return this.buildHttpResultW(current, "accountInfo", "certificate", "banks");
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 修改是否显示余额
   * @param showBalance
   * @return
   */
  @PatchMapping("updateShowBalance")
  @ApiOperation(value = "修改是否显示余额")
  public ResponseModel updateShowBalance(@RequestParam @ApiParam("是否显示余额，0：1") Boolean showBalance) {
    try {
      this.electronicAccountService.updateShowBalance(showBalance);
      return this.buildHttpResult();
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 收入
   * @param amount
   * @param businessNo
   * @param remark
   * @return
   */
  @PostMapping("receipt")
  @ApiOperation("收入")
  public ResponseModel receipt(@RequestParam @ApiParam("收入金额") BigDecimal amount,
                               @RequestParam @ApiParam("手续费率") BigDecimal chargeRate,
                               @RequestParam @ApiParam("手续费") BigDecimal chargeAmount,
                               @RequestParam @ApiParam("实际金额") BigDecimal realAmount,
                               @RequestParam @ApiParam("业务单号") String businessNo,
                               @RequestParam @ApiParam("交易来源") String dealFrom,
                               @RequestParam(required = false) @ApiParam("备注") String remark) {
    try {
      electronicAccountService.receipt(amount, chargeRate, chargeAmount, realAmount, businessNo, remark, dealFrom);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 提现
   * @param amount 提现金额
   * @param validCode 验证码
   * @return
   */
  @PostMapping("withdraw")
  @ApiOperation("提现")
  public ResponseModel withdraw(@RequestParam @ApiParam("提现金额") BigDecimal amount,
                                @RequestParam @ApiParam("验证码") String validCode) {
    try {
      electronicAccountService.withdraw(amount, validCode);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 给电子账户安全手机号发送手机验证码
   * @param businessType
   * @return
   */
  @PostMapping("sendValidCode")
  @ApiOperation(value = "给电子账户安全手机号发送手机验证码",
      notes = "经销商短信业务类型 1：解绑电子账户银行卡，2：电子账户提现，3：更改电子账户手机号码旧手机验证码，4：更改电子账户手机号码新手机验证码，5：新增电子账户手机号码，6：电子账户签约，7：绑定银行卡")
  public ResponseModel sendValidCode(@RequestParam(required = false) @ApiParam("手机号码，只有当短信业务类型为[4,5]时需要传入") String mobile,
                                     @RequestParam @ApiParam("经销商短信业务类型") Integer businessType) {
    try {
      electronicAccountService.sendValidCode(mobile, businessType);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 验证手机验证码
   * @param mobile
   * @param validCode
   * @param businessType
   * @return
   */
  @PostMapping("verifyValidCode")
  @ApiOperation(value = "验证手机验证码",
      notes = "经销商短信业务类型 1：解绑电子账户银行卡，2：电子账户提现，3：更改电子账户手机号码旧手机验证码，4：更改电子账户手机号码新手机验证码，5：新增电子账户手机号码")
  public ResponseModel verifyValidCode(@RequestParam @ApiParam("手机号码") String mobile,
                                       @RequestParam @ApiParam("验证码") String validCode,
                                       @RequestParam @ApiParam("经销商短信业务类型") Integer businessType) {
    try {
      electronicAccountService.verifyValidCode(mobile, validCode, businessType);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 绑定账号安全手机号码
   * @param mobile
   * @param validCode
   * @return
   */
  @PostMapping("bindSecurityMobile")
  @ApiOperation("绑定账号安全手机号码")
  public ResponseModel bindSecurityMobile(@RequestParam @ApiParam("手机号码") String mobile,
                                          @RequestParam @ApiParam("验证码") String validCode) {
    try {
      electronicAccountService.bindSecurityMobile(mobile, validCode);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 重新绑定账号安全手机号码
   * @param mobile 新手机号
   * @param oldMobileValidCode
   * @param newMobileValidCode
   * @return
   */
  @PostMapping("rebindSecurityMobile")
  @ApiOperation("重新绑定账号安全手机号码")
  public ResponseModel rebindSecurityMobile(@RequestParam @ApiParam("手机号码") String mobile,
                                            @RequestParam @ApiParam("旧手机验证码") String oldMobileValidCode,
                                            @RequestParam @ApiParam("新手机验证码") String newMobileValidCode) {
    try {
      electronicAccountService.rebindSecurityMobile(mobile, oldMobileValidCode, newMobileValidCode);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 获取签约验证码
   * @param isUpdate
   * @param name
   * @param idCardNo
   * @param mobile
   * @return
   */
  @PostMapping("sendSignValidCode")
  @ApiOperation(value = "获取签约验证码", notes = "如果要更新签约人，则签约人信息必须填写")
  public ResponseModel sendSignValidCode(@RequestParam(required = false) @ApiParam("是否更新签约人信息") Boolean isUpdate,
                                         @RequestParam(required = false) @ApiParam("签约人姓名") String name,
                                         @RequestParam(required = false) @ApiParam("签约人身份证号") String idCardNo,
                                         @RequestParam(required = false) @ApiParam("签约人手机号") String mobile) {
    try {
      electronicAccountService.sendSignValidCode(isUpdate, name, idCardNo, mobile);
      return this.buildHttpResult();
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 获取签约链接地址
   * @return
   */
  @GetMapping("getSignUrlByTenant")
  @ApiOperation("获取签约链接地址")
  public ResponseModel getSignUrlByTenant(@RequestParam @ApiParam("验证码") String validCode) {
    try {
      String url = electronicAccountService.getSignUrlByTenant(validCode);
      return this.buildHttpResultW(url);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据ID查询过详情信息
   * @param id 主键
   */
  @GetMapping("findDetailsById")
  @ApiOperation("根据ID查询过详情信息")
  public ResponseModel findDetailsById(@RequestParam("id") @ApiParam("主键") String id) {
    try { 
      ElectronicAccount result = this.electronicAccountService.findDetailsById(id); 
      return this.buildHttpResultW(result, "accountInfo", "certificate", "banks");
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 获取当前经销商的电子账户资料详情
   * @return
   */
  @GetMapping("findDetailsByTenant")
  @ApiOperation("获取当前经销商的电子账户资料详情")
  public ResponseModel findDetailsByTenant() {
    try {
      ElectronicAccount account = this.electronicAccountService.findDetailsByTenant();
      return this.buildHttpResultW(account, "accountInfo", "certificate", "banks");
    } catch(RuntimeException e) {
      LOGGER.error(e.getMessage() , e);
      return this.buildHttpResultForException(e);
    }
  }

} 
