package com.bizunited.empower.business.payment.controller;

import com.bizunited.empower.business.payment.common.enums.FundsChannelType;
import com.bizunited.empower.business.payment.dto.ReceiptDto;
import com.bizunited.empower.business.payment.entity.ReceiptInfo;
import com.bizunited.empower.business.payment.service.ReceiptInfoService;
import com.bizunited.empower.business.payment.vo.ReceiptInfoVo;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * ReceiptInfo业务模型的MVC Controller层实现，基于HTTP Restful风格
 *
 * @author saturn
 */
@RestController
@RequestMapping("/v1/receiptInfo")
public class ReceiptInfoController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ReceiptInfoController.class);

  @Autowired
  private ReceiptInfoService receiptInfoService;

  /**
   * 通过单据号新增收款单
   * fundsChannelType 支付类型 {@link com.bizunited.empower.business.payment.common.enums.FundsChannelType}
   */
  @ApiOperation(value = "通过单据号新增收款单 现金支付:4，信用支付:6")
  @PostMapping(value = "createByAssociatedCode")
  public ResponseModel createByAssociatedCode(@ApiParam("单据号") @RequestParam("associatedCode") String associatedCode, @ApiParam("金额") @RequestParam("amount") BigDecimal amount, @ApiParam("支付类型") @RequestParam("fundsChannelType") Integer fundsChannelType) {
    try {
      ReceiptInfoVo current = this.receiptInfoService.createByAssociatedCode(associatedCode, amount, FundsChannelType.valueOfType(fundsChannelType), null);
      return this.buildHttpResultW(current, "receivableInfo", "receiptAttachments", "extend");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 通过单据号新增收款单
   * fundsChannelType 支付类型 {@link com.bizunited.empower.business.payment.common.enums.FundsChannelType}
   */
  @ApiOperation(value = "通过单据号新增收款单 现金支付:4，信用支付:6")
  @PostMapping(value = "createByAssociatedCodeAndValidCode")
  public ResponseModel createByAssociatedCodeAndValidCode(@ApiParam("单据号") @RequestParam("associatedCode") String associatedCode, @ApiParam("金额") @RequestParam("amount") BigDecimal amount, @ApiParam("支付类型") @RequestParam("fundsChannelType") Integer fundsChannelType, @ApiParam("验证码") @RequestParam(value = "validCode") String validCode) {
    try {
      InvokeParams params = new InvokeParams();
      params.putInvokeParam("validCode", validCode);
      ReceiptInfoVo current = this.receiptInfoService.createByAssociatedCode(associatedCode, amount, FundsChannelType.valueOfType(fundsChannelType), params);
      return this.buildHttpResultW(current, "receivableInfo", "receiptAttachments", "extend");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 通过单据号新增收款单
   * fundsChannelType 支付类型 {@link com.bizunited.empower.business.payment.common.enums.FundsChannelType}
   */
  @ApiOperation(value = "通过单据号新增收款单")
  @PostMapping(value = "createByReceiptDto")
  public ResponseModel createByDto(@RequestBody ReceiptDto receiptDto) {
    try {
      ReceiptInfoVo current = this.receiptInfoService.create(receiptDto);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 通过单据号重新发起支付
   * fundsChannelType 支付类型 {@link com.bizunited.empower.business.payment.common.enums.FundsChannelType}
   */
  @ApiOperation(value = "通过单据号重新发起支付")
  @PostMapping(value = "redo")
  public ResponseModel redoByReceiptCode(@RequestParam("receiptCode") String receiptCode, @RequestParam(value = "fundsChannel", required = false) Integer fundsChannel, @ApiParam("验证码") @RequestParam(value = "validCode", required = false) String validCode) {
    try {
      InvokeParams params = new InvokeParams();
      params.putInvokeParam("validCode", validCode);
      ReceiptInfoVo current = this.receiptInfoService.redoByReceiptCode(receiptCode, fundsChannel, params);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 通过单据号新增收款单
   * fundsChannelType 支付类型 {@link com.bizunited.empower.business.payment.common.enums.FundsChannelType}
   */
  @ApiOperation(value = "通过单据号新增收款单 现金支付:4，信用支付:6")
  @PostMapping(value = "")
  public ResponseModel create(@RequestBody ReceiptInfo receiptInfo) {
    try {
      ReceiptInfo current = this.receiptInfoService.create(receiptInfo, null);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 通过单据号新增收款单(无需确认收款)
   * fundsChannelType 支付类型 {@link com.bizunited.empower.business.payment.common.enums.FundsChannelType}
   */
  @ApiOperation(value = "通过单据号新增收款单(无需确认收款) 现金支付:4，信用支付:6")
  @PostMapping(value = "createWithoutConfirm")
  public ResponseModel createWithoutConfirm(@RequestBody ReceiptInfo receiptInfo) {
    try {
      ReceiptInfo current = this.receiptInfoService.createWithoutConfirm(receiptInfo);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 通过客户编号创建应收账款以及收款单
   * <p>
   * fundsChannel 1:支付宝 2:微信小程序支付 3:正扫码支付 4:现金支付 5:银行转账  6:信用支付 7:中金余额支付
   */
  @ApiOperation(value = "通过客户编号新增收款单")
  @PostMapping(value = "createByCustomerCode")
  public ResponseModel createByCustomerCode(@ApiParam("客户编号") @RequestParam("customerCode") String customerCode, @ApiParam("金额") @RequestParam("amount") BigDecimal amount, @ApiParam("资金类型 1:支付宝 2:微信小程序支付 3:正扫码支付 4:现金支付 5:银行转账  6:信用支付 7:中金余额支付") @RequestParam("fundsChannel") Integer fundsChannel) {
    try {
      ReceiptInfoVo current = this.receiptInfoService.createByCustomerCodeAndFundsChannels(customerCode, amount, fundsChannel, null);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 确认收款信息
   */
  @ApiOperation(value = "确认收款信息")
  @PostMapping(value = "confirm")
  public ResponseModel confirm(@ApiParam("收款单号") @RequestParam("receiptCode") String receiptCode) {
    try {
      ReceiptInfo current = this.receiptInfoService.confirm(receiptCode);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 取消收款信息
   */
  @ApiOperation(value = "取消收款信息")
  @PostMapping(value = "cancel")
  public ResponseModel cancel(@ApiParam("收款单号") @RequestParam("receiptCode") String receiptCode) {
    try {
      ReceiptInfo current = this.receiptInfoService.cancelByReceiptCode(receiptCode);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 相关的查询过程，http接口。通过主键进行数据的查询
   */
  @ApiOperation(value = "相关的查询过程，http接口。通过主键进行数据的查询")
  @GetMapping(value = "/{id}")
  public ResponseModel findById(@PathVariable("id") @ApiParam(name = "id", value = "主键") String id) {
    try {
      ReceiptInfo current = this.receiptInfoService.findById(id);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照ReceiptInfo实体中的（receivableInfo）关联的 应收账款编号进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。
   *
   * @param receivableInfo 关联的 应收账款编号
   */
  @ApiOperation(value = "按照ReceiptInfo实体中的（receivableInfo）关联的 应收账款编号进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。")
  @RequestMapping(value = "/findDetailsByReceivableInfo", method = {RequestMethod.GET})
  public ResponseModel findDetailsByReceivableInfo(@RequestParam("receivableInfo") @ApiParam("关联的 应收账款编号") String receivableInfo) {
    try {
      Set<ReceiptInfo> result = this.receiptInfoService.findDetailsByReceivableInfo(receivableInfo);
      return this.buildHttpResultW(result, "receivableInfo");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照ReceiptInfo实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。
   *
   * @param id 主键
   */
  @ApiOperation(value = "按照ReceiptInfo实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。")
  @RequestMapping(value = "/findDetailsById", method = {RequestMethod.GET})
  public ResponseModel findDetailsById(@RequestParam("id") @ApiParam("主键") String id) {
    try {
      ReceiptInfo result = this.receiptInfoService.findDetailsById(id);
      return this.buildHttpResultW(result, "receivableInfo", "receiptAttachments");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照ReceiptInfo实体中的（receiptCode）收款单流水编号进行查询
   *
   * @param receiptCode 收款单流水编号
   */
  @ApiOperation(value = "按照ReceiptInfo实体中的（receiptCode）收款单流水编号进行查询")
  @RequestMapping(value = "/findByReceiptCode", method = {RequestMethod.GET})
  public ResponseModel findByReceiptCode(@RequestParam("receiptCode") @ApiParam("收款单流水编号") String receiptCode) {
    try {
      ReceiptInfo result = this.receiptInfoService.findByReceiptCode(receiptCode);
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照ReceiptInfo实体中的（receiptCode）收款单流水编号进行查询
   *
   * @param receiptCode 收款单流水编号
   */
  @ApiOperation(value = "按照ReceiptInfo实体中的（receiptCode）收款单流水编号进行查询")
  @RequestMapping(value = "/findDetailsByReceiptCode", method = {RequestMethod.GET})
  public ResponseModel findDetailsByReceiptCode(@RequestParam("receiptCode") @ApiParam("收款单流水编号") String receiptCode) {
    try {
      ReceiptInfo result = this.receiptInfoService.findDetailsByReceiptCode(receiptCode);
      return this.buildHttpResultW(result, "receivableInfo", "receiptAttachments");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照ReceiptInfo实体中的（customerCode）客户编号以及资金渠道进行查询
   *
   * @param customerCode  收款单流水编号
   * @param fundsChannels 资金渠道
   */
  @ApiOperation(value = "按照ReceiptInfo实体中的（customerCode）客户编号进行查询")
  @RequestMapping(value = "/findWaitForByCustomerCodeAndFundsChannels", method = {RequestMethod.GET})
  public ResponseModel findWaitForByCustomerCodeAndFundsChannels(@RequestParam("customerCode") @ApiParam("客户编号") String customerCode, @RequestParam("fundsChannel") @ApiParam("资金渠道编号 1：支付宝 2：微信小程序支付 3：正扫码支付 4：现金支付 5：银行转账 6：信用支付 7：中金余额支付") List<Integer> fundsChannels) {
    try {
      List<ReceiptInfo> result = this.receiptInfoService.findWaitForByCustomerCodeAndFundsChannels(customerCode, fundsChannels);
      return this.buildHttpResultW(result, "receivableInfo");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 多条件分页查询
   * <br/>
   *
   * @param pageable
   * @return
   */
  @GetMapping("findByConditions")
  @ApiOperation(value = "多条件分页查询", notes = "分页参数为page和size，page从0开始，size默认10;receiptStatus:收款单状态，收款时间：startTime开始时间|endTime结束时间，account当前业务员账号")
  public ResponseModel findByConditions(@PageableDefault(10) Pageable pageable,
                                        HttpServletRequest request) {
    try {
      Map<String, Object> params = new HashMap<>();
      Enumeration<String> names = request.getParameterNames();
      if (names != null) {
        while (names.hasMoreElements()) {
          String name = names.nextElement();
          String[] values = request.getParameterValues(name);
          if (values != null && values.length == 1) {
            params.put(name, values[0]);
          } else if (values != null && values.length > 1) {
            params.put(name, values);
          }
        }
      }
      InvokeParams conditions = new InvokeParams();
      conditions.putInvokeParams(params);
      Page<ReceiptInfoVo> page = this.receiptInfoService.findByConditions(pageable, conditions);
      return this.buildHttpResultW(page, "receivableAccount", "receivableInfo");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据筛选条件查询已付金额统计
   *
   * @param request
   * @return
   */
  @GetMapping("findTotalByConditions")
  @ApiOperation(value = "根据筛选条件查询已付金额统计")
  public ResponseModel findTotalByConditions(@PageableDefault(10) Pageable pageable, HttpServletRequest request) {
    try {
      Map<String, Object> params = new HashMap<>();
      Enumeration<String> names = request.getParameterNames();
      if (names != null) {
        while (names.hasMoreElements()) {
          String name = names.nextElement();
          String[] values = request.getParameterValues(name);
          if (values != null && values.length == 1) {
            params.put(name, values[0]);
          } else if (values != null && values.length > 1) {
            params.put(name, values);
          }
        }
      }
      InvokeParams conditions = new InvokeParams();
      conditions.putInvokeParams(params);
      BigDecimal receivedAmountTotal = this.receiptInfoService.findTotalByConditions(pageable, conditions);
      return this.buildHttpResultW(receivedAmountTotal);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
} 
