package com.bizunited.empower.business.payment.controller;

import com.bizunited.empower.business.payment.common.enums.ReceivableType;
import com.bizunited.empower.business.payment.entity.ReceivableInfo;
import com.bizunited.empower.business.payment.service.ReceivableInfoService;
import com.bizunited.empower.business.payment.vo.AssociatedReceivableVo;
import com.bizunited.empower.business.payment.vo.CustomerReceivableVo;
import com.bizunited.platform.common.controller.BaseController;
import com.bizunited.platform.common.controller.model.ResponseModel;
import com.bizunited.platform.common.service.invoke.InvokeParams;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

/**
 * ReceivableInfo业务模型的MVC Controller层实现，基于HTTP Restful风格
 *
 * @author saturn
 */
@RestController
@RequestMapping("/v1/receivableInfo")
public class ReceivableInfoController extends BaseController {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ReceivableInfoController.class);

  @Autowired
  private ReceivableInfoService receivableInfoService;

  /**
   * 相关的创建过程，http接口。请注意该创建过程除了可以创建receiptInfo中的基本信息以外，还可以对receiptInfo中属于OneToMany关联的明细信息一同进行创建注意：基于（ReceiptInfo）模型的创建操作传入的receiptInfoJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）
   */
  @ApiOperation(value = "相关的创建过程，http接口。请注意该创建过程除了可以创建receiptInfo中的基本信息以外，还可以对receiptInfo中属于OneToMany关联的明细信息一同进行创建注意：基于（ReceiptInfo）模型的创建操作传入的receiptInfoJSON对象，其主键信息不能有值，服务端将会自动为其赋予相关值。另外，创建操作成功后，系统将返回该对象的基本信息（不包括任何关联信息）")
  @PostMapping(value = "")
  public ResponseModel create(@ApiParam("客户编号") @RequestParam("customerCode") String customerCode, @ApiParam(value = "关联单据号",required = false) @RequestParam(value = "associatedCode",required = false) String associatedCode, @RequestParam("amount") BigDecimal amount) {
    try {
      ReceivableInfo current = this.receivableInfoService.createByAssociatedCode(customerCode, associatedCode, amount, ReceivableType.OTHER);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 相关的查询过程，http接口。通过主键进行数据的查询
   */
  @ApiOperation(value = "相关的查询过程，http接口。通过主键进行数据的查询")
  @GetMapping(value = "/{id}")
  public ResponseModel findById(@PathVariable("id") @ApiParam(name = "id", value = "主键") String id) {
    try {
      ReceivableInfo current = this.receivableInfoService.findById(id);
      return this.buildHttpResultW(current);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照ReceivableInfo实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。
   *
   * @param id 主键
   */
  @ApiOperation(value = "按照ReceivableInfo实体中的（id）主键进行查询明细查询，查询的明细包括当前业务表单所有的关联属性。")
  @RequestMapping(value = "/findDetailsById", method = {RequestMethod.GET})
  public ResponseModel findDetailsById(@RequestParam("id") @ApiParam("主键") String id) {
    try {
      ReceivableInfo result = this.receivableInfoService.findDetailsById(id);
      return this.buildHttpResultW(result);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照ReceivableInfo实体中的（receivableCode）应收账款流水编号进行查询
   *
   * @param receivableCode 应收账款流水编号
   */
  @ApiOperation(value = "按照ReceivableInfo实体中的（receivableCode）应收账款流水编号进行查询")
  @RequestMapping(value = "/findByReceivableCode", method = {RequestMethod.GET})
  public ResponseModel findByReceivableCode(@RequestParam("receivableCode") @ApiParam("应收账款流水编号") String receivableCode) {
    try {
      ReceivableInfo result = this.receivableInfoService.findByReceivableCode(receivableCode);
      return this.buildHttpResultW(result, new String[]{});
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 按照ReceivableInfo实体中的（AssociatedCode）关联编号进行查询
   *
   * @param associatedCode 关联编号
   */
  @ApiOperation(value = "按照ReceivableInfo实体中的（AssociatedCode）关联编号进行查询")
  @RequestMapping(value = "/findByAssociatedCode", method = {RequestMethod.GET})
  public ResponseModel findByAssociatedCode(@RequestParam("associatedCode") @ApiParam("关联编号") String associatedCode) {
    try {
      ReceivableInfo result = this.receivableInfoService.findByAssociatedCode(associatedCode);
      return this.buildHttpResultW(result, "receiptInfos");
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 多条件分页查询
   * <pre>
   * receivableStatus:应收账款单状态 1：未收款, 2：部分收款，3：已收款
   * startTime:开始时间
   * endTime:结束时间
   * receivableCode:应收账款单编号
   * </pre>
   *
   * @param pageable
   * @return
   */
  @GetMapping("findByConditionsForAssociated")
  @ApiOperation(value = "多条件分页查询", notes = "分页参数为page和size，page从0开始，size默认10,receivableStatus:应收账款单状态 1：未收款, 2：部分收款，3：已收款 startTime:开始时间 endTime:结束时间 receivableCode:应收账款单编号")
  public ResponseModel findByConditionsForAssociated(@PageableDefault(10) Pageable pageable,
                                                     HttpServletRequest request) {
    try {
      Map<String, Object> params = new HashMap<>();
      Enumeration<String> names = request.getParameterNames();
      if (names != null) {
        while (names.hasMoreElements()) {
          String name = names.nextElement();
          String[] values = request.getParameterValues(name);
          if (values != null && values.length == 1) {
            params.put(name, values[0]);
          } else if (values != null && values.length > 1) {
            params.put(name, values);
          }
        }
      }
      InvokeParams conditions = new InvokeParams();
      conditions.putInvokeParams(params);
      Page<AssociatedReceivableVo> page = this.receivableInfoService.findByConditionsForAssociated(pageable, conditions);
      return this.buildHttpResultW(page);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 多条件分页查询
   *
   * @param pageable
   * @return
   */
  @GetMapping("findByConditionsForCustomer")
  @ApiOperation(value = "多条件分页查询", notes = "分页参数为page和size，page从0开始，size默认10")
  public ResponseModel findByConditionsForCustomer(@PageableDefault(10) Pageable pageable,
                                                   HttpServletRequest request) {
    try {
      Map<String, Object> params = new HashMap<>();
      Enumeration<String> names = request.getParameterNames();
      if (names != null) {
        while (names.hasMoreElements()) {
          String name = names.nextElement();
          String[] values = request.getParameterValues(name);
          if (values != null && values.length == 1) {
            params.put(name, values[0]);
          } else if (values != null && values.length > 1) {
            params.put(name, values);
          }
        }
      }
      InvokeParams conditions = new InvokeParams();
      conditions.putInvokeParams(params);
      Page<CustomerReceivableVo> page = this.receivableInfoService.findByConditionsForCustomer(pageable, conditions);
      return this.buildHttpResultW(page);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }

  /**
   * 根据客户编号查询客户应收账款信息
   * @param customerCode
   * @return
   */
  @GetMapping("findAggregationByCustomerCode")
  @ApiOperation(value = "根据客户编号查询客户应收账款信息")
  public ResponseModel findAggregationByCustomerCode(@RequestParam("customerCode") @ApiParam("客户编号") String customerCode){
    try {
      CustomerReceivableVo customerReceivableVo = this.receivableInfoService.findAggregationByCustomerCode(customerCode);
      return this.buildHttpResultW(customerReceivableVo);
    } catch (RuntimeException e) {
      LOGGER.error(e.getMessage(), e);
      return this.buildHttpResultForException(e);
    }
  }
} 
